package com.jurismarches.vradi.migration;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.UnhandledException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ZipUtil;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyTransaction;
import org.sharengo.wikitty.jdbc.WikittyJDBCUtil;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiConstants.FormStatus;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.Configuration;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.managers.FormManager;

/**
 * Migration005.
 *
 * @author $Author: chatellier $
 * @version $Revision: 875 $ $Date: 2010-05-07 18:08:06 +0200 (ven., 07 mai 2010) $
 * @since 3 févr. 2010 12:21:20
 */
public class Migration005 {
    private static final Log log = LogFactory.getLog(Migration005.class);

    final File vradi005;
    final File vradi004;
    boolean performMigration = false;
    
    public Migration005() {
        vradi004 = new File(Configuration.getUserHome()
                + File.separator + ".vradi" + File.separator + "vradi-0.0.4");
        vradi005 = Configuration.getInstance().getDataDirAsFile();
    }
    
    /**
     * Prepare data migration from 0.0.4 to 0.0.5 of vradi databases.
     * 
     * The migration only performs if 0.0.4 database exists and if
     * 0.0.5 database does not exists.
     * 
     * Note: Completes successfully or throws UnhandledException.
     */
    public void prepareMigration() {
        if (!isDataDirExists(vradi004)) {
            if (log.isDebugEnabled()) {
                log.debug("Directory " + vradi004 + " doesn't exists");
            }
            return;
        }
        
        if (isDataDirExists(vradi005)) {
            if (log.isDebugEnabled()) {
                log.debug("Directory " + vradi005 + " already exists");
            }
            return;
        }
        
        log.info("Begin of data migration from 0.0.4 to 0.0.5");
        performMigration = true;
        
        try {
            // create backup from previous version
            archivePreviousVersion(vradi004);
            
            // rename vradi datadir
            vradi004.renameTo(vradi005);
            
            // perform some updates via jdbc
            updateExtensionVersion();
            
        } catch (Exception e) {
            throw new UnhandledException("Prepare migration failed", e);
        }
    }
    
    /**
     * Performs a datas migration from 0.0.4 to 0.0.5 of vradi databases.
     * 
     * Note: Completes successfully or throws UnhandledException.
     */
    public void performMigration(WikittyProxy proxy) {
        try {
            if (performMigration) {
                performMigration = false;
                
                // perform migration
                updateStatusNextVersion(proxy);
                // change navigability
                deleteUserClientNextVersion(proxy);
                cleanThesaurus(proxy);
                rebuildIndex(proxy);

                log.info("End of data migration !");
            }
        } catch (Exception e) {
            try {
                FileUtils.deleteDirectory(vradi005);
            } catch (IOException e1) {
            }
            
            throw new UnhandledException("Data migration failed", e);
        }
    }

    /**
     * Rebuild solr index.
     */
    protected void rebuildIndex(WikittyProxy proxy) {

        if (log.isInfoEnabled()) {
            log.info("Reindexing solr...");
        }

        proxy.getWikittyService().syncEngin();

    }

    /**
     *  check if dataDir exists and is not empty.
     */
    protected boolean isDataDirExists(File dataDir) {
        if (!dataDir.exists()) {
            return false;
        } else {
            String[] list = dataDir.list();
            
            if (list.length == 0) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Create a backup archive of previous vradi data dir.
     * @throws IOException
     */
    void archivePreviousVersion(File dataDir) throws IOException {
        File archive = new File(dataDir.getAbsolutePath() + ".zip");
        log.info("Creating backup archive: " + archive.getAbsolutePath());
        ZipUtil.compress(archive, dataDir);
    }
    
    void updateStatusNextVersion(WikittyProxy proxy) {
        try {
            Search search = Search.query().eq(Element.ELT_EXTENSION, Status.EXT_STATUS);
            Criteria criteria = search.criteria();
            PagedResult<Status> statuses = proxy.findAllByCriteria(Status.class, criteria);
            if (statuses.getNumFound() > 0) {
                for (Status status : statuses.getAll()) {
                    if ("Validé".equals(status.getName())) {
                        FormStatus selectionne = VradiConstants.FormStatus.SELECTIONNE;
                        status.setName(selectionne.getName());
                        status.setValue(selectionne.getValue());
                        status.setDescription(selectionne.getDescription());
                        status.setModifiable(false);
                        proxy.store(status);
                        
                    } else if ("À traîter".equals(status.getName())) {
                        FormStatus nonTraite = VradiConstants.FormStatus.NON_TRAITE;
                        status.setName(nonTraite.getName());
                        status.setValue(nonTraite.getValue());
                        status.setDescription(nonTraite.getDescription());
                        status.setModifiable(false);
                        proxy.store(status);
                    }
                }
            }
            
            FormManager formManager = new FormManager(proxy, null, null);
            formManager.getNonSelectionneStatus();
            formManager.getPreselectionneStatus();
            
        } catch (Exception e) {
            throw new WikittyException(e);
        }
    }

    void deleteUserClientNextVersion(WikittyProxy proxy) {
        // clients
        PagedResult<Client> clients = proxy.findAllByCriteria(Client.class, null);
        if (log.isInfoEnabled()) {
            log.info("Delete " + clients.getNumFound() + " clients");
        }
        for (Client client : clients.getAll()) {
            proxy.delete(client.getWikittyId());
        }
        
        // users
        PagedResult<User> users = proxy.findAllByCriteria(User.class, null);
        if (log.isInfoEnabled()) {
            log.info("Delete " + clients.getNumFound() + " users");
        }
        for (User user : users.getAll()) {
            proxy.delete(user.getWikittyId());
        }
    }

    /**
     * Clean thesaurus. Delete refering chidren who don't exist.
     * 
     * WARN: this is updating the source database
     * @throws Exception
     */
    protected void cleanThesaurus(WikittyProxy proxy) throws Exception {
        
        Criteria thesaurusCriteria = Search.query().eq(Element.ELT_EXTENSION,
                TreeNodeImpl.EXT_TREENODE).criteria();
        
        // use deperecated class to access protected methods
        MigrationWikittyService wikittyService = (MigrationWikittyService)
                proxy.getWikittyService();
        
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            HashSet<String> deleteList = new HashSet<String>();
            
            PagedResult<String> wikittyIds = wikittyService.getSearchEngin()
                    .findAllByCriteria(transaction, thesaurusCriteria);
            
            for (String wikittyId : wikittyIds.getAll()) {
                boolean exists = wikittyService.getWikittyStorage().exists(
                        transaction, wikittyId);
                
                if (!exists) {
                    deleteList.add(wikittyId);
                } else {
                    cleanThesaurusChildren(wikittyService, transaction, wikittyId);
                }
            }
            
            log.info("Deleting: " + deleteList);
            wikittyService.getSearchEngin().delete(transaction, deleteList);
            transaction.commit();
            
        } catch (Exception e) {
            transaction.rollback();
            throw e;
        }
    }
    
    private void cleanThesaurusChildren(MigrationWikittyService wikittyService,
            WikittyTransaction transaction, String wikittyId) {

        Wikitty wikitty = wikittyService.getWikittyStorage().restore(transaction, wikittyId);
        TreeNodeImpl node = new TreeNodeImpl(wikitty);
        Set<String> children = node.getChildren();
        
        if (children != null) {
            log.info("Checking childrens :  " + children);

            StringBuilder sb = new StringBuilder();
            for (String id : children) {
                boolean exists = wikittyService.getWikittyStorage().exists(
                        transaction, id);
                
                if (!exists) {
                    sb.append(id);
                    sb.append(',');
                }
            }

            if (sb.length() > 0) {
                log.info("Deleting children :  " + sb.toString());
                String[] splits = sb.toString().split(",");
                for (String split : splits) {
                    node.removeChildren(split);
                }
                log.info("Updating: " + wikittyId);
                log.info("   Removed thesaurus children: " + Arrays.toString(splits));
                wikittyService.getWikittyStorage().store(transaction,
                        Arrays.asList(wikitty), true);
            }
        }
    }
    
    
    /**
     * replace extensions ids to conform new wikitty-1.3,
     * eg. replace VradiUser[1] with VradiUser[1.0].
     * 
     * WARN: this is updating the source database
     */
    void updateExtensionVersion() {
        Connection connection = null;
        try {
            Properties conf = Configuration.getInstance().getJdbcConfigProperties();
            if (conf.isEmpty()) {
                conf = WikittyJDBCUtil.loadProperties(null);
            }
            Class.forName(conf.getProperty(WikittyJDBCUtil.JDBC_DRIVER));
            connection = WikittyJDBCUtil.getJDBCConnection(conf);
            
            /*
             * EXTENSION_DATA
             */
            Statement updateStatement = connection.createStatement();
            updateStatement.executeUpdate("ALTER TABLE EXTENSION_DATA SET REFERENTIAL_INTEGRITY FALSE");
            updateStatement.close();
            
            Statement statement = connection.createStatement();
            ResultSet resultset = statement.executeQuery("select ID from EXTENSION_DATA");
            while (resultset.next()) {
                String id = resultset.getString("ID");
                
                String name = WikittyExtension.computeName(id);
                String computeVersion = WikittyExtension.computeVersion(id);
                String computeVersion2 = computeVersion(id);
                
                if (!computeVersion.equals(computeVersion2)) {
                    String newID = WikittyExtension.computeId(name, computeVersion);
                    String update = "update EXTENSION_DATA set ID = '%s' where ID = '%s'";
                    
                    updateStatement = connection.createStatement();
                    updateStatement.executeUpdate(String.format(update, newID, id));
                    updateStatement.close();
                }
            }
            statement.close();
            
            /*
             * EXTENSION_ADMIN
             */
            statement = connection.createStatement();
            resultset = statement.executeQuery("select ID from EXTENSION_ADMIN");
            while (resultset.next()) {
                String id = resultset.getString("ID");
                
                String name = WikittyExtension.computeName(id);
                String computeVersion = WikittyExtension.computeVersion(id);
                String computeVersion2 = computeVersion(id);
                
                if (!computeVersion.equals(computeVersion2)) {
                    String newID = WikittyExtension.computeId(name, computeVersion);
                    String update = "update EXTENSION_ADMIN set ID = '%s', VERSION = '%s' where ID = '%s'";
                    
                    updateStatement = connection.createStatement();
                    updateStatement.executeUpdate(String.format(update, newID, computeVersion, id));
                    updateStatement.close();
                }
            }
            statement.close();
            
            updateStatement = connection.createStatement();
            updateStatement.executeUpdate("ALTER TABLE EXTENSION_DATA SET REFERENTIAL_INTEGRITY TRUE");
            updateStatement.close();
            
            
            /*
             * WIKITTY_ADMIN
             */
            statement = connection.createStatement();
            resultset = statement.executeQuery("select ID, EXTENSION_LIST from WIKITTY_ADMIN");
            while (resultset.next()) {
                String id = resultset.getString("ID");
                String extensionList = resultset.getString("EXTENSION_LIST");
                
                if (extensionList != null && !extensionList.isEmpty()) {
                    StringBuilder extensionListBuilder = new StringBuilder();
                    
                    for (String extensionId : extensionList.split(",")) {
                        String name = WikittyExtension.computeName(extensionId);
                        String computeVersion = WikittyExtension.computeVersion(extensionId);
                        String computeVersion2 = computeVersion(extensionId);
                        
                        if (!computeVersion.equals(computeVersion2)) {
                            extensionListBuilder.append(WikittyExtension.computeId(name, computeVersion)).append(",");
                            
                        } else {
                            extensionListBuilder.append(extensionId).append(",");
                        }
                    }
                    
                    String newExtensionList = extensionListBuilder.substring(0, extensionListBuilder.length() - 1);
                    String update = "update WIKITTY_ADMIN set EXTENSION_LIST = '%s' where ID = '%s'";

                    updateStatement = connection.createStatement();
                    updateStatement.executeUpdate(String.format(update, newExtensionList, id));
                    updateStatement.close();
                }
                
            }
            statement.close();
            
            connection.commit();
            
        } catch (ClassNotFoundException e) {
            throw new WikittyException(e);
            
        } catch (Exception e) {
            try {
                if (connection != null) {
                    connection.rollback();
                }
            } catch (SQLException e1) {
            }
            throw new WikittyException(e);
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
            } catch (SQLException e1) {
            }
        }
    }
    
    String computeVersion(String id) {
        int b = id.lastIndexOf("[");
        int e = id.lastIndexOf("]");
        String result = null;
        if (b != -1 && e != -1) {
            result = id.substring(b + 1, e);
        }
        return result;
    }
    
    public static void main(String[] args) {
        try {
            ServiceFactory.getWikittyProxy();
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.exit(0);
    }
}
