/*
 * #%L
 * SGQ :: Business
 * $Id: BatchDAOImpl.java 145 2012-10-09 12:44:07Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.4/sgq-business/src/main/java/com/herbocailleau/sgq/entities/BatchDAOImpl.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.entities;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.nuiton.topia.TopiaException;

import com.herbocailleau.sgq.business.model.BatchModel;
import com.herbocailleau.sgq.business.model.SearchModel;

public class BatchDAOImpl<E extends Batch> extends BatchDAOAbstract<E> {

    /**
     * Count non expired and valid batch.
     * 
     * @return non expired and valid batch
     * @throws TopiaException 
     */
    public long countNonExpired() throws TopiaException {
        String query = "select count(*)" +
                " from " + Batch.class.getName() +
                " where " + Batch.PROPERTY_INVALID + " = false" +
                " and " + Batch.PROPERTY_EXPIRED_DATE + " = null";
        
        Number result = (Number)context.findUnique(query);
        return result.longValue();
    }

    /**
     * Retourne le prochain id non utilisé parmis la liste des numeros
     * de lots entre start et end.
     * 
     * @param start range start
     * @param end range end
     * @return next non used batch number
     */
    public int getNextRangeIdBetween(int start, int end) throws TopiaException {
        String query = "select min(main." + Batch.PROPERTY_NUMBER + " + 1)" +
            " from " + Batch.class.getName() + " as main," +
            " " + Batch.class.getName() + " as sub" +
            " where :start < main." + Batch.PROPERTY_NUMBER +
            " and main." + Batch.PROPERTY_NUMBER + " < :end" +
            " and main." + Batch.PROPERTY_NUMBER + " + 1 not in (sub.number) ";

        Number id = (Number)context.findUnique(query, "start", start, "end", end);

        int result = 0;
        if (id == null) {
            // nothing found in range start..end, just return start
            result = start;
        } else {
            result = id.intValue();
        }

        return result;
    }

    /**
     * Return all batches between range of numbers (non elapsed or in error).
     * 
     * @param start start range
     * @param end end range
     * @return all batches between range of numbers
     * @throws TopiaException
     */
    public List<Batch> findAllBetweenNumbers(int start, int end) throws TopiaException {
        String query = " from " + Batch.class.getName() +
            " where :start <= " + Batch.PROPERTY_NUMBER +
            " and " + Batch.PROPERTY_NUMBER + " <= :end" +
            " and " + Batch.PROPERTY_INVALID + " = false" +
            " and " + Batch.PROPERTY_EXPIRED_DATE + " = null";

        List<Batch> result = context.findAll(query, "start", start, "end", end);

        return result;
    }

    /**
     * Retourne des lots en incluant dans le modèle, la quantité totale
     * restante pour l'ensemble du lot...
     * 
     * @param search search model
     * @return
     * @throws TopiaException
     */
    public List<BatchModel> findAllModel(SearchModel search) throws TopiaException {
        return findAllModel(search, 0, -1);
    }

    /**
     * Retourne des lots en incluant dans le modèle, la quantité totale
     * restante pour l'ensemble du lot...
     * 
     * @param search search model
     * @param offset offset
     * @param limit limit (use -1 for no limit)
     * @return
     * @throws TopiaException
     */
    public List<BatchModel> findAllModel(SearchModel search, int offset, int limit) throws TopiaException {
        String queryPrefix = getFindBatchModelQueryPrefix();

        String querySuffix = " order by B." + Batch.PROPERTY_NUMBER + " ASC";

        List<Object[]> beanAndQts = (List<Object[]>)performQueryWithFilter(search, queryPrefix, querySuffix, offset, limit);
        List<BatchModel> results = new ArrayList<BatchModel>(beanAndQts.size());
        for (Object[] beanAndQt : beanAndQts) {
            Double pres = (Double)beanAndQt[1];
            Double sale = (Double)beanAndQt[2];
            Date dmes = (Date)beanAndQt[3];
            results.add(new BatchModel((Batch)beanAndQt[0], pres == null ? 0 : pres, sale == null ? 0 : sale, dmes));
        }
        return results;
    }

    /**
     * Retourne des lots en incluant dans le modèle, la quantité totale
     * restante pour l'ensemble du lot.
     * 
     * @param search search model
     * @return
     * @throws TopiaException
     */
    public long findAllCount(SearchModel search) throws TopiaException {
        String queryPrefix = "select count(B)";

        Number result = (Number)performQueryWithFilter(search, queryPrefix, "", -1, -1);
        return result.longValue();
    }

    /**
     * Retourne le prefix de requete sql qui permet d'obtenir les meta
     * infos en plus du batch pour remplir le model
     * 
     * @return query prefix
     */
    protected String getFindBatchModelQueryPrefix() {
        String query = "select B," +
                " (select sum(P.quantity) from " + Presentation.class.getName() + " P where P.batch = B)," +
                " (select sum(S.quantity) from " + Expedition.class.getName() + " S where S.presentation.batch = B)," +
                " (select min(S.date) from " + Expedition.class.getName() + " S where S.presentation.batch = B)";
        return query;
    }

    /**
     * Construit une requete ou les champs sont comparés sans tenir
     * compte de la case et de la présence d'accents.
     * 
     * @return query
     */
    protected static String getFieldLikeInsensitive(String field1, String field2) {
        String query = "REGEXP_REPLACE(REGEXP_REPLACE(REGEXP_REPLACE(REGEXP_REPLACE(REGEXP_REPLACE(lower(" +
            field1 + "),'é|è|ê','e'),'à|â','a'),'î|ï','i'),'ô','o'),'ù','u') like lower(" + field2 + ")";
        return query;
    }

    /**
     * Construit la requete réutilisable avec des bout de requete (avant/après)
     * suivant ce que doit effectivement retourner la requete.
     * 
     * @param search le filtre de recherche
     * @param queryPrefix prefix
     * @param querySuffix suffix
     * @param offset offset (use -1 for no limit)
     * @param limit offset (use -1 for no limit)
     * @return le resultat (le type dépend du prefix)
     * @throws TopiaException 
     */
    protected Object performQueryWithFilter(SearchModel search, String queryPrefix,
            String querySuffix, int offset, int limit) throws TopiaException {

        String query = queryPrefix;
        query += " from " + Batch.class.getName() + " B";
        query += " where " + Batch.PROPERTY_INVALID + " != true";

        List<Object> params = new ArrayList<Object>();

        // query field
        if (StringUtils.isNotBlank(search.getQuery())) {
            query += " and (B.number = :batch";
            int batch = -1;
            try {
                batch = Integer.parseInt(search.getQuery());
            } catch (NumberFormatException ex) {
                
            }
            params.add("batch");
            params.add(batch);
            query += " or " + getFieldLikeInsensitive("B.product.name", ":query") + ")";
            params.add("query");
            params.add("%" + StringUtils.stripAccents(search.getQuery()) + "%");
        }

        // client field
        if (StringUtils.isNotBlank(search.getClient())) {
            query += " and " + getFieldLikeInsensitive("B.dedicatedClient.name", ":client");
            params.add("client");
            params.add("%" + StringUtils.stripAccents(search.getClient()) + "%");
        }

        // supplier
        if (StringUtils.isNotBlank(search.getSupplier())) {
            query += " and " + getFieldLikeInsensitive("B.supplier.name", ":supplier");
            params.add("supplier");
            params.add("%" + StringUtils.stripAccents(search.getSupplier()) + "%");
        }

        // origin
        if (search.getOrigin() != null) {
            query += " and B.origin = :origin";
            params.add("origin");
            params.add(search.getOrigin());
        }

        // product status
        if (CollectionUtils.isNotEmpty(search.getProductStatus())) {
            int index = 0;
            query += " and (1=1";
            for (ProductStatus status : search.getProductStatus()) {
                query += " AND :status" + index + " member of B.product.productStatus";
                params.add("status" + index);
                params.add(status);
                index++;
            }

            query += ")";
        }

        // product categories
        if (CollectionUtils.isNotEmpty(search.getProductCategories())) {
            int index = 0;
            query += " and (0=1";
            for (String cat : search.getProductCategories()) {
                query += " OR :category" + index + " = B.product.category";
                params.add("category" + index);
                params.add(cat);
                index++;
            }
            query += ")";
        }

        // dates
        if (search.getDateType() != null && (search.getBeginDate() != null || search.getEndDate() != null)) {
            String dateField = null;
            switch (search.getDateType()) {
            case ENTRY_DATE:
                dateField = Batch.PROPERTY_ENTRY_DATE;
                break;
            case DLUO:
                dateField = Batch.PROPERTY_DLUO;
                break;
            case DMES:
                dateField = "(select min(S.date) from " + Expedition.class.getName() + " S where S.presentation.batch = B)";
                break;
            case DMESD:
                dateField = Batch.PROPERTY_DMESD;
                break;
            case DPMES:
                dateField = Batch.PROPERTY_DPMES;
                break;
            case EXPIRED_DATE:
                dateField = Batch.PROPERTY_EXPIRED_DATE;
                break;
            }
            
            if (search.getBeginDate() != null) {
                query += " AND :beginDate <= " + dateField;
                params.add("beginDate");
                params.add(search.getBeginDate());
            }
            if (search.getEndDate() != null) {
                // to fix interval, take date at midnigth
                Date date = DateUtils.addDays(search.getEndDate(), 1);
                date = DateUtils.addMilliseconds(date, -1);
                query += " AND " + dateField + " <= :endDate";
                params.add("endDate");
                params.add(date);
            }
        }

        // lots epuises
        if (!search.isExpired()) {
            query += " and " + Batch.PROPERTY_EXPIRED_DATE + " = null";
        }

        // add final suffix to query
        query += querySuffix;

        // perform query
        Object result = null;
        if (offset == -1) {
            result = context.findUnique(query, params.toArray());
        } else if (limit == -1) {
            result = context.findAll(query, params.toArray());
        } else {
            result = context.find(query, offset, offset + limit - 1, params.toArray());
        }
        return result;
    }

    /**
     * Retourne un batch model (batch + meta info) à partir du topia id.
     * 
     * En plus du lot, il y a des informations issue de recherche par sous
     * requettes :
     * <ul>
     * <li>la quantité totale disponible pour l'ensemble des presentations
     * <li>la quantité totale vendue (expediée)
     * <li>la date de mise en service (DMES)
     * </ul>
     * 
     * @return le batch model
     * @throws TopiaException
     */
    public BatchModel findModelByTopiaId(String batchId) throws TopiaException {
        String query = getFindBatchModelQueryPrefix() +
                " from " + Batch.class.getName() + " B" +
                " where B." + Batch.TOPIA_ID + " = :batchId";

        BatchModel result = null;
        Object[] beanAndQt = (Object[])context.findUnique(query, "batchId", batchId);
        if (beanAndQt != null) {
            Double pres = (Double)beanAndQt[1];
            Double sale = (Double)beanAndQt[2];
            Date dmes = (Date)beanAndQt[3];
            result = new BatchModel((Batch)beanAndQt[0], pres == null ? 0 : pres, sale == null ? 0 : sale, dmes);
        }
        return result;
    }

} //BatchDAOImpl<E extends Batch>
