/*
 * #%L
 * SGQ :: Business
 * $Id: ProductionService.java 48 2012-09-11 12:46:57Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.2/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/ProductionService.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;

import au.com.bytecode.opencsv.CSVReader;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.BatchDAO;
import com.herbocailleau.sgq.entities.Presentation;
import com.herbocailleau.sgq.entities.PresentationCode;
import com.herbocailleau.sgq.entities.PresentationDAO;
import com.herbocailleau.sgq.entities.Sale;
import com.herbocailleau.sgq.entities.SaleDAO;

/**
 * Service gerant principalement:
 * <ul>
 * <li>Les ventes et transformations
 * </ul>
 * 
 * @author echatellier
 */
public class ProductionService extends SgqService {

    private static final Log log = LogFactory.getLog(ProductionService.class);

    /**
     * Recherche parmis des les présentations ou les ventes la dernière date
     * d'une données qui a donné lieu à un enregistrement en base de données.
     * 
     * C'est ensuite à partir de cette date qu'il faut reprendre l'import
     * des données jusqu'à la veille de la date du jour.
     * 
     * @return la date du dernier enregistrement, ou {@code null} si aucun
     *  enregistrement d'existe en base.
     */
    public Date getProductionStartDate() {
        return null;
    }

    /**
     * Recherche parmis des les présentations ou les ventes la dernière date
     * d'une données qui a donné lieu à un enregistrement en base de données.
     * 
     * C'est ensuite à partir de cette date qu'il faut reprendre l'import
     * des données jusqu'à la veille de la date du jour.
     * 
     * @return la date du dernier enregistrement, ou {@code null} si aucun
     *  enregistrement d'existe en base.
     */
    public Date getSaleStartDate() {
        return null;
    }

    /**
     * Import label file.
     * Use file name to import production file or expedition file.
     * 
     * @param fileFilename
     * @param file
     */
    public void importLabelsFile(String fileFilename, File file) {

        String productionFileName = config.getLabelProductionFilename();
        if (productionFileName.equalsIgnoreCase(productionFileName)) {
            importLabelsProductionFile(file);
        } else {
            String expeditionFileName = config.getLabelProductionFilename();
            if (expeditionFileName.equalsIgnoreCase(productionFileName)) {
                importLabelsExpeditionFile(file);
            } else {
                throw new SgqBusinessException(_("Invalid name '%s' for label file", fileFilename));
            }
        }
    }

    /**
     * Le fichier est structuré ainsi:
     * D,20110701,B,147136,C,11520,P,PFEN21,L,20036,Q,40.000
     * 
     * Date 
     * B : N° de Bon de commande interne
     * C : Code client
     * P : Code présentation + Code plante
     * L : N° de lot
     * Q : Quantité
     * R : Remplacement lot
     * 
     * @param file file to import
     */
    protected void importLabelsProductionFile(File file) {
        if (log.isInfoEnabled()) {
            log.info("Importing production file : " + file);
        }

        try {
            CSVReader reader = new CSVReader(new FileReader(file));

            // get date format
            DateFormat df = new SimpleDateFormat("yyyyMMdd");

            // get dao
            //ClientDAO clientDAO = daoHelper.getClientDAO();
            //ProductDAO productDAO = daoHelper.getProductDAO();
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            SaleDAO saleDAO = daoHelper.getSaleDAO();

            String[] data = null;
            while ((data = reader.readNext()) != null) {
                if (data.length <= 1) {
                    continue;
                }

                String dateddMMyy = data[1];
                String clientId = data[5];
                String productId = data[7];
                String batchId = data[9];
                String quantityStr = data[11];

                Date date = df.parse(dateddMMyy);
                Batch batch = batchDAO.findByNumber(Integer.parseInt(batchId));
                if (batch == null) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't find batch " + batchId);
                    }
                    continue;
                }
                char presentationChar = productId.charAt(0);
                PresentationCode presentationCode = PresentationCode.getPresentationCodeFor(presentationChar);
                double quantity = Double.parseDouble(quantityStr);
                int client = Integer.parseInt(clientId);

                // production
                List<Presentation> presentations = presentationDAO.findAllByBatch(batch);

                boolean originPresentationFound = false;
                Presentation destPresentation = null;
                for (Presentation presentation : presentations) {
                    if (presentation.getPresentationCode() == PresentationCode._) {
                        originPresentationFound = true;

                        double presQuantity = presentation.getQuantity();
                        presQuantity -= quantity;
                        presentation.setQuantity(presQuantity);
                        presentationDAO.update(presentation);
                    }

                    // les deux cas peuvent être vrai en même temps
                    if (presentationCode == presentation.getPresentationCode()) {
                        destPresentation = presentation;
                        /*double presQuantity = presentation.getQuantity();
                        presQuantity += quantity;
                        presentation.setQuantity(presQuantity);
                        presentationDAO.update(presentation);*/
                    }
                }

                if (SgqUtils.isInternalClient(config, client)) {
                    if (!originPresentationFound) {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find original presentation for row " + Arrays.toString(data));
                        }
                    }

                    // dans le cas ou la presentation produite n'existe pas,
                    // on doit en creer une nouvelle
                    if (destPresentation == null) {
                        destPresentation = presentationDAO.create();
                        destPresentation.setBatch(batch);
                        destPresentation.setName(batch.getProduct().getName());
                        destPresentation.setPresentationCode(presentationCode);

                        if (log.isDebugEnabled()) {
                            log.debug("New presentation for batch " + batch.getNumber());
                        }
                    }

                    double presQuantity = destPresentation.getQuantity();
                    presQuantity += quantity;
                    destPresentation.setQuantity(presQuantity);
                    presentationDAO.update(destPresentation);

                } else {

                    if (destPresentation == null) {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find expedition presentation for batch " + batch.getNumber());
                        }
                    } else {
                        // reduction de la quantité de la presentation
                        double presQuantity = destPresentation.getQuantity();
                        presQuantity += quantity;
                        destPresentation.setQuantity(presQuantity);
                        presentationDAO.update(destPresentation);
                        
                        // ajout d'une expedition
                        Sale sale = saleDAO.create();
                        sale.setDate(date);
                        sale.setPresentation(destPresentation);
                        sale.setQuantity(quantity);
                        saleDAO.update(sale);
                    }
                }
            }
            
            reader.close();
            daoHelper.commit();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } catch (ParseException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        }
    }

    protected void importLabelsExpeditionFile(File file) {
        if (log.isInfoEnabled()) {
            log.info("Importing expedition file : " + file);
        }
    }
}
