package com.franciaflex.faxtomail.ui.swing.util;

/*
 * #%L
 * FaxToMail :: UI
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.DemandType;
import com.franciaflex.faxtomail.persistence.entities.ExtensionCommand;
import com.franciaflex.faxtomail.persistence.entities.MailField;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.persistence.entities.Reply;
import com.franciaflex.faxtomail.persistence.entities.ReplyContent;
import com.franciaflex.faxtomail.services.FaxToMailServiceContext;
import com.franciaflex.faxtomail.services.FaxToMailServiceUtils;
import com.franciaflex.faxtomail.services.service.ConfigurationService;
import com.franciaflex.faxtomail.services.service.EmailService;
import com.franciaflex.faxtomail.ui.swing.FaxToMailUIContext;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIHandler;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.content.demande.RangeRowModel;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.pdf.PdfWriter;
import com.sun.media.jai.codec.ByteArraySeekableStream;
import com.sun.media.jai.codec.ImageCodec;
import com.sun.media.jai.codec.ImageDecoder;
import com.sun.media.jai.codec.SeekableStream;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.nuiton.jaxx.application.ApplicationTechnicalException;
import org.nuiton.jaxx.application.swing.util.ApplicationUIUtil;
import org.nuiton.util.FileUtil;
import org.nuiton.util.StringUtil;

import javax.media.jai.PlanarImage;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.swing.Box;
import javax.swing.Icon;
import javax.swing.JEditorPane;
import javax.swing.JPanel;
import javax.swing.JTextPane;
import javax.swing.JTree;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import java.awt.Color;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin - morin@codelutin.com
 *
 */
public final class FaxToMailUIUtil extends ApplicationUIUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(FaxToMailUIUtil.class);

    public static final List<String> EDITABLE_EXTENSIONS = Lists.newArrayList("PDF", "PNG", "JPG", "JPEG", "GIF",
                                                                              "TIF", "BMP", "TXT");

    private FaxToMailUIUtil() {
        // never instanciate util class
    }

    public static FaxToMailUIContext getApplicationContext(JAXXObject ui) {
        return (FaxToMailUIContext) ApplicationUIUtil.getApplicationContext(ui);
    }

    public static void setParentUI(JAXXObject ui, FaxToMailUI<?, ?> parentUI) {
        JAXXUtil.initContext(ui, parentUI);
        setApplicationContext(ui, parentUI.getHandler().getContext());
    }

    public static Map<MailFolder, FolderTreeNode> initFolderTree(final FaxToMailUIContext context,
                                                                         JTree navigationTree,
                                                                         Collection<MailFolder> folders,
                                                                         List<MailFolder> foldersToExpand,
                                                                         final boolean colorNotAffectedFolders) {
        DefaultMutableTreeNode root = new DefaultMutableTreeNode("root");

        Map<MailFolder, FolderTreeNode> nodesByFolder = new HashMap<MailFolder, FolderTreeNode>();
        List<MailFolder> orderedFolders = new ArrayList<MailFolder>(folders);
        Collections.sort(orderedFolders, Ordering.natural().onResultOf(new Function<MailFolder, Comparable>() {
            @Override
            public Comparable apply(MailFolder folder) {
                return folder.getName();
            }
        }));
        for (MailFolder folder : orderedFolders) {
            nodesByFolder.putAll(FaxToMailUIUtil.createFolderTree(context, root, folder));

            FolderTreeNode node = nodesByFolder.get(folder);
            MailFolder parent = folder.getParent();

            while (parent != null) {
                FolderTreeNode parentNode = nodesByFolder.get(parent);
                if (parentNode == null) {
                    parentNode = new FolderTreeNode(parent);
                    parentNode.setCanRead(false);
                    parentNode.setCanSelect(true);
                    nodesByFolder.put(parent, parentNode);
                }

                parentNode.add(node);
                node = parentNode;
                parent = parent.getParent();
            }
            root.add(node);
        }

        TreeModel treeModel = new DefaultTreeModel(root);
        navigationTree.setModel(treeModel);

        // use the folder icon for the leaf
        DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer() {
            @Override
            public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel,
                                                          boolean expanded, boolean leaf, int row, boolean hasFocus) {
                Component component = super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);

                if (FolderTreeNode.class.isAssignableFrom(value.getClass())) {
                    FolderTreeNode node = (FolderTreeNode) value;
                    setEnabled(node.isCanSelect());

                    // En italic si l'utilsateur n'a pas le droit de lecture (sur un parent par exemple)
                    Font font = component.getFont();
                    if (node.isCanRead()) {
                        font = font.deriveFont(Font.PLAIN);
                    } else {
                        font = font.deriveFont(Font.ITALIC);
                    }
                    component.setFont(font);

                    // Affichage en gris par defaut, mais en noir si le dossier est vraiment affecté
                    // à l'utilisateur
                    Color foreground;
                    if (!colorNotAffectedFolders
                            || context.getCurrentUser().isAffectedFoldersEmpty()
                            || context.getCurrentUser().containsAffectedFolders(node.getMailFolder())) {

                        foreground = sel ? Color.WHITE : Color.BLACK;

                    } else {
                        foreground = Color.GRAY;
                    }
                    component.setForeground(foreground);

                }

                return component;
            }
        };
        Icon folderIcon = renderer.getDefaultClosedIcon();
        renderer.setLeafIcon(folderIcon);
        navigationTree.setCellRenderer(renderer);

        // expand nodes
        for (MailFolder folder : foldersToExpand) {
            DefaultMutableTreeNode node = nodesByFolder.get(folder);
            // le node peut être null, s'il n'est pas dans l'arbre car
            // l'utilisateur n'a pas les droits de déplacement par exemple
            if (node != null) {
                TreePath treePath = new TreePath(node.getPath());
                navigationTree.expandPath(treePath);
            }
        }

        return nodesByFolder;
    }

    public static List<MailFolder> getExpandedFolders(JTree tree) {
        TreeModel treeModel = tree.getModel();
        Enumeration<TreePath> paths = tree.getExpandedDescendants(new TreePath(treeModel.getRoot()));
        java.util.List<MailFolder> folders = new ArrayList<MailFolder>();
        if (paths != null) {
            while (paths.hasMoreElements()) {
                TreePath path = paths.nextElement();
                Object lastPathComponent = path.getLastPathComponent();
                if (FolderTreeNode.class.isAssignableFrom(lastPathComponent.getClass())) {
                    folders.add(((FolderTreeNode) lastPathComponent).getMailFolder());
                }
            }
        }
        return folders;
    }

    /**
     * Creates the tree nodes and returns a map of the nodes by folder
     * @param parent the node parent
     * @param folder the folder to add to the tree
     * @return the map of the nodes by folder
     */
    protected static Map<MailFolder, FolderTreeNode> createFolderTree(FaxToMailUIContext context,
                                                                      DefaultMutableTreeNode parent,
                                                                       MailFolder folder) {
        Map<MailFolder, FolderTreeNode> result = new HashMap<MailFolder, FolderTreeNode>();

        FolderTreeNode node = new FolderTreeNode(folder);
        node.setCanSelect(true);
        node.setCanRead(true);
        result.put(folder, node);

        parent.add(node);
        if (folder.isChildrenNotEmpty()) {
            List<MailFolder> children = new ArrayList<MailFolder>(folder.getChildren());
            CollectionUtils.filter(children, new Predicate<MailFolder>() {
                @Override
                public boolean evaluate(MailFolder object) {
                    return !object.isArchiveFolder();
                }
            });
            Collections.sort(children, new Comparator<MailFolder>() {
                @Override
                public int compare(MailFolder o1, MailFolder o2) {
                    return o1.getName().compareTo(o2.getName());
                }
            });
            for (MailFolder child : children) {
                result.putAll(createFolderTree(context, node, child));
            }
        }

        // set the folder instance with the needed data in the context
        MailFolder currentFolder = context.getCurrentMailFolder();
        if (currentFolder != null && currentFolder.getTopiaId().equals(folder.getTopiaId())) {
            context.setCurrentMailFolder(folder);
        }

        return result;
    }

    public static Map<String, Integer> computeQuantities(Collection<RangeRowModel> rangeRows) {
        int quotationNb = 0;
        int pfNb = 0;
        int savNb = 0;
        for (RangeRowModel rangeRow : rangeRows) {
            if (rangeRow.isValid()) {
                Integer quotationQuantity = rangeRow.getQuotationQuantity();
                if (quotationQuantity != null) {
                    quotationNb += quotationQuantity;
                }

                Integer productQuantity = rangeRow.getProductQuantity();
                if (productQuantity != null) {
                    pfNb += productQuantity;
                }

                Integer savQuantity = rangeRow.getSavQuantity();
                if (savQuantity != null) {
                    savNb += savQuantity;
                }
            }
        }
        Map<String, Integer> result = new HashMap<String, Integer>();
        result.put(DemandeUIModel.PROPERTY_QUOTATION_NB, quotationNb);
        result.put(DemandeUIModel.PROPERTY_PF_NB, pfNb);
        result.put(DemandeUIModel.PROPERTY_SAV_NB, savNb);
        return result;
    }

    public static boolean isFileAPDF(AttachmentFile file) {
        return FileUtil.extension(file.getFilename()).toUpperCase().equals("PDF");
    }

    public static boolean isFileATxt(AttachmentFile file) {
        return FileUtil.extension(file.getFilename()).toUpperCase().equals("TXT");
    }

    public static boolean isFileATif(AttachmentFile file) {
        return FileUtil.extension(file.getFilename()).toUpperCase().equals("TIF");
    }

    public static boolean isFileTypeEditable(String fileName) {
        String extension = FileUtil.extension(fileName);
        return EDITABLE_EXTENSIONS.contains(extension.toUpperCase());
    }

    public static void setEmailContentInTextPane(DemandeUIHandler handler,
                                                 DemandeUIModel demandeUIModel,
                                                 JPanel textPanePanel) {

        List<String> contents = demandeUIModel.getHtmlContent();

        if (contents != null) {

            for (String content : contents) {
                if (content != null) {
                    addHtmlTextPane(handler, demandeUIModel, textPanePanel, content);
                }
            }

        }

        // if there is no html content or if no html content can be correctly displayed (ie throws an exception)
        if (textPanePanel.getComponentCount() == 0) {

            contents = demandeUIModel.getPlainContent();

            if (contents != null) {

                for (String content : contents) {

                    if (content != null) {
                        JTextPane textPane = new JTextPane();
                        textPane.setText(content);
                        textPanePanel.add(textPane);
                        textPanePanel.add(Box.createVerticalStrut(3));
                    }
                }
            }
        }
    }

    /**
     * Force le chargement des attachmentFile (edited and original) pour un attachment.
     *
     * @param context context
     * @param attachment attachment
     */
    public static void forceAttachmentFileLoading(FaxToMailUIContext context, Attachment attachment) {
        if (log.isDebugEnabled()) {
            log.debug("Force attachment loading " + attachment.getOriginalFileName());
        }
        if (attachment.isPersisted()) {
            FaxToMailServiceContext serviceContext = context.newServiceContext();
            EmailService service = serviceContext.getEmailService();
            if (attachment.getOriginalFile() == null) {
                AttachmentFile file = service.getAttachmentFile(attachment.getTopiaId(), true);
                attachment.setOriginalFile(file);
            }
            if (attachment.getEditedFile() == null) {
                AttachmentFile file = service.getAttachmentFile(attachment.getTopiaId(), false);
                attachment.setEditedFile(file);
            }
        }
    }
    
    /**
     * Force le chargement des contenu source binaire des réponses.
     *
     * @param context context
     * @param reply reply
     */
    public static void forceReplyContentLoading(FaxToMailUIContext context, Reply reply) {
        if (log.isDebugEnabled()) {
            log.debug("Force source loading " + reply.getSubject());
        }
        if (reply.isPersisted() && reply.getReplyContent() == null) {
            FaxToMailServiceContext serviceContext = context.newServiceContext();
            EmailService service = serviceContext.getEmailService();
            ReplyContent replyContent = service.getReplyContent(reply.getTopiaId());
            reply.setReplyContent(replyContent);
        }
    }

    /**
     * Imprime un attachment file.
     * 
     * @param attachmentFile
     * @param defaultPrinter if {@code true}, do not display print dialog and print with default printer
     * @return true if file has been printed, false otherwise
     */
    public static boolean print(AttachmentFile attachmentFile, boolean defaultPrinter) {
        boolean result = false;
        FileInputStream fileInputStream = null;
        try {
            File file;
            if (!FaxToMailUIUtil.isFileAPDF(attachmentFile)) {
                file = convertFileToPdf(attachmentFile);
            } else {
                file = attachmentFile.getFile();
            }

            fileInputStream = new FileInputStream(file);

            result = printWithPdfRenderer(attachmentFile.getFilename(), fileInputStream, defaultPrinter);
            
        } catch (IOException | DocumentException e) {
            throw new ApplicationTechnicalException(
                    t("jaxx.application.error.cannot.print"), e);

        } finally {
            IOUtils.closeQuietly(fileInputStream);
        }
        return result;
    }

    /**
     * Imprime du texte.
     * 
     * @param printName print job name
     * @param text text to print
     * @param defaultPrinter if {@code true}, do not display print dialog and print with default printer
     * @return true if file has been printed, false otherwise
     */
    public static boolean printText(String printName, String text, boolean defaultPrinter) {
        boolean result = false;
        FileInputStream fileInputStream = null;
        ByteArrayInputStream byteArrayInputStream = null;

        try {
            // convert text content to inputstream
            byte[] content = text.getBytes(Charsets.UTF_8);
            byteArrayInputStream = new ByteArrayInputStream(content);

            File file = convertFileToPdf(byteArrayInputStream, FileType.TEXT);

            fileInputStream = new FileInputStream(file);
            result = printWithPdfRenderer(printName, fileInputStream, defaultPrinter);

        } catch (IOException | DocumentException e) {
            throw new ApplicationTechnicalException(
                    t("jaxx.application.error.cannot.print"), e);

        } finally {
            IOUtils.closeQuietly(fileInputStream);
            IOUtils.closeQuietly(byteArrayInputStream);
        }
        return result;
    }

    /**
     * Imprime un attachment file.
     * 
     * @param printName print job name
     * @param fis stream to print
     * @param defaultPrinter if {@code true}, do not display print dialog and print with default printer
     * @return true if file has been printed, false otherwise
     */
    //TODO kmorin print in an action
    protected static boolean printWithPdfRenderer(String printName, FileInputStream fis, boolean defaultPrinter) {
        boolean result = false;
        PDDocument pdDocument = null;

        try {

            PrinterJob printJob = PrinterJob.getPrinterJob();
            printJob.setJobName(printName);

            // Send print job to default printer
            PrintRequestAttributeSet attributes = new HashPrintRequestAttributeSet();
            result = defaultPrinter || printJob.printDialog(attributes);

            if (result) {
//                pdDocument = PDDocument.loadLegacy(fis);
//                if (pdDocument.isEncrypted()) {
//                    pdDocument.decrypt("");
//                }
                pdDocument = PDDocument.load(fis);
                final PDFRenderer renderer = new PDFRenderer(pdDocument);
                final int numOfPages = pdDocument.getNumberOfPages();

                printJob.setPrintable(new Printable() {
                    public int print(Graphics graphics, PageFormat pageFormat, int pageIndex) throws PrinterException {
                        try {
                            if (pageIndex >= numOfPages) {
                                return NO_SUCH_PAGE;
                            }

                            double paperWidth = (int) pageFormat.getImageableWidth();
                            double paperHeight = (int) pageFormat.getImageableHeight();
                            int paperX = (int) pageFormat.getImageableX();
                            int paperY = (int) pageFormat.getImageableY();

                            if (log.isDebugEnabled()) {
                                log.debug("page width " + paperWidth);
                                log.debug("page height " + paperHeight);
                                log.debug("page x " + paperX);
                                log.debug("page y " + paperY);
                            }

                            BufferedImage image = renderer.renderImageWithDPI(pageIndex, 300);

                            int imageWidth = image.getWidth();
                            int imageHeight = image.getHeight();
                            double widthRatio = paperWidth / imageWidth;
                            double heightRatio = paperHeight / imageHeight;

                            if (log.isDebugEnabled()) {
                                log.debug("width ratio : " + widthRatio);
                                log.debug("height ratio : " + heightRatio);
                            }

                            int width;
                            int height;
                            // if the image is smaller than the page
                            if (widthRatio >= 1 && heightRatio >= 1) {
                                width = imageWidth;
                                height = imageHeight;

                            } else {
                                double minRatio = Math.min(widthRatio, heightRatio);
                                width = (int) (minRatio * imageWidth);
                                height = (int) (minRatio * imageHeight);
                            }

                            if (log.isDebugEnabled()) {
                                log.debug("image width : " + width);
                                log.debug("image height : " + height);
                            }
                            graphics.drawImage(image, paperX, paperY, width, height, null);

                            return PAGE_EXISTS;

                        } catch (Exception e) {
                            if (log.isErrorEnabled()) {
                                log.error("error while printing", e);
                            }
                            return NO_SUCH_PAGE;
                        }
                    }
                });

                printJob.print(attributes);
            }


        } catch (PrinterException ex) {
            if (log.isErrorEnabled()) {
                log.error("can't print", ex);
            }

        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("", e);
            }
        } finally {
            if (log.isDebugEnabled()) {
                log.debug("finally close the pdf file");
            }
            IOUtils.closeQuietly(pdDocument);
        }

        return result;
    }

    public static File convertFileToPdf(AttachmentFile attachmentFile) throws IOException, DocumentException {
        File file = attachmentFile.getFile();
        FileType type;
        if (FaxToMailUIUtil.isFileATxt(attachmentFile)) {
            type = FileType.TEXT;

        } else if (FaxToMailUIUtil.isFileATif(attachmentFile)) {
            type = FileType.TIF;

        } else {
            type = FileType.OTHER;
        }
        File result = convertFileToPdf(new FileInputStream(file), type);
        return result;
    }

    protected enum FileType {
        TEXT, TIF, OTHER
    }

    protected static File convertFileToPdf(InputStream inputStream, FileType type) throws IOException, DocumentException {
        File target = File.createTempFile("faxtomail-", ".tmp");
        target.deleteOnExit();

        Document document = new Document();
        FileOutputStream fos = new FileOutputStream(target);
        PdfWriter writer = PdfWriter.getInstance(document, fos);
        writer.open();

        document.setPageSize(PageSize.A4);
        document.open();

        if (type == FileType.TEXT) {
            BufferedReader br = null;
            try {
                br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
                StringBuilder text = new StringBuilder();
                String line = br.readLine();
                while (line != null) {
                    text.append(line).append("\n");
                    line = br.readLine();
                }
                document.add(new Paragraph(text.toString()));
                br.close();

            } finally {
                IOUtils.closeQuietly(br);
            }

        } else if (type == FileType.TIF) {
            try {
                FileInputStream fis = (FileInputStream) inputStream;
                FileChannel channel = fis.getChannel();
                ByteBuffer buffer = ByteBuffer.allocate((int) channel.size());
                channel.read(buffer);
                SeekableStream stream = new ByteArraySeekableStream(buffer.array());
                String[] names = ImageCodec.getDecoderNames(stream);
                ImageDecoder tifImageDecoder = ImageCodec.createImageDecoder(names[0], stream, null);

                for (int i = 0 ; i < tifImageDecoder.getNumPages() ; i++) {
                    RenderedImage renderedImage = tifImageDecoder.decodeAsRenderedImage(i);
                    java.awt.Image awtImage = PlanarImage.wrapRenderedImage(renderedImage).getAsBufferedImage();
                    document.newPage();
                    com.itextpdf.text.Image image = com.itextpdf.text.Image.getInstance(writer, awtImage, 1.0f);
                    image.scaleToFit(PageSize.A4.getWidth() - document.leftMargin() - document.rightMargin(),
                                     PageSize.A4.getHeight() - document.topMargin() - document.bottomMargin());
                    document.add(image);
                }
                fis.close();
            } finally {
                IOUtils.closeQuietly(inputStream);
            }

        } else {
            byte[] bytes = IOUtils.toByteArray(inputStream);
            com.itextpdf.text.Image image = com.itextpdf.text.Image.getInstance(bytes);
            float pageWidth = document.getPageSize().getWidth()- document.leftMargin() - document.rightMargin();
            float pageHeight = document.getPageSize().getHeight()- document.topMargin() - document.bottomMargin();
            float imageWidth = image.getWidth();
            float imageHeight = image.getHeight();
            if (pageWidth < imageWidth || pageHeight < imageHeight) {
                float scaler = Math.min(pageWidth / imageWidth, pageHeight / imageHeight) * 100;
                image.scalePercent(scaler);
            }
            document.add(image);
        }

        document.close();
        writer.close();

        return target;
    }


    public static String getEditedFileName(String originalFileName) {
        return t("faxtomail.attachment.editedFile.name", originalFileName) + ".pdf";
    }

    /**
     * Ouvre un attachment en fonction de la configuration des extensions, ou à default le open système.
     *
     * @param context
     * @param attachment
     */
    public static void openFile(FaxToMailUIContext context, AttachmentFile attachment) {

        File file = attachment.getFile();
        String filename = attachment.getFilename();
        String extension = FilenameUtils.getExtension(filename);

        // get configuration extension command
        ExtensionCommand extCommand = null;
        if (StringUtils.isNotBlank(extension)) {
            FaxToMailServiceContext serviceContext = context.newServiceContext();
            ConfigurationService service = serviceContext.getConfigurationService();
            extCommand = service.getExtensionCommand(extension);
        }

        // open file
        if (extCommand != null && StringUtils.isNotBlank(extCommand.getOpenAttachmentCommand())) {
            String command = extCommand.getOpenAttachmentCommand();
            String[] args = StringUtil.split(command, " ");
            List<String> comArgs = new ArrayList<String>();
            for (String arg : args) {
                String localArg = arg;
                localArg = localArg.replace("%f", file.getAbsolutePath());
                comArgs.add(localArg);
            }
            ProcessBuilder pb = new ProcessBuilder(comArgs);
            // run process
            if (log.isDebugEnabled()) {
                log.debug("Open attachment with command : " + comArgs);
            }
            try {
                pb.start();
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Cannot run convert command", e);
                }
            }
        } else {

            try {
                if (SystemUtils.IS_OS_WINDOWS) {
                    // tentative pour résoudre http://forge.codelutin.com/issues/5650
                    // pas d'idée précise du bug, mais ca semble mieux fonctionner à l'arrache
                    String command = "rundll32 SHELL32.DLL,ShellExec_RunDLL "+ file.getAbsoluteFile();
                    if (log.isDebugEnabled()) {
                        log.debug("Opening file with command : " + command);
                    }
                    Runtime.getRuntime().exec(command);
                } else {
                    // version to use for xfce
                    // DesktopUtil.open(file);
                    Desktop desktop = getDesktopForOpen();
                    desktop.open(file);
                }
            } catch (Exception ex) {
                throw new ApplicationTechnicalException(
                        t("jaxx.application.error.cannot.open"), ex);
            }
        }
    }

    /**
     * Send email.
     * 
     * @param subject
     * @param body
     */
    public static void email(String subject, String body) {

        try {
            URI mailtoURI = new URI("mailto", null, null, "subject=" + subject + "&body=" + body, null);
            Desktop desktop = getDesktopForMail();
            desktop.mail(mailtoURI);

            // version for xfce
            // DesktopUtil.mail(mailtoURI);

        } catch (Exception e) {

            throw new ApplicationTechnicalException(
                    t("jaxx.application.error.cannot.mail"), e);
        }
    }

    public static boolean isRangePanelVisible(DemandType demandType) {
        return demandType != null && FaxToMailServiceUtils.contains(demandType.getRequiredFields(), MailField.RANGE_ROW);
    }

    protected static void addHtmlTextPane(DemandeUIHandler handler,
                                          DemandeUIModel demandeUIModel,
                                          JPanel textPanePanel,
                                          String content) {

        Preconditions.checkNotNull(content);

        JTextPane textPane = createHtmlTextPane();

        // the meta tag makes the content is not displayed
        content = content.replaceAll("<meta (.*?)>(</meta>)?", "");

        if (log.isTraceEnabled()) {
            log.trace("Content before mail = " + content);
        }

        for (Attachment attachment : demandeUIModel.getAttachment()) {
            String key = attachment.getContentId();
            if (key == null) {
                key = attachment.getOriginalFileName();
            }

            // get file content
            forceAttachmentFileLoading(handler.getContext(), attachment);
            AttachmentFile attachmentFile = attachment.getOriginalFile();
            File file = attachmentFile.getFile();

            // replace the inline attachments with the extracted attachment file url
            // match les patterns:
            // <td background="cid:bg.gif" height="52">
            // <img border=0 src="cid:bg.gif" />
            // <img src='cid:5e9ef859-ea65-4f9b-a9fa-30d4a2c5837c'
            content = content.replaceAll("(\\w+)=([\"'])cid:" + Pattern.quote(key) + "([\"'])", "$1=$2" + file.toURI() + "$3");

            if (log.isDebugEnabled()) {
                log.debug("Mapping attachment id " + key + " to file " + file.toURI());
            }
        }

        if (log.isTraceEnabled()) {
            log.trace("Content after mail = " + content);
        }

        try {
            // on reformate les urls pour supprimer les caractères qui vont pas (ex espaces)
            // cf #7741
            content = FaxToMailServiceUtils.encodeImageSourcesInEmail(content);

            textPane.setText(content);

            textPanePanel.add(textPane);

            textPanePanel.add(Box.createVerticalStrut(3));

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Error when displaying email content", e);
            }
        }
    }

    protected static JTextPane createHtmlTextPane() {
        JTextPane textPane = new JTextPane();
        textPane.setEditable(false);

        textPane.setContentType("text/html");
        HTMLEditorKit htmlEditorKit = new HTMLEditorKit();
        textPane.setEditorKit(htmlEditorKit);

        textPane.putClientProperty(JEditorPane.HONOR_DISPLAY_PROPERTIES, true);
        Font font = new Font(Font.MONOSPACED, Font.PLAIN, 16);
        textPane.setFont(font);

        textPane.addHyperlinkListener(new HyperlinkListener() {
            @Override
            public void hyperlinkUpdate(HyperlinkEvent e) {
                if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
                    FaxToMailUIUtil.openLink(e.getURL());
                }
            }
        });
        return textPane;
    }

}
