/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty.hbase;

import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.*;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.util.Bytes;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyStorage;
import org.sharengo.wikitty.WikittyTransaction;
import org.sharengo.wikitty.WikittyUtil;

/**
 *
 * @author poussin
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyStorageHBase implements WikittyStorage {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyStorageHBase.class);

    protected WikittyExtensionStorage extensionStorage;

    /** storage for wikitty object */
    protected HTable hTable;

    public WikittyStorageHBase(WikittyExtensionStorage extensionStorage) {
        this.extensionStorage = extensionStorage;
        try {
            hTable = new HTable(WikittyHBaseUtil.getHBaseConfiguration(), T_WIKITTY);
        } catch (IOException eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public UpdateResponse store(WikittyTransaction transaction,
            Collection<Wikitty> wikitties,
            boolean disableAutoVersionIncrement) throws WikittyException {
        try {
            UpdateResponse result = new UpdateResponse();
            List<Put> puts = new ArrayList<Put>();
            for (Wikitty wikitty : wikitties) {
                byte[] id = Bytes.toBytes(wikitty.getId());
                // test if wikitty version is more recent that database version
                Get get = new Get(id);
                get.addColumn(F_ADMIN, Q_VERSION);
                get.addColumn(F_ADMIN, Q_DELETE_DATE);
                Result row = hTable.get(get);
                
                byte[] valueVersion = row.getValue(F_ADMIN, Q_VERSION);
                String actualVersion = valueVersion == null ? null : Bytes.toString(valueVersion);
                String requestedVersion = wikitty.getVersion();

                // compute new version, but not change wikitty during prepare
                String newVersion = null;
                if (disableAutoVersionIncrement) {
                    if (actualVersion == null) { //no version in place
                        if (requestedVersion == null) { //no version requested
                            newVersion = WikittyUtil.DEFAULT_VERSION;
                        } else { //version requested
                            newVersion = requestedVersion;
                        }
                    } else { //version in place is not null
                        if (requestedVersion == null) { //no version requested
                            newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                        } else if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) { //same version
                            // wikitty is not modified, do nothing
                            continue;
                        } else if (WikittyUtil.versionGreaterThan(requestedVersion, actualVersion)) { //requested version is newer
                            newVersion = requestedVersion;
                        } else { //requested version is obsolete
                            throw new WikittyException(String.format(
                                    "Your wikitty '%s' is obsolete (saving: '%s'; existing: '%s')", wikitty.getId(), requestedVersion, actualVersion));
                        }
                    }
                } else {
                    if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) {
                        // wikitty is not modified, do nothing
                        continue;
                    } else if (WikittyUtil.versionGreaterThan(actualVersion, requestedVersion)) {
                        throw new WikittyException(String.format(
                                "Your wikitty '%s' is obsolete", wikitty.getId()));
                    } else {
                        newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                    }
                }

                // test if wikitty is deleted
                byte[] valueDeleteDate = row.getValue(F_ADMIN, Q_DELETE_DATE);
                if (valueDeleteDate != null) {
                    throw new WikittyException(String.format(
                            "Your wikitty '%s' is deleted, you can't save it",
                            wikitty.getId()));
                }

                // update/store wikitty object
                Put put = new Put(id);
                put.add(F_ADMIN, Q_ID, id);
                put.add(F_ADMIN, Q_VERSION, Bytes.toBytes(newVersion));
                String extensionList = "";
                for (WikittyExtension ext : wikitty.getExtensions()) {
                    extensionList += "," + ext.getId();
                    for (String fieldName : ext.getFieldNames()) {
                        FieldType type = ext.getFieldType(fieldName);
                        String fqFieldName = ext.getName() + "." + fieldName;

                        if (type.isCollection()) {
                            List<Object> list = wikitty.getFieldAsList(ext.getName(), fieldName, Object.class);
                            if (list != null) {
                                //FIXME: jru 20091010 list implement serializable object
                                byte[] byteValue = WikittyHBaseUtil.toBytes(new ArrayList(list));
                                put.add(F_DATA, Bytes.toBytes(fqFieldName), byteValue);

                            } else {
                                if (type.isNotNull()) {
                                    throw new WikittyException(String.format(
                                            "Field %s in extension %s can't be null",
                                            fieldName, ext.getName()));
                                }
                                
                                // Force null value if wikitty containt field
                                if(wikitty.fieldNames().contains(fqFieldName)) {
                                    put.add(F_DATA, Bytes.toBytes(fqFieldName), NULL);
                                }
                            }
                            
                        } else {
                            Object value = wikitty.getFieldAsObject(ext.getName(), fieldName);
                            if (value != null) {
                                byte[] byteValue = WikittyHBaseUtil.toBytes(type, value);
                                put.add(F_DATA, Bytes.toBytes(fqFieldName), byteValue);
                                
                            } else {
                                if (type.isNotNull()) {
                                    throw new WikittyException(String.format(
                                            "Field %s in extension %s can't be null",
                                            fieldName, ext.getName()));
                                }
                                
                                // Force null value if wikitty containt field
                                if(wikitty.fieldNames().contains(fqFieldName)) {
                                    put.add(F_DATA, Bytes.toBytes(fqFieldName), NULL);
                                }
                            }
                        }
                    }
                }
                if (extensionList.length() > 0) {
                    // delete first ','
                    extensionList = extensionList.substring(1);
                }
                put.add(F_ADMIN, Q_EXTENSION, Bytes.toBytes(extensionList));

                wikitty.setVersion(newVersion);
                // FIXME poussin 20090831 perhaps we must not modify wikitty, and user must use UpdateResponse to update their objects
                wikitty.clearDirty();
                result.addVersionUpdate(wikitty.getId(), newVersion);
                
                puts.add(put);
            }

            hTable.put(puts);
            
            return result;
        } catch (IOException eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public UpdateResponse delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        try {
            UpdateResponse result = new UpdateResponse();
            List<Put> puts = new ArrayList<Put>();
            Date now = new Date();

            for (String id : ids) {
                if (!exists(transaction, id)) {
                    throw new WikittyException(String.format(
                            "Wikitty with id '%s' don't exists", id));
                }
                // addVersionUpdate delete date field
                Put put = new Put(Bytes.toBytes(id));
                put.add(F_ADMIN, Q_DELETE_DATE, Bytes.toBytes(now.getTime()));
                puts.add(put);

                result.addDeletionDateUpdate(id, now);
            }

            hTable.put(puts);
            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public boolean exists(WikittyTransaction transaction, String id) {
        try {
            Get get = new Get(Bytes.toBytes(id));
            boolean result = hTable.exists(get);
            return result;
        } catch (IOException eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public boolean isDeleted(WikittyTransaction transaction, String id) {
        try {
            Get get = new Get(Bytes.toBytes(id));
            get.addColumn(F_ADMIN, Q_DELETE_DATE);
            Result row = hTable.get(get);
            byte[] value = row.getValue(F_ADMIN, Q_DELETE_DATE);
            // if there is date, then object is deleted
            boolean result = value != null;
            return result;
        } catch (IOException eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public Wikitty restore(WikittyTransaction transaction, String id, String ... fqFieldName) {
        try {
            Get get = new Get(Bytes.toBytes(id));
            Result row = hTable.get(get);
            if(row.isEmpty()) {
                throw new WikittyException(String.format(
                    "The wikitty not exsist with id '%s'", id));
            }
            Wikitty result = constructWikitty(transaction, row, fqFieldName);
            return result;
        } catch (IOException eee) {
            throw new WikittyException(String.format(
                    "Can't restore wikitty '%s'", id), eee);
        }
    }

    @Override
    public void scanWikitties(WikittyTransaction transaction, Scanner scanner) {
        try {
            Scan scan = new Scan();
            ResultScanner resultScanner = hTable.getScanner(scan);

            Result row = resultScanner.next();
            while(row != null) {
                Wikitty wikitty = constructWikitty(transaction, row);
                scanner.scan(wikitty);
                row = resultScanner.next();
            }
        } catch (IOException eee) {
            throw new WikittyException(
                    String.format("Can't scan wikitties"), eee);
        }
    }

    /**
     * Create Wikitty from hbase row
     * @param row contains all data to be restored
     * @param fqFieldName minimum field to restore
     * @return
     */
    protected Wikitty constructWikitty(WikittyTransaction transaction, Result row, String ... fqFieldName) {
        Set<String> acceptedField = new HashSet<String>(Arrays.asList(fqFieldName));
        String id = Bytes.toString(row.getValue(F_ADMIN, Q_ID));
        Wikitty result = new Wikitty(id);

        byte[] version = row.getValue(F_ADMIN, Q_VERSION);
        if (version != null) {
            result.setVersion(Bytes.toString(version));
        }

        // load deleted date
        byte[] deleteDate = row.getValue(F_ADMIN, Q_DELETE_DATE);
        if (deleteDate != null) {
            Date date = new Date(Bytes.toLong(deleteDate));
            result.setDeleteDate(date);
        }

        // load extension
        byte[] extensions = row.getValue(F_ADMIN, Q_EXTENSION);
        if (extensions != null) {
            String extensionList = Bytes.toString(extensions);
            if (!"".equals(extensionList)) {
                for (String ext : extensionList.split(",")) {
                    String extName = WikittyExtension.computeName(ext);
                    String extVersion = WikittyExtension.computeVersion(ext);

                    WikittyExtension extension = extensionStorage.restore(transaction, extName, extVersion);
                    result.addExtension(extension);
                }
            }
        }

        // load field
        for (KeyValue keyValue : row.list()) {
            // only load data
            if (Bytes.equals(F_DATA, keyValue.getFamily())) {
                // fqfieldName fully qualified fieldName (extention.fieldname)
                byte[] qualifier = keyValue.getQualifier();
                String fqfieldName = Bytes.toString(qualifier);
                
                if (isAcceptedField(acceptedField, fqfieldName)) {

                    String[] field = fqfieldName.split("\\.");
                    WikittyExtension ext = result.getExtension(field[0]);
                    if(ext != null) {
                        String fieldName = field[1];
                        int crochet = fieldName.indexOf("[");
                        if (crochet != -1) {
                            fieldName = fieldName.substring(0, crochet);
                        }
                        FieldType type = ext.getFieldType(fieldName);

                        if (type != null) {
                            byte[] value = keyValue.getValue();
                            if(!Arrays.equals(value, NULL)) {
                                if (type.isCollection()) {
                                    List list = (List) WikittyHBaseUtil.fromBytes(value);
                                    result.setFqField(fqfieldName, list);
                                } else {
                                    Object object = WikittyHBaseUtil.fromBytes(type, value);
                                    result.setFqField(fqfieldName, object);
                                }
                            }
                        }
                    }
                }
            }
        }

        return result;
    }

    /**
     * Test if fqfieldName is in acceptedField
     * @param acceptedField list of all accepted field
     * @param fqfieldName fully qualified field name with potential [n/m] at end
     * @return if fqfieldName without potential [n/m] is in acceptedField or if acceptedField is empty
     */
    protected boolean isAcceptedField(Set<String> acceptedField, String fqfieldName) {
        boolean result = acceptedField.isEmpty();
        if(!result) {
            int crochet = fqfieldName.indexOf("[");
            if (crochet != -1) {
                fqfieldName = fqfieldName.substring(0, crochet);
            }
            result = acceptedField.contains(fqfieldName);
        }
        return result;

    }

    @Override
    public void clear(WikittyTransaction transaction) {
        // Nothing do
    }
}
