/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty.hbase;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigDecimal;
import java.util.Date;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.util.Bytes;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyUtil;

/**
 *
 * @author poussin
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyHBaseUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyHBaseUtil.class);

    /** table for extension */
    static final public byte[] T_EXTENSION = Bytes.toBytes("extension");
    /** table for wikitty */
    static final public byte[] T_WIKITTY = Bytes.toBytes("wikitty");
    /** family for administrative data: id, version, ... */
    static final public byte[] F_ADMIN = Bytes.toBytes("admin");
    /** family for data */
    static final public byte[] F_DATA = Bytes.toBytes("data");
    /** extension qualifier in hbase */
    static final public byte[] Q_EXTENSION  = Bytes.toBytes("extension");
    /** version qualifier in hbase */
    static final public byte[] Q_VERSION = Bytes.toBytes("version");
    /** requires qualifier in hbase */
    static final public byte[] Q_REQUIRES = Bytes.toBytes("requires");
    /** tagvalues qualifier in hbase */
    static final public byte[] Q_TAGVALUES  = Bytes.toBytes("tagvalues");
    /** wikitty id qualifier in hbase */
    static final public byte[] Q_ID = Bytes.toBytes("id");
    /** extension name qualifier in hbase */
    static final public byte[] Q_NAME = Bytes.toBytes("name");
    /** delete date qualifier in hbase */
    static final public byte[] Q_DELETE_DATE = Bytes.toBytes("deleteDate");
    /** null value in hbase */
    static final public byte[] NULL = Bytes.toBytes("null_3525743624876732845");

    /** Maximum retries.  Used as maximum for all retryable operations such as
     * fetching of the root region from root region server, getting a cell's
     * value, starting a row update, etc. Default: 10.
     */
    static final public String HBASE_CLIENT_RETRIES_NUMBER = "hbase.client.retries.number";
    /** General client pause value.  Used mostly as value to wait before running
     * a retry of a failed get, region lookup, etc. */
    static final public String HBASE_CLIENT_PAUSE = "hbase.client.pause";

    static protected HBaseConfiguration conf = new HBaseConfiguration();

    public static HBaseConfiguration getHBaseConfiguration() {
        if (conf == null) {
            conf = new HBaseConfiguration();
            conf.set(WikittyHBaseUtil.HBASE_CLIENT_PAUSE, "1000");
            conf.set(WikittyHBaseUtil.HBASE_CLIENT_RETRIES_NUMBER, "5");
        }
        return conf;
    }

    /**
     * Convert any object serializale to byte array
     */
    public static byte[] toBytes(Object obj) {
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream(bos);
            oos.writeObject(obj);
            oos.close();
            bos.close();
            byte [] data = bos.toByteArray();
            return data;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    /**
     * Convert any byte array to object
     */
    public static Object fromBytes(byte[] obj) {
        try {
            ByteArrayInputStream bis = new ByteArrayInputStream(obj);
            ObjectInputStream ois = new ObjectInputStream(bis);
            Object object = ois.readObject();
            ois.close();
            bis.close();
            return object;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    /**
     * Convert object to byte[] representation
     * - boolean representation use 0 for false and 1 for true
     * - BigDecimal use String representation
     *
     * @param field used to determinate type of o
     * @param o
     * @return
     */
    static public byte[] toBytes(FieldType field, Object o) {
        byte[] result = null;
        switch (field.getType()) {
            case BOOLEAN:
                if (Boolean.TRUE.equals(o)) {
                    result = Bytes.toBytes(1);
                } else {
                    result = Bytes.toBytes(0);
                }
                break;
            case DATE:
                // FIXME quick patch ...
                result = Bytes.toBytes( WikittyUtil.toDate(o).getTime() );
                break;
            case NUMERIC:
                result = Bytes.toBytes( WikittyUtil.toBigDecimal(o).toString() );
                break;
            case STRING:
                result = Bytes.toBytes(String.valueOf(o));
                break;
            default:
                // must be wikitty id if here
                result = Bytes.toBytes(String.valueOf(o));
                break;

        }
        return result;
    }

    static public Object fromBytes(FieldType field, byte[] b) {
        Object result = null;
        switch (field.getType()) {
            case BOOLEAN:
                // 0 = false, 1 = true
                result = Bytes.toInt(b) != 0;
                break;
            case DATE:
                long date = Bytes.toLong(b);
                result = new Date(date);
                break;
            case NUMERIC:
                String val = Bytes.toString(b);
                result = new BigDecimal(val);
                break;
            case STRING:
                result = Bytes.toString(b);
                break;
            default:
                // must be wikitty id if here
                result = Bytes.toString(b);
                break;

        }
        return result;
    }

}
