/*
 * #%L
 * Wikitty :: ui-zk
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpSession;

import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.Wikitty;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyImpl;
import org.nuiton.wikitty.WikittyProxy;
import org.zkoss.zk.ui.Component;
import org.zkoss.zk.ui.Executions;
import org.zkoss.zk.ui.util.GenericForwardComposer;
import org.zkoss.zul.Button;
import org.zkoss.zul.Label;
import org.zkoss.zul.ListModel;
import org.zkoss.zul.Listbox;
import org.zkoss.zul.Listcell;
import org.zkoss.zul.Listitem;
import org.zkoss.zul.ListitemRenderer;
import org.zkoss.zul.Textbox;
import org.zkoss.zul.Window;

/**
 *
 * @author sherkhan
 */
public class WikittyController extends GenericForwardComposer {

    protected Model _model = null;
    protected List<WikittyExtension> _extensions = new ArrayList<WikittyExtension>();
    protected WikittyExtension _selectedExtension = null;
    protected WikittyExtension _selectedNewExtension;
    protected Listbox formListBox;
    protected WikittyForm _wikittyForm = new WikittyForm();
    protected Listbox wikittiesListBox;
    protected WikittyModel _wikittyModel = new WikittyModel();

    public WikittyController() {
        HttpSession Session = (HttpSession) Executions.getCurrent().getDesktop().getSession().getNativeSession();
        WikittyProxy proxy = (WikittyProxy) Session.getAttribute("proxy");
        if (proxy != null) {
            _model = new Model(proxy.getWikittyService());
            _extensions = _model.retrieveWikittyExtensions();
        } else {
            Executions.sendRedirect("/");
        }
    }

    @Override
    public void doAfterCompose(Component comp) throws Exception {
        super.doAfterCompose(comp);

        Window window = (Window) comp;
        Button add = (Button) window.getFellow("addNewInput");
        add.setDisabled(true);
        Button delete = (Button) window.getFellow("deleteInput");
        delete.setDisabled(true);
    }

    /*
     * Events
     */
    public void onClick$home() {
        Executions.sendRedirect("/");
    }

    public void onOK$Search() {
        Textbox textbox = (Textbox) self.getFellow("Search");
        String search = textbox.getValue();
        if (search != null && !search.isEmpty()) {
            List<Wikitty> wikitties = _model.retrieveWikittiesByKeyword(search);
            _wikittyModel.setWikitties(wikitties);
            _selectedExtension = null;
        }
    }

    public void onClick$addNewInput() {
        _wikittyForm.addNewInput(formListBox);
    }

    public void onClick$deleteInput() {
        _wikittyForm.deleteInput(formListBox);
    }

    public void onClick$add() {
        Wikitty newWikitty = _wikittyForm.retriveWikittyInForm(formListBox, new WikittyImpl());
        _model.create(newWikitty);
        if (_selectedExtension != null) {
            List<Wikitty> list = _model.retrieveWikittiesByExtensionName(_selectedExtension.getName());
            _wikittyModel.setWikitties(list);
        } else {
            onOK$Search();
        }
    }

    public void onClick$update() {
        if (_wikittyForm.isSelected()) {
            Wikitty newWikitty = _wikittyForm.retriveWikittyInForm(formListBox, null);
            _model.update(newWikitty);
        }
    }

    public void onClick$delete() {
        Set<Listitem> set = wikittiesListBox.getSelectedItems();
        List<Listitem> items = new ArrayList<Listitem>(set);
        for (Listitem item : items) {
            Wikitty wikitty = (Wikitty) item.getValue();
            _model.deleteWikittyById(wikitty.getId());
            _wikittyModel.removeData(wikitty);
        }
        wikittiesListBox.setSelectedIndex(0);
    }

    public void onClick$addNewExtension() {
        _wikittyForm.addExtension(_selectedNewExtension);
    }

    public void onSelect$formListBox() {
        Listitem item = formListBox.getSelectedItem();
        if (item != null) {
            FieldExtension fieldExtension = (FieldExtension) item.getValue();
            FieldType fieldType = fieldExtension.getFieldType();
            Button add = (Button) self.getFellow("addNewInput");
            Button delete = (Button) self.getFellow("deleteInput");
            if (!fieldType.isCollection()) {
                add.setDisabled(true);
                delete.setDisabled(true);
            } else {
                add.setDisabled(false);
                delete.setDisabled(false);
            }
        }
    }

    /*
     * Renderers
     */
    public ListitemRenderer getWikittyRenderer() {
        return new ListitemRenderer() {

            @Override
            public void render(Listitem item, Object data) throws Exception {
                item.setValue(data);
                Listcell cell1 = new Listcell(((Wikitty) data).getId());
                Listcell cell2 = new Listcell();
                item.appendChild(cell1);
                item.appendChild(cell2);
                Collection<String> extensions = ((Wikitty) data).getExtensionNames();
                Iterator<String> itCurrent = extensions.iterator();
                while (itCurrent.hasNext()) {
                    String extension = itCurrent.next();
                    String toolTip = getToolTipsByExtension(extension, (Wikitty) data);
                    if (itCurrent.hasNext()) {
                        extension += ", ";
                    }
                    Label label = new Label(extension);
                    label.setTooltiptext(toolTip);
                    cell2.appendChild(label);
                }
            }
        };
    }

    /*
     * Getters
     */
    public List<WikittyExtension> getExtensions() {
        return _extensions;
    }

    public WikittyExtension getSelectedExtension() {
        return _selectedExtension;
    }

    public WikittyExtension getSelectedNewExtension() {
        return _selectedNewExtension;
    }

    public WikittyForm getWikittyForm() {
        return _wikittyForm;
    }

    public ListModel getWikittyModel() {
        return _wikittyModel;
    }

    public Wikitty getSelectedWikitty() {
        return _wikittyForm.getWikitty();
    }

    /*
     * Setters
     */
    public void setSelectedExtension(WikittyExtension current) {
        _selectedExtension = current;
        _wikittyForm.setHoldExtensions(current);
        List<Wikitty> list = _model.retrieveWikittiesByExtensionName(current.getName());
        _wikittyModel.setWikitties(list);
        Window window = (Window) this.self;
        Button add = (Button) window.getFellow("addNewInput");
        add.setDisabled(true);
        Button delete = (Button) window.getFellow("deleteInput");
        delete.setDisabled(true);
    }

    public void setSelectedNewExtension(WikittyExtension current) {
        _selectedNewExtension = current;
    }

    public void setSelectedWikitty(Wikitty current) {
        _wikittyForm.setWikitty(current);
    }

    /*
     * Methods
     */
    protected String getToolTipsByExtension(String extension, Wikitty wikitty) {
        String toolTip = "";
        WikittyExtension wikittyExtension = wikitty.getExtension(extension);
        Collection<String> fieldNames = wikittyExtension.getFieldNames();
        for (String fieldName : fieldNames) {
            String result = wikitty.getFieldAsString(extension, fieldName);
            if (result == null) {
                result = "";
            }
            if (result.length() > 50) {
                result = result.substring(0, 46);
                if (wikittyExtension.getFieldType(fieldName).isCollection()) {
                    result += "...]";
                } else {
                    result += "...";
                }
            }
            toolTip += fieldName + " : " + result + "\n";
        }
        return toolTip;
    }
}
