/*
 * #%L
 * ToPIA :: Service Replication
 * 
 * $Id: TopiaReplicationImplementor.java 1894 2010-04-15 15:44:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.4/topia-service-replication/src/main/java/org/nuiton/topia/replication/TopiaReplicationImplementor.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/* *
 * TopiaReplicationImplementor.java
 *
 * Created: 13 oct. 06 11:38:28
 *
 * @author tchemit
 * @version $Revision: 1894 $
 *
 * Last update: $Date: 2010-04-15 17:44:51 +0200 (jeu., 15 avril 2010) $
 * by : $Author: tchemit $
 */
package org.nuiton.topia.replication;

import org.nuiton.topia.replication.model.ReplicationModel;
import org.nuiton.topia.replication.model.ReplicationNode;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.replication.model.ReplicationOperationPhase;

/**
 * Class used internaly in framework, when you want implement new replication
 * engin, you must used this interface
 * 
 * @author chemit
 * @since 2.2.0
 */
public interface TopiaReplicationImplementor extends TopiaReplicationService {

    /**
     * Creation d'un modele de replication non initialise.
     *
     * @param contracts le contrats d'entites autorises
     * @param topiaIds les ids des entites a repliquer
     * @return le modele instancie mais non initialise
     *
     * @throws TopiaException pour toute erreur
     */
    ReplicationModel createModel(TopiaEntityEnum[] contracts, String... topiaIds) throws TopiaException;

    /**
     * Creation d'un modele de replication (en mode tout dupliquer) non initialise.
     *
     * @param contracts le contrats d'entites autorises
     * @return le modele instancie mais non initialise
     *
     * @throws TopiaException pour toute erreur
     */
    ReplicationModel createModelForAll(TopiaEntityEnum[] contracts) throws TopiaException;

    /**
     * Creation d'un modele de replication non initialise avec un ordre fixe
     * (celui des contrats donnés).
     *
     * @param contracts le contrats d'entites autorises
     * @param topiaIds les ids des entities a repliquer
     * @return le modele instancie mais non initialise
     *
     * @throws TopiaException pour toute erreur
     */
    ReplicationModel createModelWithComputedOrder(TopiaEntityEnum[] contracts, String... topiaIds) throws TopiaException;

    /**
     * Initialisation du modele de replication.
     *
     * Cette methode realise les calculs sur le modele pour obtenir l'ordre
     * de replication des entites {@link ReplicationModel#getOrder()}
     * et les operations a realiser lors de la replication sur chaque noeud
     * de replication {@link ReplicationNode#getOperations()}.
     *
     * @param model le modele a initialise
     * @param computeOrder drapeau pour effectuer le calcul d'ordre des
     *        entites a repliquer, sinon on utilise l'ordre de creation des
     *        noeuds dans le model.
     * @return le modele initialise
     *
     * @throws TopiaException
     */
    ReplicationModel initModel(ReplicationModel model, boolean computeOrder) throws TopiaException;

    Set<Class<? extends TopiaEntity>> detectTypes(TopiaEntityEnum[] contracts, String... ids) throws TopiaException;

    /**
     * 
     * @param operationClass le type de l'opération
     * @return l'opération du type donne
     */
    TopiaReplicationOperation getOperation(Class<? extends TopiaReplicationOperation> operationClass);

    /**
     * 
     * @param model le modele en cours
     * @param type le contrat de l'entite concerne
     * @param phase la phase de l'operation
     * @param operationClass le type de l'operation
     * @param parameters les parametres dynamiques de l'operation
     */
    void createOperation(ReplicationModel model,
            TopiaEntityEnum type,
            ReplicationOperationPhase phase,
            Class<? extends TopiaReplicationOperation> operationClass,
            Object... parameters);

    /**
     * Recupere les ids des entites a repliquer.
     *
     * @param model le modele utilise
     * @param srcCtxt le context de la base source
     * @return les ids des donnes a repliquer.
     * @throws TopiaException pour toute erreur de recuperation d'ids en base
     */
    Map<Class<? extends TopiaEntity>, List<String>> getIds(
            ReplicationModel model,
            TopiaContextImplementor srcCtxt) throws TopiaException;

    /**
     *
     * @param <E> le type de l'entite a repliquer
     * @param node le noeud de replication
     * @param srcCtxt la base source
     * @param dstCtxt la base destination
     * @param data le dictionnaire des ids a repliquer
     * @param treated l'ensemble des noeuds deja repliques
     * @throws Exception pour tout erreur
     */
    <E extends TopiaEntity> void doReplicateNode(
            ReplicationNode node,
            TopiaContext srcCtxt,
            TopiaContext dstCtxt,
            Map<Class<? extends TopiaEntity>, List<String>> data,
            Set<ReplicationNode> treated) throws Exception;
}
