/*
 * %%Ignore-License
 *
 * 08/13/2009
 *
 * TipUtil.java - Utility methods for homemade tool tips.
 * Copyright (C) 2009 Robert Futrell
 * robert_futrell at users.sourceforge.net
 * http://fifesoft.com/rsyntaxtextarea
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package org.nuiton.widget.tooltip;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.SystemColor;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JEditorPane;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.UIManager;
import javax.swing.plaf.ColorUIResource;
import javax.swing.text.html.HTMLDocument;
import org.apache.commons.lang.StringUtils;
import org.nuiton.util.Resource;

import static org.nuiton.i18n.I18n._;

/**
 * Static utility methods for focusable tips.
 *
 * @author Robert Futrell
 * @version 1.0
 */
public class TipUtil {
    
    protected static final String COPY_ICON_PATH = "copy.png";

	private TipUtil() {
	}


	/**
	 * Returns the default background color to use for tool tip windows.
	 *
	 * @return The default background color.
	 */
	public static Color getToolTipBackground() {

		Color c = UIManager.getColor("ToolTip.background");

		// Tooltip.background is wrong color on Nimbus (!)
		if (c==null || UIManager.getLookAndFeel().getName().equals("Nimbus")) {
			c = UIManager.getColor("info"); // Used by Nimbus (and others)
			if (c==null) {
				c = SystemColor.info; // System default
			}
		}

		// Workaround for a bug (?) with Nimbus - calling JLabel.setBackground()
		// with a ColorUIResource does nothing, must be a normal Color
		if (c instanceof ColorUIResource) {
			c = new Color(c.getRGB());
		}

		return c;

	}


	/**
	 * Tweaks a <code>JEditorPane</code> so it can be used to render the
	 * content in a focusable pseudo-tool tip.  It is assumed that the editor
	 * pane is using an <code>HTMLDocument</code>.
	 *
	 * @param textArea The editor pane to tweak.
	 */
	public static void tweakTipEditorPane(JEditorPane textArea) {

		// Jump through a few hoops to get things looking nice in Nimbus
		if (UIManager.getLookAndFeel().getName().equals("Nimbus")) {
			Color selBG = textArea.getSelectionColor();
			Color selFG = textArea.getSelectedTextColor();
			textArea.setUI(new javax.swing.plaf.basic.BasicEditorPaneUI());
			textArea.setSelectedTextColor(selFG);
			textArea.setSelectionColor(selBG);
		}

		textArea.setEditable(false); // Required for links to work!
		textArea.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

		// Make selection visible even though we are not (initially) focusable.
		textArea.getCaret().setSelectionVisible(true);

		// Make it use the "tool tip" background color.
		textArea.setBackground(TipUtil.getToolTipBackground());

		// Force JEditorPane to use a certain font even in HTML.
		// All standard LookAndFeels, even Nimbus (!), define Label.font.
		Font font = UIManager.getFont("Label.font");
		if (font == null) { // Try to make a sensible default
			font = new Font("SansSerif", Font.PLAIN, 12);
		}
		HTMLDocument doc = (HTMLDocument) textArea.getDocument();
		doc.getStyleSheet().addRule(
				"body { font-family: " + font.getFamily() + "; font-size: "
						+ font.getSize() + "pt; }");
	}

    /**
     * Add context menu for paste selected content
     *
     * @param focusableTip to add copy context menu
     */
    public static void addCopyContextMenu(FocusableTip focusableTip) {
        if (focusableTip == null) {
            throw new NullPointerException("Focusable tip must be non null");
        }

        TipWindow tipWindows = focusableTip.getTipWindows();
        addCopyContextMenu(tipWindows);
    }

    protected static void addCopyContextMenu(TipWindow tipWindows) {
        if (tipWindows == null) {
            throw new NullPointerException("TipWindows must be non null");
        }

        Component component = tipWindows.getComponent();
        component.addMouseListener(new MouseAdapter(){

            @Override
            public void mouseClicked(MouseEvent e) {

                JEditorPane textArea = (JEditorPane)e.getSource();
                final String selectedText = textArea.getSelectedText();
                if (e.getButton() == MouseEvent.BUTTON3 && !StringUtils.isBlank(selectedText)) {

                    JPopupMenu popupMenu = new JPopupMenu();

                    Icon copyIcon = Resource.getIcon(COPY_ICON_PATH);
                    JMenuItem copyMenuItem = new JMenuItem(_("nuitonwidgets.common.copy"), copyIcon);
                    copyMenuItem.addActionListener(new ActionListener() {

                        @Override
                        public void actionPerformed(ActionEvent e) {
                            StringSelection selection = new StringSelection(selectedText);

                            Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
                            clipboard.setContents(selection, null);
                        }
                    });

                    popupMenu.add(copyMenuItem);
                    popupMenu.show(textArea, e.getX(), e.getY());
                }
            }
        });
    }

}