/* *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */



package org.nuiton.log;

import java.awt.Component;
import java.awt.HeadlessException;
import java.util.TimerTask;

import javax.swing.SwingUtilities;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.impl.Log4JLogger;

/**
 * LutinLog.java
 *
 * Created: 9 sept. 06 03:53:09
 *
 * @author poussin
 * @version $Revision: 1637 $
 *
 * Last update: $Date: 2009-06-21 04:53:31 +0200 (dim., 21 juin 2009) $
 * by : $Author: tchemit $
 * 
 * @deprecated on 20090327 this class seam to produce a frame on each
 *  log call (produce huge memory consumption)
 */
public class LutinLog implements Log {
    
    private LutinLogFactory factory = null;
    private Log parent = null;
    private long activeProgressMonitorTime;
    private ProgressState progressState = ProgressState.end;
    private long timeout = 0;
    private int min;
    private int max;
    private int value;
    private ProgressMonitorFrame progressMonitorFrame;
    private TimerTask activeProgressMonitorTask = new TimerTask() {
        @Override
        public void run() {
            if (SwingUtilities.isEventDispatchThread()) {
                Component frame = getProgressMonitorFrame();
                if (frame != null) {
                    frame.setVisible(true);
                }
            } else {
                SwingUtilities.invokeLater(this);
            }
        }
    };
    
    private TimerTask timeoutTask = new TimerTask() {
        @Override
        public void run() {
            end();
        }
    };

    protected LutinLog(LutinLogFactory factory, Log parent) {
        this.factory = factory;
        this.parent  = parent;
        activeProgressMonitorTime = factory.getDefaultActiveProgressMonitorTime();
        timeout = factory.getDefaultTimeout();
        
        // Force progressMonitorFrame instanciation to have good message
        // displayed when frame is showing
        getProgressMonitorFrame();
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        String result = "LutinLog min: " + getMin() + " value:" + getValue() + " max:" + getMax() + " timeout:" + getTimeout();
        return result;
    }
    
    ///////////////////////////////////////////////////////////////////////////
    //
    // L I S T E N E R
    //
    ///////////////////////////////////////////////////////////////////////////
    
    public void addLutinLogListener(LutinLogListener l) {
        factory.addLutinLogListener(this, l);
    }

    public void removeLutinLogListener(LutinLogListener l) {
        factory.removeLutinLogListener(this, l);
    }
    
    ///////////////////////////////////////////////////////////////////////////
    //
    // F I R E  E V E N T
    //
    ///////////////////////////////////////////////////////////////////////////
    
    protected void fireLogEvent(LutinLogEvent.LogType type, Object msg, Throwable eee) {
        LutinLogEvent e = new LutinLogEvent(this, type, String.valueOf(msg), eee);
        try {
            factory.listeners.fire(this, "logEvent", e);
        } catch (Exception eee1) {
            if (parent.isDebugEnabled()) {
                parent.debug("Can't fire event", eee1);
            }
        }
    }
    
    protected void fireProgressEvent(LutinProgressEvent.ProgressType type) {
        LutinProgressEvent e = new LutinProgressEvent(this, type);
        try {
            factory.listeners.fire(this, "progressEvent", e);
        } catch (Exception eee1) {
            if (parent.isDebugEnabled()) {
                parent.debug("Can't fire event", eee1);
            }
        }
    }
    
    ///////////////////////////////////////////////////////////////////////////
    //
    // P R O G R E S S
    //
    ///////////////////////////////////////////////////////////////////////////
    
    /**
     * apres un start le temps avant de mettre le progress en marche (<=0 pour jamais)
     * @param time time
     */
    public void setActivateProgressMonitorTime(long time) {
        activeProgressMonitorTime = time;
    }
    
    /**
     * une tache indique son debut
     */
    public void start(){
        progressState = ProgressState.start;
        value = -1;
        
        if (activeProgressMonitorTime > 0) {
            factory.getTimer().schedule(activeProgressMonitorTask, activeProgressMonitorTime);
        }
        if (timeout > 0) {
            factory.getTimer().schedule(timeoutTask, timeout);
        }
        fireProgressEvent(LutinProgressEvent.ProgressType.start);
    }
    /**
     * @return ProgressMonitorFrame
     */
    protected Component getProgressMonitorFrame() {
        if (progressMonitorFrame == null) {
            try {
                progressMonitorFrame = new ProgressMonitorFrame(this, null);
            } catch (HeadlessException he) {
                //Unable to create frame
            }
        }
        return progressMonitorFrame;
    }

    /**
     * Si on a demande a ce que la tache soit arretee.
     * 
     * @return stopped state
     */
    public boolean isAskStopTask() {
        boolean result = progressState == ProgressState.stopAsked;
        return result;
    }
    /**
     * demande ou annule la demande de fin de tache
     * @param v
     */
    public void setAskStopTask(boolean v) {
        if (progressState != ProgressState.end) {
            if (v) {
                progressState = ProgressState.stopAsked;
                fireProgressEvent(LutinProgressEvent.ProgressType.askStop);
            } else {
                progressState = ProgressState.start;
                fireProgressEvent(LutinProgressEvent.ProgressType.cancelAskStop);
            }
        }
    }
    
    /**
     * indique la fin complete d'une tache
     */
    public void end() {
        progressState = ProgressState.end;
        activeProgressMonitorTask.cancel();
        timeoutTask.cancel();
        if (progressMonitorFrame != null) {
            progressMonitorFrame.setVisible(false);
        }
        fireProgressEvent(LutinProgressEvent.ProgressType.end);
    }
    
    /**
     * Donne le timeout particulier pour une tache.
     * 
     * @return timeout
     */
    public long getTimeout() {
        return timeout;
    }
    /**
     * modifie le temps specifiquement pour une tache. A la fin du time out
     * cela n'arrete pas la t�che mais arrete simplement de la suivre
     * @param time
     */
    public void setTimeout(long time) {
        timeout = time;
    }
    public int getMin() {
        return min;
    }
    /**
     * une tache indique le min que peut prendre value
     * @param value
     */
    public void setMin(int value) {
        min = value;
        fireProgressEvent(LutinProgressEvent.ProgressType.min);
    }
    public int getMax() {
        return max;
    }
    /**
     * une tache indique le max que peut prendre value
     * @param value
     */
    public void setMax(int value) {
        max = value;
        fireProgressEvent(LutinProgressEvent.ProgressType.max);
    }
    public int getValue() {
        return value;
    }
    /**
     * la valeur courante pour la tache, -1 indique l'indetermination de la valeur
     * @param value
     */
    public void setValue(int value) {
        this.value = value;
        fireProgressEvent(LutinProgressEvent.ProgressType.value);
    }
    
    ///////////////////////////////////////////////////////////////////////////
    //
    // L O G
    //
    ///////////////////////////////////////////////////////////////////////////
        
    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#isTraceEnabled()
     */
    public boolean isTraceEnabled() {
        boolean result = parent.isTraceEnabled();
        return result;
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#isDebugEnabled()
     */
    public boolean isDebugEnabled() {
        boolean result = parent.isDebugEnabled();
        return result;
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#isInfoEnabled()
     */
    public boolean isInfoEnabled() {
        boolean result = parent.isInfoEnabled();
        return result;
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#isWarnEnabled()
     */
    public boolean isWarnEnabled() {
        boolean result = parent.isWarnEnabled();
        return result;
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#isErrorEnabled()
     */
    public boolean isErrorEnabled() {
        boolean result = parent.isErrorEnabled();
        return result;
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#isFatalEnabled()
     */
    public boolean isFatalEnabled() {
        boolean result = parent.isFatalEnabled();
        return result;
    }

    public void user(Object msg) {
        this.user(msg, null);
    }
    
    public void user(Object msg, Throwable eee) {
        if(isInfoEnabled()) {
            if (parent instanceof Log4JLogger) {
                Log4JWrapper.info(((Log4JLogger)parent), msg, eee);
            } else {
                parent.info(msg, eee);
            }
        }        
        fireLogEvent(LutinLogEvent.LogType.user, msg, eee);
    }
    
    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#trace(java.lang.Object)
     */
    public void trace(Object msg) {
        this.trace(msg, null);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#trace(java.lang.Object, java.lang.Throwable)
     */
    public void trace(Object msg, Throwable eee) {
        if (parent instanceof Log4JLogger) {
            Log4JWrapper.trace(((Log4JLogger)parent), msg, eee);
        } else {
            parent.trace(msg, eee);
        }
        fireLogEvent(LutinLogEvent.LogType.trace, msg, eee);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#debug(java.lang.Object)
     */
    public void debug(Object msg) {
        this.debug(msg, null);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#debug(java.lang.Object, java.lang.Throwable)
     */
    public void debug(Object msg, Throwable eee) {
        if (parent instanceof Log4JLogger) {
            Log4JWrapper.debug(((Log4JLogger)parent), msg, eee);
        } else {
            parent.debug(msg, eee);
        }
        fireLogEvent(LutinLogEvent.LogType.debug, msg, eee);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#info(java.lang.Object)
     */
    public void info(Object msg) {
        this.info(msg, null);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#info(java.lang.Object, java.lang.Throwable)
     */
    public void info(Object msg, Throwable eee) {
        if (parent instanceof Log4JLogger) {
            Log4JWrapper.info(((Log4JLogger)parent), msg, eee);
        } else {
            parent.info(msg, eee);
        }
        fireLogEvent(LutinLogEvent.LogType.info, msg, eee);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#warn(java.lang.Object)
     */
    public void warn(Object msg) {
        this.warn(msg, null);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#warn(java.lang.Object, java.lang.Throwable)
     */
    public void warn(Object msg, Throwable eee) {
        if (parent instanceof Log4JLogger) {
            Log4JWrapper.warn(((Log4JLogger)parent), msg, eee);
        } else {
            parent.warn(msg, eee);
        }
        fireLogEvent(LutinLogEvent.LogType.warn, msg, eee);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#error(java.lang.Object)
     */
    public void error(Object msg) {
        this.error(msg, null);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#error(java.lang.Object, java.lang.Throwable)
     */
    public void error(Object msg, Throwable eee) {
        if (parent instanceof Log4JLogger) {
            Log4JWrapper.error(((Log4JLogger)parent), msg, eee);
        } else {
            parent.error(msg, eee);
        }
        fireLogEvent(LutinLogEvent.LogType.error, msg, eee);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#fatal(java.lang.Object)
     */
    public void fatal(Object msg) {
        this.fatal(msg, null);
    }

    /* (non-Javadoc)
     * @see org.apache.commons.logging.Log#fatal(java.lang.Object, java.lang.Throwable)
     */
    public void fatal(Object msg, Throwable eee) {
        if (parent instanceof Log4JLogger) {
            Log4JWrapper.fatal(((Log4JLogger)parent), msg, eee);
        } else {
            parent.fatal(msg, eee);
        }
        fireLogEvent(LutinLogEvent.LogType.fatal, msg, eee);
    }

}


