/* *##% Nuiton Java-2-R library
 * Copyright (C) 2006 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/
package org.nuiton.j2r.types;

import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.j2r.REngine;
import org.nuiton.j2r.RException;
import org.nuiton.j2r.RInstructions;

/**
 * Abstract class for REXP interface, in order to avoid duplicate code for attributes related methods.
 *
 * @author couteau
 */
public abstract class REXPAbstract implements REXP {

    private Log log = LogFactory.getLog(REXPAbstract.class);

    //Name of the data.frame in R
    protected String variable;

    //Attributes of the data.frame under a map -> name : R expression
    protected Map<String, Object> attributes;

    //Engine used for R instructions.
    protected REngine engine;

    //Vector containing the names of the data.frame vectors
    protected List<String> names;
    protected String indexExceptionText =
            "Cannot perform operation, index is superior to size.\nIndex : %s\nSize : %s";
    protected String dataInconsistencyText =
            "There is an inconsistency between the local and distant data.\nLocal data size : %s\nDistant data size : %s";
    protected String noVariable = "No variable name given";

    /** {@inheritDoc} */
    @Override
    public void setAttributes(Map<String, Object> attributes) throws RException {

        //TODO should be useful to test the validity of attributes before
        //assigning it.
        this.attributes = attributes;

        if ((this.attributes != null) && (!this.attributes.isEmpty())) {
            Set<String> keyset = attributes.keySet();
            String[] keys = keyset.toArray(new String[0]);
            for (int i = 0; i < keys.length; i++) {
                //foreach type of attribute, adapt the R instruction.
                if (this.attributes.get(keys[i]) instanceof String) {
                    engine.eval(String.format(RInstructions.SET_ATTRIBUTE,
                            this.variable, keys[i],
                            "\"" + this.attributes.get(keys[i]) + "\""));
                //String, value between quotes
                } else if (this.attributes.get(keys[i]) instanceof Double) {
                    engine.eval(String.format(RInstructions.SET_ATTRIBUTE,
                            this.variable,
                            keys[i], this.attributes.get(keys[i])));
                } else if (this.attributes.get(keys[i]) instanceof Integer) {
                    engine.eval(String.format(RInstructions.SET_ATTRIBUTE,
                            this.variable,
                            keys[i], String.format(RInstructions.AS_INTEGER,
                            this.attributes.get(keys[i]))));
                //Integer, value formated by the R function : as.integer()
                } else if (this.attributes.get(keys[i]) instanceof Boolean) {
                    if ((Boolean) this.attributes.get(keys[i])) {
                        engine.eval(String.format(RInstructions.SET_ATTRIBUTE,
                                this.variable,
                                keys[i], RInstructions.TRUE));
                    //Boolean true replaced by TRUE
                    } else {
                        engine.eval(String.format(RInstructions.SET_ATTRIBUTE,
                                this.variable,
                                keys[i], RInstructions.FALSE));
                    //Boolean false replaced by REPLACE
                    }
                } else if (this.attributes.get(keys[i]) instanceof REXP) {
                    engine.eval(String.format(RInstructions.SET_ATTRIBUTE,
                            this.variable,
                            keys[i],
                            ((REXP) this.attributes.get(keys[i])).toRString()));
                //REXP replaced by the result of their toRString() method.
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("This attribute is not valid : " + keys[i] +
                                " ; " + this.attributes.get(keys[i]) +
                                ". It will not be sent to R");
                    //if the attribute is not recognised.
                    }
                }
            }
        }
    }

    /** {@inheritDoc} */
    @Override
    public Map<String, Object> getAttributes() throws RException {
        if (engine.isAutoCommit()) {
            //get the number of attributes
            Integer attributeslength = (Integer) engine.eval(String.format(
                    RInstructions.LENGTH_ATTRIBUTES, this.variable));

            for (int i = 0; i < attributeslength; i++) {
                //Get the name of the i attribute
                String key = (String) engine.eval(String.format(
                        RInstructions.GET_ATTRIBUTE_NAME, this.variable, i + 1));

                //Get the attribute
                getAttribute(key);
            }
        }
        return this.attributes;
    }

    /** {@inheritDoc} */
    @Override
    public Object getAttribute(String attribute) throws RException {
        if (engine.isAutoCommit()) {
            Object returnedAttribute;
            //test if the attribute is a data.frame
            if ((Boolean) engine.eval("is.data.frame(" + String.format(
                    RInstructions.GET_ATTRIBUTE, this.variable, attribute) + ")")) {
                //if attribute is a list, import the data.frame from R
                returnedAttribute = new RDataFrame(engine);
                ((RDataFrame) returnedAttribute).getFrom(String.format(
                        RInstructions.GET_ATTRIBUTE, this.variable, attribute));
            //test if the attribute is a list
            } else if ((Boolean) engine.eval("is.list(" + String.format(
                    RInstructions.GET_ATTRIBUTE, this.variable, attribute) + ")")) {
                //if attribute is a list, import it from R.
                returnedAttribute = new RList(engine);
                ((RList) returnedAttribute).getFrom(String.format(
                        RInstructions.GET_ATTRIBUTE, this.variable, attribute));
            } else {
                //else attribute is imported as any other R expression.
                returnedAttribute = engine.eval(String.format(
                        RInstructions.GET_ATTRIBUTE, this.variable, attribute));
            }

            //put the attribute in the attribute list
            if (returnedAttribute != null) {
                if (attributes.containsKey(attribute)) {
                    //if the attribute already exists, remove it first (or you
                    //will have too times the same attribubte with different
                    //values)
                    attributes.remove(attribute);
                    attributes.put(attribute, returnedAttribute);
                } else {
                    attributes.put(attribute, returnedAttribute);
                }
            } else {
                throw new RException("Attribute does not exist");
            }
        } else if (!attributes.containsKey(attribute)) {
            //if no autocommit and attribute does not exist
            throw new RException("Attribute does not exist");
        }
        return attributes.get(attribute);
    }

    /** {@inheritDoc} */
    @Override
    public void setAttribute(String attribute, Object value) throws RException {
        //variable to check if the attribute is valid before setting it.
        boolean isOK = false;
        if (value instanceof String) {
            isOK = true;
            engine.voidEval(String.format(RInstructions.SET_ATTRIBUTE,
                    this.variable, attribute, "\"" + value + "\""));
        //if String, between quotes
        } else if (value instanceof Double) {
            isOK = true;
            engine.voidEval(String.format(RInstructions.SET_ATTRIBUTE,
                    this.variable, attribute, value));
        } else if (value instanceof Integer) {
            isOK = true;
            engine.voidEval(String.format(RInstructions.SET_ATTRIBUTE,
                    this.variable, attribute, String.format(
                    RInstructions.AS_INTEGER, value)));
        //If integer in the R function : as.integer()
        } else if (value instanceof Boolean) {
            isOK = true;
            //R boolean is upper case
            engine.voidEval(String.format(RInstructions.SET_ATTRIBUTE,
                    this.variable, attribute, value.toString().toUpperCase()));
        } else if (value instanceof REXP) {
            isOK = true;
            engine.voidEval(String.format(RInstructions.SET_ATTRIBUTE,
                    this.variable, attribute, ((REXP) value).toRString()));
        //if REXP, use its method toRString()
        } else {
            log.warn("This attribute is not valid : " + attribute + " ; " +
                    value + ". It will not be processed");
        }
        //if attribute is valid, out it in the attribute list.
        if (isOK) {
            if (attributes.containsKey(attribute)) {
                attributes.remove(attribute);
                attributes.put(attribute, value);
            } else {
                attributes.put(attribute, value);
            }
        }
    }

    /** {@inheritDoc} */
    @Override
    public String getVariable() {
        return this.variable;
    }

    /**
     * Method to set the R expression variable name
     *
     * @param variable
     *            the new variable name
     * @throws RException
     *             if something wrong happen while assigning the R expression to
     *             the new variable in R.
     */
    public void setVariable(String variable) throws RException {
        this.variable = variable;
        try {
            engine.voidEval(toRString());
        } catch (RException eee) {
            throw new RException(
                    "An error occured while trying to assign the list in R",
                    eee);
        }
    }

    /**
     * Method to get the names of the ArrayLists of the R data.frame.
     *
     * @return a ArrayList of strings containing the names of each ArrayList of
     *         the R data.frame
     * @throws RException
     *             if an error occurs while getting back the names from R.
     * @throws IndexOutOfBoundsException
     *             when the row.names size get from R is bigger than the local
     *             data size.
     */
    public List<String> getNames() throws RException {

        if (engine.isAutoCommit()) {
            //Get back the names from R.
            String[] namesArray = (String[]) engine.eval(String.format(
                    RInstructions.GET_NAMES, this.variable));

            //Check if size is correct, if yes, modify local data.
            checkX(namesArray.length);
            this.names.clear();
            for (int i = 0; i < namesArray.length; i++) {
                names.add(namesArray[i]);
            }
            return names;

        } else {
            return this.names;
        }

    }

    /**
     * Method to get the name of the ArrayList that is at this x index of the R
     * data.frame.
     *
     * @param x index of the ArrayList (0 to n-1)
     *
     * @return the name of the ArrayList
     *
     * @throws RException
     *             if an error occurs while getting back the name from R.
     *
     * @throws IndexOutOfBoundsException
     *             if the x index is out of bounds.
     */
    public String getName(int x) throws RException {

        //check if the index is valid
        if (x < names.size()) {
            if (engine.isAutoCommit()) {
                //Get back the names from R.
                String name = (String) engine.eval(String.format(
                        RInstructions.GET_NAME, this.variable, x + 1));

                //Check if the String is returned.
                if ((name != null) && (!name.equals(""))) {
                    this.names.set(x, name);
                }
            }
        } else {
            //if String is not returned, throw an IndexOutOfBOundsException.
            throw new IndexOutOfBoundsException(String.format(indexExceptionText,
                    x, this.names.size()));
        }

        return this.names.get(x);

    }

    /**
     * Test that the variable name has been set so that the REXP can be sent to
     * R.
     *
     * @throws org.nuiton.j2r.RException if the variable name have not been set.
     */
    protected void checkVariable() throws RException {
        if ((this.variable.equals("")) || (this.variable == null)) {
            throw new RException(
                    noVariable);
        }
    }

    /**
     * Method to set the engine used by the REXP. The REXP is also sent to the
     * engine.
     *
     * @param engine
     *            Engine to be set
     * @throws RException
     *             If an error occur while assigning the list to the new
     *             REngine.
     */
    public void setEngine(REngine engine) throws RException {
        this.engine = engine;
        try {
            engine.eval(toRString());
        } catch (RException eee) {
            throw new RException(
                    "Cannot assign the data.frame to the new REngine", eee);
        }
    }

    /**
     * Method to get the engine used by the REXP
     *
     * @return the engine used.
     */
    public REngine getEngine() {
        return engine;
    }

    /**
     * Method to assign names of the complex REXP objects.
     *
     * @param names
     *            a ArrayList containing the names of the REXP.
     * @throws RException
     *             when the names list is longer than the REXP length.
     */
    public void setNames(List<String> names) throws RException {

        //Check if the size is correct, if yes, do the modifications and send them to R.
        checkX(names.size());

        //Now we know we do not have a size problem.
        this.names = names;

        //Create the r instruction (names(var)<-c("name 1",...,"name x")).
        String namesString = "";
        for (int i = 0; i < this.names.size(); i++) {
            if (i != 0) {
                namesString += ",";
            }
            namesString += "\"" + names.get(i) + "\"";
        }
        String rexp = String.format(RInstructions.SET_NAMES, this.variable,
                namesString);

        //Send the r instruction to the engine.
        engine.voidEval(rexp);
    }

    /**
     * Method to assign the name of the Object at the index x.
     *
     * @param x
     *            index of the object (0 to n-1)
     * @param name
     *            Name of the object.
     * @throws RException
     *             if an error occur while in R
     */
    public void setName(int x, String name) throws RException {
        //check if the index is valid
        checkX(x);
        //Now, index is valid.
        for (int i = 0; i <= x; i++) {
            try {
                names.get(i);
                if (x == i) {
                    names.set(x, name);
                }
            } catch (IndexOutOfBoundsException eee) {
                if (x == i) {
                    names.add(name);
                } else {
                    names.add(null);
                }
            }

        }

        //create the r instruction (names(var)[x]<-"name")
        String rexp = String.format(RInstructions.SET_NAME, this.variable, x + 1,
                name);

        //Send the R instruction to the engine.
        engine.voidEval(rexp);
    }
}
