package org.nuiton.jredmine;
/*
 * #%L
 * JRedmine :: Client
 * $Id: RedmineConfigurationUtil.java 320 2012-07-19 09:09:40Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-client/src/main/java/org/nuiton/jredmine/RedmineConfigurationUtil.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.beanutils.BeanUtilsBean;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.io.rest.RestException;
import org.nuiton.jredmine.rest.RedmineRestClient;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

/**
 * Helper methods about configuration.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.5
 */
public class RedmineConfigurationUtil {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(RedmineConfigurationUtil.class);

    public static final String PROPERTY_PREFIX = "jredmine-test.";

    protected RedmineConfigurationUtil() {
        // avoid instance
    }

    public static RedmineServiceConfiguration obtainRedmineConfiguration(RedmineServiceConfiguration anoConf) throws IOException {

        RedmineServiceConfiguration conf = newLogguedConfiguration(anoConf);

        boolean ok = false;

        RedmineRestClient client = new RedmineRestClient(conf);

        try {

            client.open();
            ok = client.isOpen();
        } catch (Exception e) {

            if (log.isDebugEnabled()) {
                log.debug("Could not connect to redmine with configuration: " + conf, e);
            }

        } finally {
            try {
                client.close();
            } catch (RestException e) {
                if (log.isErrorEnabled()) {
                    log.error("Could not close session", e);
                }
            }
        }

        if (!ok) {

            // try to connect anonymous
            conf = cloneConfiguration(anoConf);

            client = new RedmineRestClient(conf);

            try {

                client.open();
                ok = client.isOpen();
            } catch (Exception e) {

                if (log.isDebugEnabled()) {
                    log.debug("Could not connect to redmine with configuration: " + conf, e);
                }
            } finally {
                try {
                    client.close();
                } catch (RestException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Could not close session", e);
                    }
                }
            }
        }

        if (!ok) {

            // no conf available
            conf = null;
        }
        return conf;
    }

    public static RedmineServiceConfiguration newAnonymousConfiguration(String propertiesFromEnv,
                                                                        String classPathPropertiesLocation)
            throws IOException {

        Properties props = new Properties();

        InputStream in = null;
        try {
            String jredmineConfiguration = System.getenv(propertiesFromEnv);
            if (jredmineConfiguration == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Could not find environement variable " +
                             "'jredmine-test.properties' will use " +
                             "default test configuration");
                }

                in = RedmineConfigurationUtil.class.getResourceAsStream(classPathPropertiesLocation);
            } else {

                File file = new File(jredmineConfiguration);

                if (!file.exists()) {
                    throw new IllegalStateException("Could not find " + jredmineConfiguration +
                                                    " file");
                }
                in = FileUtils.openInputStream(file);
            }
            props.load(in);
            in.close();
        } finally {
            IOUtils.closeQuietly(in);
        }

        RedmineServiceConfiguration result = new SimpleRedmineServiceConfiguration();

        overridePropertyFromProperties(result, "url", "restUrl", props);
        overridePropertyFromProperties(result, "encoding", props);
        overridePropertyFromProperties(result, "verbose", props);

        overridePropertyFromEnv(result, "url", "restUrl");
        overridePropertyFromEnv(result, "encoding");
        overridePropertyFromEnv(result, "verbose");
        result.setAnonymous(true);
        return result;
    }

    public static RedmineServiceConfiguration newLogguedConfiguration(RedmineServiceConfiguration anoConf)
            throws IOException {

        // use anonymous configuration

        RedmineServiceConfiguration result = cloneConfiguration(anoConf);

        overridePropertyFromEnv(result, "login", "restUsername");
        overridePropertyFromEnv(result, "password", "restPassword");
        overridePropertyFromEnv(result, "apiKey", "apiKey");

        result.setAnonymous(false);

        return result;
    }

    public static RedmineServiceConfiguration cloneConfiguration(
            RedmineServiceConfiguration src) {
        RedmineServiceConfiguration dst = new SimpleRedmineServiceConfiguration();
        dst.setRestUrl(src.getRestUrl());
        dst.setRestUsername(src.getRestUsername());
        dst.setRestPassword(src.getRestPassword());
        dst.setApiKey(src.getApiKey());
        dst.setEncoding(src.getEncoding());
        dst.setVerbose(src.isVerbose());
        dst.setAnonymous(src.isAnonymous());
        return dst;
    }

    public static void copyConfiguration(RedmineServiceConfiguration src,
                                         RedmineServiceConfiguration dst) {
        dst.setRestUrl(src.getRestUrl());
        dst.setRestUsername(src.getRestUsername());
        dst.setRestPassword(src.getRestPassword());
        dst.setApiKey(src.getApiKey());
        dst.setEncoding(src.getEncoding());
        dst.setVerbose(src.isVerbose());
        dst.setAnonymous(src.isAnonymous());
    }

    protected static void overridePropertyFromProperties(RedmineServiceConfiguration conf,
                                                         String prop,
                                                         Properties props) throws IOException {
        overridePropertyFromProperties(conf, prop, prop, props);
    }

    protected static void overridePropertyFromProperties(RedmineServiceConfiguration conf,
                                                         String prop,
                                                         String beanProp,
                                                         Properties props) throws IOException {
        String value = props.getProperty(PROPERTY_PREFIX + prop);
        if (StringUtils.isNotEmpty(value)) {
            try {
                BeanUtilsBean.getInstance().setProperty(conf, beanProp, value);
            } catch (Exception e) {
                throw new IOException(
                        "Could not set property '" + beanProp +
                        "' with value '" + value + "' to configuration");
            }
        }
    }

    protected static void overridePropertyFromEnv(RedmineServiceConfiguration conf,
                                                  String prop) throws IOException {
        overridePropertyFromEnv(conf, prop, prop);
    }

    protected static void overridePropertyFromEnv(RedmineServiceConfiguration conf,
                                                  String prop,
                                                  String beanProp) throws IOException {
        String value = System.getenv(PROPERTY_PREFIX + prop);
        if (StringUtils.isNotEmpty(value) && !"null".equals(value)) {
            try {
                BeanUtilsBean.getInstance().setProperty(conf, beanProp, value);
            } catch (Exception e) {
                throw new IOException(
                        "Could not set property '" + prop +
                        "' with value '" + value + "' to configuration");
            }
        }
    }

    /**
     * Configuration of a redmine service for test purposes.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 1.4
     */
    public static class SimpleRedmineServiceConfiguration implements RedmineServiceConfiguration {

        URL restUrl;

        String restUsername;

        String restPassword;

        boolean verbose;

        boolean anonymous;

        String encoding;

        String apiKey;

        @Override
        public String getEncoding() {
            return encoding;
        }

        @Override
        public void setEncoding(String encoding) {
            this.encoding = encoding;
        }

        @Override
        public String getRestPassword() {
            return restPassword;
        }

        @Override
        public void setRestPassword(String restPassword) {
            this.restPassword = restPassword;
        }

        @Override
        public URL getRestUrl() {
            return restUrl;
        }

        @Override
        public void setRestUrl(URL restUrl) {
            this.restUrl = restUrl;
        }

        @Override
        public String getRestUsername() {
            return restUsername;
        }

        @Override
        public void setRestUsername(String restUsername) {
            this.restUsername = restUsername;
        }

        @Override
        public boolean isVerbose() {
            return verbose;
        }

        @Override
        public void setVerbose(boolean verbose) {
            this.verbose = verbose;
        }

        @Override
        public boolean isAnonymous() {
            return anonymous;
        }

        @Override
        public void setAnonymous(boolean anonymous) {
            this.anonymous = anonymous;
        }

        @Override
        public String toString() {
            ToStringBuilder b = new ToStringBuilder(
                    this,
                    ToStringStyle.MULTI_LINE_STYLE
            );
            b.append("redmineUrl", restUrl);
            if (anonymous) {
                b.append("anonymous", true);
            } else {
                b.append("redmineUsername", restUsername);
                b.append("redminePassword", "***");
            }
            b.append("encoding", encoding);
            b.append("verbose", verbose);
            return b.toString();
        }

        @Override
        public String getApiKey() {
            return apiKey;
        }

        @Override
        public void setApiKey(String apiKey) {
            this.apiKey = apiKey;
        }
    }
}
