/*
 * #%L
 * JRedmine :: Client
 * 
 * $Id: DefaultRedmineAnonymousService.java 344 2012-10-04 11:57:42Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-client/src/main/java/org/nuiton/jredmine/DefaultRedmineAnonymousService.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.TimeEntry;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;

/**
 * Default implementation of {@link RedmineAnonymousService}.
 * <p/>
 * Created: 2 janv. 2010
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.jredmine.RedmineAnonymousService" role-hint="default"
 * @see RestClientConfiguration#isAnonymous()
 * @since 1.0.3
 */
public class DefaultRedmineAnonymousService implements RedmineAnonymousService {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(DefaultRedmineAnonymousService.class);

    /**
     * Delegate all the technical layer to an implementor.
     *
     * @see RedmineServiceImplementor
     */
    protected final RedmineServiceImplementor delegateImplementor;

    /**
     * Default constructor.
     * <p/>
     * Make this service use the default {@link RedmineServiceImplementor}.
     *
     * @see DefaultRedmineServiceImplementor
     * @since 1.4
     */
    public DefaultRedmineAnonymousService() {
        this(new DefaultRedmineServiceImplementor());
    }

    /**
     * Service constructor using the givne implementator to deal this technical layer.
     *
     * @param delegateImplementor the implementator to use
     * @since 1.4
     */
    public DefaultRedmineAnonymousService(RedmineServiceImplementor delegateImplementor) {
        this.delegateImplementor = delegateImplementor;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineAnonymousService implementation
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public void init(RedmineServiceConfiguration configuration) throws RedmineServiceException {
        // Force to not be loggued
        configuration.setAnonymous(true);
        if (log.isDebugEnabled()) {
            log.debug("init service with configuration: " + configuration);
        }
        if (!delegateImplementor.isInit()) {

            // only init once
            delegateImplementor.init(configuration);
        }
    }

    @Override
    public void destroy() throws RedmineServiceException {
        delegateImplementor.destroy();
    }

    @Override
    public RedmineServiceImplementor getDelegateImplementor() {
        return delegateImplementor;
    }

    @Override
    public IssueStatus[] getIssueStatuses() throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_ISSUE_STATUS_REQUEST_NAME,
                                            IssueStatus.class
        );
    }

    @Override
    public IssuePriority[] getIssuePriorities() throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_ISSUE_PRIORITY_REQUEST_NAME,
                                            IssuePriority.class
        );
    }

    @Override
    public Project[] getProjects() throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_PROJECT_REQUEST_NAME, Project.class);
    }

    @Override
    public IssueCategory[] getIssueCategories(String projectName) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_ISSUE_CATEGORY_REQUEST_NAME,
                                            IssueCategory.class,
                                            projectName
        );
    }

    @Override
    public Project getProject(String projectName) throws RedmineServiceException {
        return delegateImplementor.getData(ModelHelper.GET_PROJECT_REQUEST_NAME,
                                           Project.class,
                                           projectName
        );
    }

    @Override
    public Tracker[] getTrackers(String projectName) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_TRACKER_REQUEST_NAME,
                                            Tracker.class,
                                            projectName
        );
    }

    @Override
    public News[] getNews(String projectName) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_NEWS_REQUEST_NAME,
                                            News.class,
                                            projectName
        );
    }

    @Override
    public User[] getProjectMembers(String projectName) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_USER_REQUEST_NAME,
                                            User.class,
                                            projectName
        );
    }

    @Override
    public Version[] getVersions(String projectName) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_VERSION_REQUEST_NAME,
                                            Version.class,
                                            projectName
        );
    }

    @Override
    public Version getVersion(String projectName,
                              String versionName) throws RedmineServiceException {
        return delegateImplementor.getData(ModelHelper.GET_VERSION_REQUEST_NAME,
                                           Version.class,
                                           projectName,
                                           versionName
        );
    }

    @Override
    public Attachment[] getAttachments(String projectName,
                                       String versionName) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_ATTACHMENTS_REQUEST_NAME,
                                            Attachment.class,
                                            projectName,
                                            versionName
        );
    }

    @Override
    public Issue[] getIssues(String projectName,
                             String versionName) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_ISSUES_REQUEST_NAME,
                                            Issue.class,
                                            projectName,
                                            versionName
        );
    }

    @Override
    public TimeEntry[] getIssueTimeEntries(String projectName,
                                           String issueId) throws RedmineServiceException {
        return delegateImplementor.getDatas(ModelHelper.GET_ALL_ISSUE_TIME_ENTRY_REQUEST_NAME,
                                            TimeEntry.class,
                                            projectName,
                                            issueId
        );
    }

    @Override
    public Issue[] getIssues(String projectName) throws RedmineServiceException {
        Issue[] result = delegateImplementor.getDatas(ModelHelper.GET_ALL_PROJECT_ISSUES_REQUEST_NAME,
                                                      Issue.class,
                                                      projectName
        );
        return result;
    }

    @Override
    public Issue[] getOpenedIssues(String projectName) throws RedmineServiceException {
        Issue[] result = delegateImplementor.getDatas(ModelHelper.GET_ALL_PROJECT_OPENED_ISSUES_REQUEST_NAME,
                                                      Issue.class,
                                                      projectName
        );
        return result;
    }

    @Override
    public Issue[] getClosedIssues(String projectName) throws RedmineServiceException {
        Issue[] result = delegateImplementor.getDatas(ModelHelper.GET_ALL_PROJECT_CLOSED_ISSUES_REQUEST_NAME,
                                                      Issue.class,
                                                      projectName
        );
        return result;
    }

}
