package jaxx.runtime.swing.editor;

/*-
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2021 Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.DefaultBoundedRangeModel;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import jaxx.runtime.JAXXBinding;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXObjectDescriptor;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.binding.SimpleJAXXObjectBinding;
import jaxx.runtime.context.DefaultJAXXContext;
import jaxx.runtime.swing.Table;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n.t;

public class TimeEditor extends JPanel implements JAXXObject {

    /*-----------------------------------------------------------------------*/
    /*---------------- Constants for all javaBean properties ----------------*/
    /*-----------------------------------------------------------------------*/

    public static final String PROPERTY_BEAN = "bean";
    public static final String PROPERTY_DATE = "date";
    public static final String PROPERTY_LABEL = "label";
    public static final String PROPERTY_PROPERTY = "property";
    public static final String PROPERTY_TIME_MODEL = "timeModel";

    /*-----------------------------------------------------------------------*/
    /*------------------ Constants for all public bindings ------------------*/
    /*-----------------------------------------------------------------------*/

    public static final String BINDING_HOUR_ENABLED = "hour.enabled";
    public static final String BINDING_HOUR_VALUE = "hour.value";
    public static final String BINDING_MINUTE_ENABLED = "minute.enabled";
    public static final String BINDING_MINUTE_MODEL_VALUE = "minuteModel.value";
    public static final String BINDING_SLIDER_ENABLED = "slider.enabled";
    public static final String BINDING_SLIDER_VALUE = "slider.value";
    public static final String BINDING_TITLE_TEXT = "title.text";

    /*-----------------------------------------------------------------------*/
    /*------------------------- Other static fields -------------------------*/
    /*-----------------------------------------------------------------------*/

    private static final String $jaxxObjectDescriptor = "H4sIAAAAAAAAAJVVzW8bRRQfu/6IkyZNG1oVNZUWSCW7lLXTonAIgqb5UBI5AdVGqvCFsXdqTxjPLjOzzaZWEX8CfwLcuSBx6wlx6JkDl6r/QlVx4Ip4M7v22MmiuFGy2Xmfv/eb997+8hrlpUDvHeEockXIFe0Td3/j0aMv2keko7aI7AgaKF+g+CeTRdkWmvNGcqnQB626dq8m7tVNvx/4nPAx7/U6mpXqhBHZI0QpdHPSoyNltTFSr0dBKIZRR6DSov7095vsj94PP2cRigJAdwlKcc7zspXk6ihLPYWuQKYnuMow7wIMQXkX8M5r2SbDUh7iPvkOfY+KdVQIsIBgCr0/fckmhvGPAoVmVva/xJywmkIrE0jlMeR1iUfB122CZNu8BoHxLYBnIPyACHWi0KLG5mq8bozXWuUZbhOmDwtWmGsTzCfc4uu1FiWN4cD3CFPosjXb44p0yRiInIcVUWjBmISKMncLBFY/16c8VMNIy9osSkprBJRzIrS9UVunYg9zj0Fp6MNpKNmNrcf8V5q4zQhQej3F3+i0ZcmSpKhiUMaVcXz79SFzjuV8JZbW9PnWpC7X80OA/M5EjKTIScuCuZPdlBiFmC59un1KIxn1NCVLk/GNNKlGoBsT9cIYuXaMbJ9nWigvQhADQa2zk/cQVPHMXT81czqg0f57bemv569+2xkOWhFyX001HdsTMABJy1Kd+lI8Zbppqgc4WG+hkiQMutAskeUUYI1EDeAg32Xbc7tY9iBEvvjy9z+uffPnBZTdQbPMx94O1vZ70M49ASz4zIuCz+8bRBePZ+C5CH9ZILXnC/rU5wqzDUa7vG+GutCBf5rznCIRnC8OukSZ+y9XnkXAz3IKPyOQ7dKLf5Yav94fcpQBzO/+r7nlKf81KlDOKCdmHSWbJnX9zAWShJ5vN0rajkHD/g6SCfnEPNfP8HABDPvxoN4acHLsJM17GPbbRJgZLdfuOPB7994dZ7XyDOyfYBZCa13VzDSHK6NccarOWg30RcL1tMFOnR9QuR0fDHkawoZhRb9tomR+zsM4/V1phpNVsZuWLhnCKRLOdzAj3MNihxIGpdwY2M7bTFTuwd7hV83tMUrKmpJkNZUrriTqwK7CMuj04itX0rkYX5tTIBxe2/xgzO8t6U9Wz3nZZgBcgCmP50BqvoUud9YIm7TzrZWVHsMtfSTpU3jPrq7CB6ePj3yhjRoB7sACA/lazeKvmbbbIo9xyNQDP+Qe8R7Cd4eMmm/V9N9azbnt3P1Yd+AiwD4d8l7NXsPiqc58W1bitXseKxnTTyd+CJ238Cl8E7HTptwDOJ+diZrRz1mI+B9sfVKxbAkAAA==";
    private static final Log log = LogFactory.getLog(TimeEditor.class);
    private static final long serialVersionUID = 1L;

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Internal states ---------------------------*/
    /*-----------------------------------------------------------------------*/

    protected List<Object> $activeBindings = new ArrayList<Object>();
    protected Map<String, Object> $bindingSources = new HashMap<String, Object>();
    protected final Map<String, JAXXBinding> $bindings = new TreeMap<String, JAXXBinding>();
    protected Map<String, Object> $objectMap = new HashMap<String, Object>();
    protected Map<?, ?> $previousValues = new HashMap<Object, Object>();
    private boolean allComponentsCreated;
    protected final JAXXContext delegateContext = new DefaultJAXXContext();

    /*-----------------------------------------------------------------------*/
    /*------------------------ Protected components  ------------------------*/
    /*-----------------------------------------------------------------------*/

    protected Object bean;
    protected Date date;
    protected TimeEditorHandler handler;
    protected JSpinner hour;
    protected String label;
    protected JLabel labelH;
    protected JSpinner minute;
    protected SpinnerDateModel minuteModel;
    protected String property;
    protected JSlider slider;
    protected Integer timeModel;
    protected JLabel title;

    /*-----------------------------------------------------------------------*/
    /*------------------------- Private components  -------------------------*/
    /*-----------------------------------------------------------------------*/

    private JLabel $JLabel0;
    private TimeEditor $JPanel0 = this;
    private Table $Table0;

    /*-----------------------------------------------------------------------*/
    /*---------------------- Raw body code from script ----------------------*/
    /*-----------------------------------------------------------------------*/

    public void init() {
        minute.setEditor(new JSpinner.DateEditor(minute, "mm"));
        handler.init();
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Constructors  ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public TimeEditor(LayoutManager param0, boolean param1) {
        super(param0 ,param1);
        $initialize();
    }

    public TimeEditor(JAXXContext parentContext, LayoutManager param1, boolean param2) {
        super(param1 ,param2);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public TimeEditor(LayoutManager param0) {
        super(param0);
        $initialize();
    }

    public TimeEditor(JAXXContext parentContext, LayoutManager param1) {
        super(param1);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public TimeEditor(boolean param0) {
        super(param0);
        $initialize();
    }

    public TimeEditor(JAXXContext parentContext, boolean param1) {
        super(param1);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public TimeEditor() {
        $initialize();
    }

    public TimeEditor(JAXXContext parentContext) {
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Statics methods ---------------------------*/
    /*-----------------------------------------------------------------------*/

    public static JAXXObjectDescriptor $getJAXXObjectDescriptor() {
        return JAXXUtil.decodeCompressedJAXXObjectDescriptor($jaxxObjectDescriptor);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------- JAXXObject implementation ----------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public void applyDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).applyDataBinding();
        }
        processDataBinding($binding);
    }

    @Override
    public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        super.firePropertyChange(propertyName, oldValue, newValue);
    }

    @Override
    public Map<String, Object> get$objectMap() {
        return $objectMap;
    }

    @Override
    public JAXXBinding getDataBinding(String bindingId) {
        return $bindings.get(bindingId);
    }

    @Override
    public JAXXBinding[] getDataBindings() {
        return $bindings.values().toArray(new JAXXBinding[$bindings.size()]);
    }

    @Override
    public Object getObjectById(String id) {
        return $objectMap.get(id);
    }

    @Override
    public void processDataBinding(String $binding, boolean $force) {
        if (!$force && $activeBindings.contains($binding)) { 
            return;
        }
        $activeBindings.add($binding);
        try {
            if (allComponentsCreated && $bindings.containsKey($binding)) {
                getDataBinding($binding).processDataBinding();
            }
        } finally {
            $activeBindings.remove($binding);
        }
    }

    @Override
    public void processDataBinding(String $binding) {
        processDataBinding($binding, false);
    }

    @Override
    public void registerDataBinding(JAXXBinding binding) {
        $bindings.put(binding.getId(), binding);
    }

    @Override
    public void removeDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).removeDataBinding();
        }
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- JAXXContext implementation  ---------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public <T> T getContextValue(Class<T> clazz) {
        return delegateContext.getContextValue(clazz, null);
    }

    @Override
    public <T> T getContextValue(Class<T> clazz, String name) {
        return delegateContext.getContextValue(clazz, name);
    }

    @Override
    public JAXXContext getDelegateContext() {
        return delegateContext;
    }

    @Override
    public <O extends Container> O getParentContainer(Class<O> clazz) {
        return SwingUtil.getParentContainer(this, clazz);
    }

    @Override
    public <O extends Container> O getParentContainer(Object source, Class<O> clazz) {
        return SwingUtil.getParentContainer(source, clazz);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz) {
        delegateContext.removeContextValue(clazz, null);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz, String name) {
        delegateContext.removeContextValue(clazz, name);
    }

    @Override
    public <T> void setContextValue(T o) {
        delegateContext.setContextValue(o, null);
    }

    @Override
    public <T> void setContextValue(T o, String name) {
        delegateContext.setContextValue(o, name);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Event methods ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public void doStateChanged__on__hour(ChangeEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        setTimeModel((Integer)hour.getValue() * 60 + getHandler().getMinute());
    }

    public void doStateChanged__on__minute(ChangeEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        getHandler().updateTimeModelFromMinuteModel(minuteModel.getDate());
    }

    public void doStateChanged__on__slider(ChangeEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        if (!slider.getValueIsAdjusting()) setTimeModel(slider.getValue());
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public acessor methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public Object getBean() {
        return bean;
    }

    public Date getDate() {
        return date;
    }

    public TimeEditorHandler getHandler() {
        return handler;
    }

    public JSpinner getHour() {
        return hour;
    }

    public String getLabel() {
        return label;
    }

    public JLabel getLabelH() {
        return labelH;
    }

    public JSpinner getMinute() {
        return minute;
    }

    public SpinnerDateModel getMinuteModel() {
        return minuteModel;
    }

    public String getProperty() {
        return property;
    }

    public JSlider getSlider() {
        return slider;
    }

    public Integer getTimeModel() {
        return timeModel;
    }

    public JLabel getTitle() {
        return title;
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public mutator methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public void setBean(Object bean) {
        Object oldValue = this.bean;
        this.bean = bean;
        firePropertyChange(PROPERTY_BEAN, oldValue, bean);
    }

    public void setDate(Date date) {
        Date oldValue = this.date;
        this.date = date;
        firePropertyChange(PROPERTY_DATE, oldValue, date);
    }

    public void setLabel(String label) {
        String oldValue = this.label;
        this.label = label;
        firePropertyChange(PROPERTY_LABEL, oldValue, label);
    }

    public void setProperty(String property) {
        String oldValue = this.property;
        this.property = property;
        firePropertyChange(PROPERTY_PROPERTY, oldValue, property);
    }

    public void setTimeModel(Integer timeModel) {
        Integer oldValue = this.timeModel;
        this.timeModel = timeModel;
        firePropertyChange(PROPERTY_TIME_MODEL, oldValue, timeModel);
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Protected acessors methods  ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected JLabel get$JLabel0() {
        return $JLabel0;
    }

    protected Table get$Table0() {
        return $Table0;
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Components creation methods ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected void createBean() {
        $objectMap.put("bean", bean = null);
    }

    protected void createDate() {
        $objectMap.put("date", date = null);
    }

    protected void createHandler() {
        $objectMap.put("handler", handler = new TimeEditorHandler(this));
    }

    protected void createHour() {
        $objectMap.put("hour", hour = new JSpinner());
        
        hour.setName("hour");
        hour.addChangeListener(JAXXUtil.getEventListener(ChangeListener.class, "stateChanged", this, "doStateChanged__on__hour"));
    }

    protected void createLabel() {
        $objectMap.put("label", label = "");
    }

    protected void createLabelH() {
        $objectMap.put("labelH", labelH = new JLabel());
        
        labelH.setName("labelH");
        labelH.setHorizontalAlignment(0);
        labelH.setText(t("timeeditor.H"));
    }

    protected void createMinute() {
        $objectMap.put("minute", minute = new JSpinner());
        
        minute.setName("minute");
        minute.addChangeListener(JAXXUtil.getEventListener(ChangeListener.class, "stateChanged", this, "doStateChanged__on__minute"));
    }

    protected void createMinuteModel() {
        $objectMap.put("minuteModel", minuteModel = new SpinnerDateModel());
    }

    protected void createProperty() {
        $objectMap.put("property", property = "");
    }

    protected void createSlider() {
        $objectMap.put("slider", slider = new JSlider());
        
        slider.setName("slider");
        slider.setValue(0);
        slider.setPaintLabels(true);
        slider.setPaintTicks(true);
        if (slider.getFont() != null) {
            slider.setFont(slider.getFont().deriveFont((float) 11));
        }
        slider.setMajorTickSpacing(60);
        slider.setMinorTickSpacing(30);
        slider.addChangeListener(JAXXUtil.getEventListener(ChangeListener.class, "stateChanged", this, "doStateChanged__on__slider"));
    }

    protected void createTimeModel() {
        $objectMap.put("timeModel", timeModel = 0);
    }

    protected void createTitle() {
        $objectMap.put("title", title = new JLabel());
        
        title.setName("title");
        title.setHorizontalAlignment(0);
    }

    /*-----------------------------------------------------------------------*/
    /*------------------------ Internal jaxx methods ------------------------*/
    /*-----------------------------------------------------------------------*/

    private void $completeSetup() {
        allComponentsCreated = true;
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        // inline complete setup of $JPanel0
        add($Table0, BorderLayout.NORTH);
        add(slider, BorderLayout.SOUTH);
        // inline complete setup of $Table0
        $Table0.add(title, new GridBagConstraints(0, 0, 1, 1, 0.0, 0.0, 10, 2, new Insets(0, 0, 0, 0), 0, 0));
        $Table0.add($JLabel0, new GridBagConstraints(1, 0, 1, 1, 1.0, 0.0, 10, 2, new Insets(0, 0, 0, 0), 0, 0));
        $Table0.add(hour, new GridBagConstraints(2, 0, 1, 1, 0.0, 0.0, 10, 2, new Insets(0, 0, 0, 0), 0, 0));
        $Table0.add(labelH, new GridBagConstraints(3, 0, 1, 1, 0.0, 0.0, 10, 2, new Insets(0, 0, 0, 0), 0, 0));
        $Table0.add(minute, new GridBagConstraints(4, 0, 1, 1, 0.0, 0.0, 10, 2, new Insets(0, 0, 0, 0), 0, 0));
        
        // apply 7 data bindings
        JAXXUtil.applyDataBinding(this, $bindings.keySet());
        
        // apply 5 property setters
        minuteModel.setCalendarField(java.util.Calendar.MINUTE);
        hour.setModel(new SpinnerNumberModel(0, 0, 23, 1));
        minute.setModel(minuteModel);
        slider.setModel(new DefaultBoundedRangeModel(0, 1, 0, 60 * 24));
    }

    private void $initialize() {
        if (allComponentsCreated) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        $objectMap.put("$JPanel0", $JPanel0);
        createProperty();
        createLabel();
        createBean();
        createTimeModel();
        createDate();
        createMinuteModel();
        createHandler();
        // inline creation of $Table0
        $objectMap.put("$Table0", $Table0 = new Table());
        
        $Table0.setName("$Table0");
        createTitle();
        // inline creation of $JLabel0
        $objectMap.put("$JLabel0", $JLabel0 = new JLabel());
        
        $JLabel0.setName("$JLabel0");
        createHour();
        createLabelH();
        createMinute();
        createSlider();
        // inline creation of $JPanel0
        setName("$JPanel0");
        setLayout(new BorderLayout());
        
        // registers 7 data bindings
        $registerDefaultBindings();
        $completeSetup();
    }

    private void $registerDefaultBindings() {
        // register 7 data bindings
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MINUTE_MODEL_VALUE, true ,"date") {
        
            @Override
            public void processDataBinding() {
                minuteModel.setValue(getHandler().setMinuteModel(getDate()));
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_TITLE_TEXT, true ,"label") {
        
            @Override
            public void processDataBinding() {
                title.setText(t(getLabel()));
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_HOUR_VALUE, true ,"timeModel") {
        
            @Override
            public void processDataBinding() {
                hour.setValue(getTimeModel() / 60);
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_HOUR_ENABLED, true ,"enabled") {
        
            @Override
            public void processDataBinding() {
                hour.setEnabled(isEnabled());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_MINUTE_ENABLED, true ,"enabled") {
        
            @Override
            public void processDataBinding() {
                minute.setEnabled(isEnabled());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_SLIDER_VALUE, true ,"timeModel") {
        
            @Override
            public void processDataBinding() {
                slider.setValue(getTimeModel());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_SLIDER_ENABLED, true ,"enabled") {
        
            @Override
            public void processDataBinding() {
                slider.setEnabled(isEnabled());
            }
        });
    }

}
