package jaxx.runtime.swing.editor.bean;

/*-
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2017 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import jaxx.runtime.JAXXBinding;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXObjectDescriptor;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.bean.BeanTypeAware;
import jaxx.runtime.binding.DefaultJAXXBinding;
import jaxx.runtime.binding.SimpleJAXXObjectBinding;
import jaxx.runtime.context.DefaultJAXXContext;
import jaxx.runtime.swing.JAXXButtonGroup;
import jaxx.runtime.swing.Table;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.decorator.JXPathDecorator;
import static org.nuiton.i18n.I18n.t;

public class BeanDoubleList<O> extends Table implements BeanTypeAware<O>, JAXXObject {

    /*-----------------------------------------------------------------------*/
    /*---------------- Constants for all javaBean properties ----------------*/
    /*-----------------------------------------------------------------------*/

    public static final String PROPERTY_BEAN = "bean";
    public static final String PROPERTY_BEAN_TYPE = "beanType";
    public static final String PROPERTY_FILTERABLE = "filterable";
    public static final String PROPERTY_HIGHLIGHT_FILTER_TEXT = "highlightFilterText";
    public static final String PROPERTY_I18N_PREFIX = "i18nPrefix";
    public static final String PROPERTY_INDEX = "index";
    public static final String PROPERTY_MODEL = "model";
    public static final String PROPERTY_PROPERTY = "property";
    public static final String PROPERTY_REVERSE_SORT = "reverseSort";
    public static final String PROPERTY_SHOW_DECORATOR = "showDecorator";
    public static final String PROPERTY_SHOW_RESET = "showReset";
    public static final String PROPERTY_SHOW_SELECT_POPUP_ENABLED = "showSelectPopupEnabled";
    public static final String PROPERTY_USE_MULTI_SELECT = "useMultiSelect";

    /*-----------------------------------------------------------------------*/
    /*------------------ Constants for all public bindings ------------------*/
    /*-----------------------------------------------------------------------*/

    public static final String BINDING_ADD_BUTTON_ENABLED = "addButton.enabled";
    public static final String BINDING_CHANGE_DECORATOR_ENABLED = "changeDecorator.enabled";
    public static final String BINDING_FILTER_PANEL_VISIBLE = "filterPanel.visible";
    public static final String BINDING_REMOVE_BUTTON_ENABLED = "removeButton.enabled";
    public static final String BINDING_RESET_BUTTON_ENABLED = "resetButton.enabled";
    public static final String BINDING_SELECTED_LIST_COMPONENT_POPUP_MENU = "selectedList.componentPopupMenu";
    public static final String BINDING_SELECTED_LIST_MODEL = "selectedList.model";
    public static final String BINDING_SELECTED_LIST_MOVE_DOWN_ACTION_ENABLED = "selectedListMoveDownAction.enabled";
    public static final String BINDING_SELECTED_LIST_MOVE_UP_ACTION_ENABLED = "selectedListMoveUpAction.enabled";
    public static final String BINDING_SELECTED_LIST_REMOVE_ACTION_ENABLED = "selectedListRemoveAction.enabled";
    public static final String BINDING_SORT_DOWN_SELECTED = "sortDown.selected";
    public static final String BINDING_SORT_UP_SELECTED = "sortUp.selected";
    public static final String BINDING_TOOLBAR_LEFT_VISIBLE = "toolbarLeft.visible";
    public static final String BINDING_TOOLBAR_RIGHT_VISIBLE = "toolbarRight.visible";
    public static final String BINDING_UNIVERSE_LIST_MODEL = "universeList.model";

    /*-----------------------------------------------------------------------*/
    /*------------------------- Other static fields -------------------------*/
    /*-----------------------------------------------------------------------*/

    private static final String $jaxxObjectDescriptor = "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";
    private static final Log log = LogFactory.getLog(BeanDoubleList.class);
    private static final long serialVersionUID = 1L;

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Internal states ---------------------------*/
    /*-----------------------------------------------------------------------*/

    protected List<Object> $activeBindings = new ArrayList<Object>();
    protected Map<String, Object> $bindingSources = new HashMap<String, Object>();
    protected final Map<String, JAXXBinding> $bindings = new TreeMap<String, JAXXBinding>();
    protected Map<String, Object> $objectMap = new HashMap<String, Object>();
    protected Map<?, ?> $previousValues = new HashMap<Object, Object>();
    private boolean allComponentsCreated;
    protected final JAXXContext delegateContext = new DefaultJAXXContext();

    /*-----------------------------------------------------------------------*/
    /*------------------------ Protected components  ------------------------*/
    /*-----------------------------------------------------------------------*/

    protected JButton addButton;
    protected Object bean;
    protected Class<O> beanType;
    protected JPanel beforeFilterPanel;
    protected JToggleButton changeDecorator;
    protected JTextField filterField;
    protected JLabel filterFieldLabel;
    protected JPanel filterPanel;
    protected Boolean filterable;
    protected BeanDoubleListHandler<O> handler;
    protected Boolean highlightFilterText;
    protected String i18nPrefix;
    protected Integer index;
    protected JAXXButtonGroup indexes;
    protected BeanDoubleListModel<O> model;
    protected JPopupMenu popup;
    protected JLabel popupLabel;
    protected JSeparator popupSeparator;
    protected JLabel popupSortLabel;
    protected String property;
    protected JButton removeButton;
    protected JButton resetButton;
    protected Boolean reverseSort;
    protected JList selectedList;
    protected JMenuItem selectedListMoveDownAction;
    protected JMenuItem selectedListMoveUpAction;
    protected JPopupMenu selectedListPopup;
    protected JMenuItem selectedListRemoveAction;
    protected Boolean showDecorator;
    protected Boolean showReset;
    protected Boolean showSelectPopupEnabled;
    protected JRadioButtonMenuItem sortDown;
    protected JAXXButtonGroup sortGroup;
    protected JRadioButtonMenuItem sortUp;
    protected JToolBar toolbarLeft;
    protected JToolBar toolbarRight;
    protected JList universeList;
    protected Boolean useMultiSelect;

    /*-----------------------------------------------------------------------*/
    /*------------------------- Private components  -------------------------*/
    /*-----------------------------------------------------------------------*/

    private JPanel $JPanel0;
    private JPanel $JPanel1;
    private JScrollPane $JScrollPane0;
    private JScrollPane $JScrollPane1;
    private JSeparator $JSeparator0;
    private BeanDoubleList<O> $Table0 = this;

    /*-----------------------------------------------------------------------*/
    /*---------------------- Raw body code from script ----------------------*/
    /*-----------------------------------------------------------------------*/

    public void init(JXPathDecorator<O> decorator, JXPathDecorator<O> decorator2, List<O> universe, List<O> selected) {
          handler.init(decorator, decorator2, universe, selected);
      }
    
    public void init(JXPathDecorator<O> decorator, List<O> universe, List<O> selected) {
        handler.init(decorator, universe, selected);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Constructors  ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public BeanDoubleList() {
        $initialize();
    }

    public BeanDoubleList(JAXXContext parentContext) {
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Statics methods ---------------------------*/
    /*-----------------------------------------------------------------------*/

    public static JAXXObjectDescriptor $getJAXXObjectDescriptor() {
        return JAXXUtil.decodeCompressedJAXXObjectDescriptor($jaxxObjectDescriptor);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------- JAXXObject implementation ----------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public void applyDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).applyDataBinding();
        }
        processDataBinding($binding);
    }

    @Override
    public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        super.firePropertyChange(propertyName, oldValue, newValue);
    }

    @Override
    public Map<String, Object> get$objectMap() {
        return $objectMap;
    }

    @Override
    public JAXXBinding getDataBinding(String bindingId) {
        return $bindings.get(bindingId);
    }

    @Override
    public JAXXBinding[] getDataBindings() {
        return $bindings.values().toArray(new JAXXBinding[$bindings.size()]);
    }

    @Override
    public Object getObjectById(String id) {
        return $objectMap.get(id);
    }

    @Override
    public void processDataBinding(String $binding, boolean $force) {
        if (!$force && $activeBindings.contains($binding)) { 
            return;
        }
        $activeBindings.add($binding);
        try {
            if (allComponentsCreated && $bindings.containsKey($binding)) {
                getDataBinding($binding).processDataBinding();
            }
        } finally {
            $activeBindings.remove($binding);
        }
    }

    @Override
    public void processDataBinding(String $binding) {
        processDataBinding($binding, false);
    }

    @Override
    public void registerDataBinding(JAXXBinding binding) {
        $bindings.put(binding.getId(), binding);
    }

    @Override
    public void removeDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).removeDataBinding();
        }
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- JAXXContext implementation  ---------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public <T> T getContextValue(Class<T> clazz) {
        return delegateContext.getContextValue(clazz, null);
    }

    @Override
    public <T> T getContextValue(Class<T> clazz, String name) {
        return delegateContext.getContextValue(clazz, name);
    }

    @Override
    public JAXXContext getDelegateContext() {
        return delegateContext;
    }

    @Override
    public <O extends Container> O getParentContainer(Class<O> clazz) {
        return SwingUtil.getParentContainer(this, clazz);
    }

    @Override
    public <O extends Container> O getParentContainer(Object source, Class<O> clazz) {
        return SwingUtil.getParentContainer(source, clazz);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz) {
        delegateContext.removeContextValue(clazz, null);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz, String name) {
        delegateContext.removeContextValue(clazz, name);
    }

    @Override
    public <T> void setContextValue(T o) {
        delegateContext.setContextValue(o, null);
    }

    @Override
    public <T> void setContextValue(T o, String name) {
        delegateContext.setContextValue(o, name);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Event methods ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public void doActionPerformed__on__addButton(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.select();
    }

    public void doActionPerformed__on__changeDecorator(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        getHandler().togglePopup();
    }

    public void doActionPerformed__on__removeButton(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.unselect();
    }

    public void doActionPerformed__on__resetButton(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        filterField.setText("");
    }

    public void doActionPerformed__on__selectedListMoveDownAction(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.moveDownSelected( (O) selectedList.getSelectedValue());
    }

    public void doActionPerformed__on__selectedListMoveUpAction(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.moveUpSelected( (O) selectedList.getSelectedValue());
    }

    public void doActionPerformed__on__selectedListRemoveAction(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        model.removeFromSelected( (O) selectedList.getSelectedValue());
    }

    public void doFocusGained__on__$JScrollPane0(FocusEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        universeList.requestFocus();
    }

    public void doFocusGained__on__$JScrollPane1(FocusEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        selectedList.requestFocus();
    }

    public void doFocusGained__on__selectedList(FocusEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.selectFirstRowIfNoSelection(event);
    }

    public void doFocusGained__on__universeList(FocusEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.selectFirstRowIfNoSelection(event);
    }

    public void doKeyPressed__on__selectedList(KeyEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.onKeyPressedOnSelectedList(event);
    }

    public void doKeyPressed__on__universeList(KeyEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.onKeyPressedOnUniverseList(event);
    }

    public void doMouseClicked__on__selectedList(MouseEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.onSelectedListClicked(event);
    }

    public void doMouseClicked__on__universeList(MouseEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.onUniverseListClicked(event);
    }

    public void doPopupMenuCanceled__on__popup(PopupMenuEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        getChangeDecorator().setSelected(false);
    }

    public void doPopupMenuWillBecomeInvisible__on__popup(PopupMenuEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        getChangeDecorator().setSelected(false);
    }

    public void doStateChanged__on__indexes(ChangeEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        setIndex((Integer)indexes.getSelectedValue());
    }

    public void doStateChanged__on__sortGroup(ChangeEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        setReverseSort((Boolean)sortGroup.getSelectedValue());
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public acessor methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public JButton getAddButton() {
        return addButton;
    }

    public Object getBean() {
        return bean;
    }

    public Class<O> getBeanType() {
        return beanType;
    }

    public JPanel getBeforeFilterPanel() {
        return beforeFilterPanel;
    }

    public JToggleButton getChangeDecorator() {
        return changeDecorator;
    }

    public JTextField getFilterField() {
        return filterField;
    }

    public JLabel getFilterFieldLabel() {
        return filterFieldLabel;
    }

    public JPanel getFilterPanel() {
        return filterPanel;
    }

    public Boolean getFilterable() {
        return filterable;
    }

    public BeanDoubleListHandler<O> getHandler() {
        return handler;
    }

    public Boolean getHighlightFilterText() {
        return highlightFilterText;
    }

    public String getI18nPrefix() {
        return i18nPrefix;
    }

    public Integer getIndex() {
        return index;
    }

    public JAXXButtonGroup getIndexes() {
        return indexes;
    }

    public BeanDoubleListModel<O> getModel() {
        return model;
    }

    public JPopupMenu getPopup() {
        return popup;
    }

    public JLabel getPopupLabel() {
        return popupLabel;
    }

    public JSeparator getPopupSeparator() {
        return popupSeparator;
    }

    public JLabel getPopupSortLabel() {
        return popupSortLabel;
    }

    public String getProperty() {
        return property;
    }

    public JButton getRemoveButton() {
        return removeButton;
    }

    public JButton getResetButton() {
        return resetButton;
    }

    public Boolean getReverseSort() {
        return reverseSort;
    }

    public JList getSelectedList() {
        return selectedList;
    }

    public JMenuItem getSelectedListMoveDownAction() {
        return selectedListMoveDownAction;
    }

    public JMenuItem getSelectedListMoveUpAction() {
        return selectedListMoveUpAction;
    }

    public JPopupMenu getSelectedListPopup() {
        return selectedListPopup;
    }

    public JMenuItem getSelectedListRemoveAction() {
        return selectedListRemoveAction;
    }

    public Boolean getShowDecorator() {
        return showDecorator;
    }

    public Boolean getShowReset() {
        return showReset;
    }

    public Boolean getShowSelectPopupEnabled() {
        return showSelectPopupEnabled;
    }

    public JRadioButtonMenuItem getSortDown() {
        return sortDown;
    }

    public JAXXButtonGroup getSortGroup() {
        return sortGroup;
    }

    public JRadioButtonMenuItem getSortUp() {
        return sortUp;
    }

    public JToolBar getToolbarLeft() {
        return toolbarLeft;
    }

    public JToolBar getToolbarRight() {
        return toolbarRight;
    }

    public JList getUniverseList() {
        return universeList;
    }

    public Boolean getUseMultiSelect() {
        return useMultiSelect;
    }

    public Boolean isFilterable() {
        return filterable !=null && filterable;
    }

    public Boolean isHighlightFilterText() {
        return highlightFilterText !=null && highlightFilterText;
    }

    public Boolean isReverseSort() {
        return reverseSort !=null && reverseSort;
    }

    public Boolean isShowDecorator() {
        return showDecorator !=null && showDecorator;
    }

    public Boolean isShowReset() {
        return showReset !=null && showReset;
    }

    public Boolean isShowSelectPopupEnabled() {
        return showSelectPopupEnabled !=null && showSelectPopupEnabled;
    }

    public Boolean isUseMultiSelect() {
        return useMultiSelect !=null && useMultiSelect;
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public mutator methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public void setBean(Object bean) {
        Object oldValue = this.bean;
        this.bean = bean;
        firePropertyChange(PROPERTY_BEAN, oldValue, bean);
    }

    public void setBeanType(Class<O> beanType) {
        Class<O> oldValue = this.beanType;
        this.beanType = beanType;
        firePropertyChange(PROPERTY_BEAN_TYPE, oldValue, beanType);
    }

    public void setFilterable(Boolean filterable) {
        Boolean oldValue = this.filterable;
        this.filterable = filterable;
        firePropertyChange(PROPERTY_FILTERABLE, oldValue, filterable);
    }

    public void setHighlightFilterText(Boolean highlightFilterText) {
        Boolean oldValue = this.highlightFilterText;
        this.highlightFilterText = highlightFilterText;
        firePropertyChange(PROPERTY_HIGHLIGHT_FILTER_TEXT, oldValue, highlightFilterText);
    }

    public void setI18nPrefix(String i18nPrefix) {
        String oldValue = this.i18nPrefix;
        this.i18nPrefix = i18nPrefix;
        firePropertyChange(PROPERTY_I18N_PREFIX, oldValue, i18nPrefix);
    }

    public void setIndex(Integer index) {
        Integer oldValue = this.index;
        this.index = index;
        firePropertyChange(PROPERTY_INDEX, oldValue, index);
    }

    public void setModel(BeanDoubleListModel<O> model) {
        BeanDoubleListModel<O> oldValue = this.model;
        this.model = model;
        firePropertyChange(PROPERTY_MODEL, oldValue, model);
    }

    public void setProperty(String property) {
        String oldValue = this.property;
        this.property = property;
        firePropertyChange(PROPERTY_PROPERTY, oldValue, property);
    }

    public void setReverseSort(Boolean reverseSort) {
        Boolean oldValue = this.reverseSort;
        this.reverseSort = reverseSort;
        firePropertyChange(PROPERTY_REVERSE_SORT, oldValue, reverseSort);
    }

    public void setShowDecorator(Boolean showDecorator) {
        Boolean oldValue = this.showDecorator;
        this.showDecorator = showDecorator;
        firePropertyChange(PROPERTY_SHOW_DECORATOR, oldValue, showDecorator);
    }

    public void setShowReset(Boolean showReset) {
        Boolean oldValue = this.showReset;
        this.showReset = showReset;
        firePropertyChange(PROPERTY_SHOW_RESET, oldValue, showReset);
    }

    public void setShowSelectPopupEnabled(Boolean showSelectPopupEnabled) {
        Boolean oldValue = this.showSelectPopupEnabled;
        this.showSelectPopupEnabled = showSelectPopupEnabled;
        firePropertyChange(PROPERTY_SHOW_SELECT_POPUP_ENABLED, oldValue, showSelectPopupEnabled);
    }

    public void setUseMultiSelect(Boolean useMultiSelect) {
        Boolean oldValue = this.useMultiSelect;
        this.useMultiSelect = useMultiSelect;
        firePropertyChange(PROPERTY_USE_MULTI_SELECT, oldValue, useMultiSelect);
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Protected acessors methods  ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected JPanel get$JPanel0() {
        return $JPanel0;
    }

    protected JPanel get$JPanel1() {
        return $JPanel1;
    }

    protected JScrollPane get$JScrollPane0() {
        return $JScrollPane0;
    }

    protected JScrollPane get$JScrollPane1() {
        return $JScrollPane1;
    }

    protected JSeparator get$JSeparator0() {
        return $JSeparator0;
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Components creation methods ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected void addChildrenToFilterPanel() {
        if (!allComponentsCreated) {
            return;
        }
        filterPanel.add($JPanel0, BorderLayout.WEST);
        filterPanel.add(filterField, BorderLayout.CENTER);
        filterPanel.add(toolbarRight, BorderLayout.EAST);
    }

    protected void addChildrenToPopup() {
        if (!allComponentsCreated) {
            return;
        }
        popup.add(popupSortLabel);
        popup.add(sortUp);
        popup.add(sortDown);
        popup.add(popupSeparator);
        popup.add(popupLabel);
        popup.add($JSeparator0);
    }

    protected void addChildrenToSelectedListPopup() {
        if (!allComponentsCreated) {
            return;
        }
        selectedListPopup.add(selectedListMoveUpAction);
        selectedListPopup.add(selectedListMoveDownAction);
        selectedListPopup.add(selectedListRemoveAction);
    }

    protected void addChildrenToSortDown() {
        if (!allComponentsCreated) {
            return;
        }
        { ButtonGroup $buttonGroup = sortGroup; sortDown.putClientProperty("$buttonGroup", $buttonGroup); $buttonGroup.add(sortDown); }
    }

    protected void addChildrenToSortUp() {
        if (!allComponentsCreated) {
            return;
        }
        { ButtonGroup $buttonGroup = sortGroup; sortUp.putClientProperty("$buttonGroup", $buttonGroup); $buttonGroup.add(sortUp); }
    }

    protected void addChildrenToToolbarLeft() {
        if (!allComponentsCreated) {
            return;
        }
        toolbarLeft.add(resetButton);
    }

    protected void addChildrenToToolbarRight() {
        if (!allComponentsCreated) {
            return;
        }
        toolbarRight.add(changeDecorator);
    }

    protected void createAddButton() {
        $objectMap.put("addButton", addButton = new JButton());
        
        addButton.setName("addButton");
        addButton.setToolTipText(t("beandoublelist.button.add"));
        addButton.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__addButton"));
    }

    protected void createBean() {
        $objectMap.put("bean", bean = null);
    }

    protected void createBeanType() {
        $objectMap.put("beanType", beanType = null);
    }

    protected void createBeforeFilterPanel() {
        $objectMap.put("beforeFilterPanel", beforeFilterPanel = new JPanel());
        
        beforeFilterPanel.setName("beforeFilterPanel");
        beforeFilterPanel.setLayout(new BorderLayout());
    }

    protected void createChangeDecorator() {
        $objectMap.put("changeDecorator", changeDecorator = new JToggleButton());
        
        changeDecorator.setName("changeDecorator");
        changeDecorator.setFocusable(false);
        changeDecorator.setToolTipText(t("beandoublelist.action.sort.tip"));
        changeDecorator.setFocusPainted(false);
        changeDecorator.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__changeDecorator"));
    }

    protected void createFilterField() {
        $objectMap.put("filterField", filterField = new JTextField());
        
        filterField.setName("filterField");
        filterField.setColumns(15);
    }

    protected void createFilterFieldLabel() {
        $objectMap.put("filterFieldLabel", filterFieldLabel = new JLabel());
        
        filterFieldLabel.setName("filterFieldLabel");
    }

    protected void createFilterPanel() {
        $objectMap.put("filterPanel", filterPanel = new JPanel());
        
        filterPanel.setName("filterPanel");
        filterPanel.setLayout(new BorderLayout());
    }

    protected void createFilterable() {
        $objectMap.put("filterable", filterable = true);
    }

    protected void createHandler() {
        $objectMap.put("handler", handler = new BeanDoubleListHandler<O>(this));
    }

    protected void createHighlightFilterText() {
        $objectMap.put("highlightFilterText", highlightFilterText = false);
    }

    protected void createI18nPrefix() {
        $objectMap.put("i18nPrefix", i18nPrefix = "beanlist.common.");
    }

    protected void createIndex() {
        $objectMap.put("index", index = 0);
    }

    protected void createIndexes() {
        $objectMap.put("indexes", indexes = new JAXXButtonGroup());
        
        indexes.addChangeListener(JAXXUtil.getEventListener(ChangeListener.class, "stateChanged", this, "doStateChanged__on__indexes"));
    }

    protected void createModel() {
        $objectMap.put("model", model = new BeanDoubleListModel<O>());
    }

    protected void createPopup() {
        $objectMap.put("popup", popup = new JPopupMenu());
        
        popup.setName("popup");
        popup.addPopupMenuListener(JAXXUtil.getEventListener(PopupMenuListener.class, "popupMenuCanceled", this, "doPopupMenuCanceled__on__popup"));
        popup.addPopupMenuListener(JAXXUtil.getEventListener(PopupMenuListener.class, "popupMenuWillBecomeInvisible", this, "doPopupMenuWillBecomeInvisible__on__popup"));
    }

    protected void createPopupLabel() {
        $objectMap.put("popupLabel", popupLabel = new JLabel());
        
        popupLabel.setName("popupLabel");
    }

    protected void createPopupSeparator() {
        $objectMap.put("popupSeparator", popupSeparator = new JSeparator());
        
        popupSeparator.setName("popupSeparator");
    }

    protected void createPopupSortLabel() {
        $objectMap.put("popupSortLabel", popupSortLabel = new JLabel());
        
        popupSortLabel.setName("popupSortLabel");
        popupSortLabel.setText(t("bean.sort.label"));
    }

    protected void createProperty() {
        $objectMap.put("property", property = "");
    }

    protected void createRemoveButton() {
        $objectMap.put("removeButton", removeButton = new JButton());
        
        removeButton.setName("removeButton");
        removeButton.setToolTipText(t("beandoublelist.button.remove"));
        removeButton.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__removeButton"));
    }

    protected void createResetButton() {
        $objectMap.put("resetButton", resetButton = new JButton());
        
        resetButton.setName("resetButton");
        resetButton.setFocusable(false);
        resetButton.setToolTipText(t("beandoublelist.action.reset.tip"));
        resetButton.setFocusPainted(false);
        resetButton.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__resetButton"));
    }

    protected void createReverseSort() {
        $objectMap.put("reverseSort", reverseSort = false);
    }

    protected void createSelectedList() {
        $objectMap.put("selectedList", selectedList = new JList());
        
        selectedList.setName("selectedList");
        selectedList.addFocusListener(JAXXUtil.getEventListener(FocusListener.class, "focusGained", this, "doFocusGained__on__selectedList"));
        selectedList.addKeyListener(JAXXUtil.getEventListener(KeyListener.class, "keyPressed", this, "doKeyPressed__on__selectedList"));
        selectedList.addMouseListener(JAXXUtil.getEventListener(MouseListener.class, "mouseClicked", this, "doMouseClicked__on__selectedList"));
    }

    protected void createSelectedListMoveDownAction() {
        $objectMap.put("selectedListMoveDownAction", selectedListMoveDownAction = new JMenuItem());
        
        selectedListMoveDownAction.setName("selectedListMoveDownAction");
        selectedListMoveDownAction.setText(t("bean.action.selected.move.down"));
        selectedListMoveDownAction.setToolTipText(t("bean.action.selected.move.down.tip"));
        selectedListMoveDownAction.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__selectedListMoveDownAction"));
    }

    protected void createSelectedListMoveUpAction() {
        $objectMap.put("selectedListMoveUpAction", selectedListMoveUpAction = new JMenuItem());
        
        selectedListMoveUpAction.setName("selectedListMoveUpAction");
        selectedListMoveUpAction.setText(t("bean.action.selected.move.up"));
        selectedListMoveUpAction.setToolTipText(t("bean.action.selected.move.up.tip"));
        selectedListMoveUpAction.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__selectedListMoveUpAction"));
    }

    protected void createSelectedListPopup() {
        $objectMap.put("selectedListPopup", selectedListPopup = new JPopupMenu());
        
        selectedListPopup.setName("selectedListPopup");
    }

    protected void createSelectedListRemoveAction() {
        $objectMap.put("selectedListRemoveAction", selectedListRemoveAction = new JMenuItem());
        
        selectedListRemoveAction.setName("selectedListRemoveAction");
        selectedListRemoveAction.setText(t("bean.action.selected.remove"));
        selectedListRemoveAction.setToolTipText(t("bean.action.selected.remove.tip"));
        selectedListRemoveAction.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__selectedListRemoveAction"));
    }

    protected void createShowDecorator() {
        $objectMap.put("showDecorator", showDecorator = true);
    }

    protected void createShowReset() {
        $objectMap.put("showReset", showReset = true);
    }

    protected void createShowSelectPopupEnabled() {
        $objectMap.put("showSelectPopupEnabled", showSelectPopupEnabled = true);
    }

    protected void createSortDown() {
        $objectMap.put("sortDown", sortDown = new JRadioButtonMenuItem());
        
        sortDown.setName("sortDown");
        sortDown.setText(t("bean.sort.down"));
    }

    protected void createSortGroup() {
        $objectMap.put("sortGroup", sortGroup = new JAXXButtonGroup());
        
        sortGroup.addChangeListener(JAXXUtil.getEventListener(ChangeListener.class, "stateChanged", this, "doStateChanged__on__sortGroup"));
    }

    protected void createSortUp() {
        $objectMap.put("sortUp", sortUp = new JRadioButtonMenuItem());
        
        sortUp.setName("sortUp");
        sortUp.setText(t("bean.sort.up"));
    }

    protected void createToolbarLeft() {
        $objectMap.put("toolbarLeft", toolbarLeft = new JToolBar());
        
        toolbarLeft.setName("toolbarLeft");
        toolbarLeft.setBorderPainted(false);
        toolbarLeft.setFloatable(false);
    }

    protected void createToolbarRight() {
        $objectMap.put("toolbarRight", toolbarRight = new JToolBar());
        
        toolbarRight.setName("toolbarRight");
        toolbarRight.setBorderPainted(false);
        toolbarRight.setFloatable(false);
    }

    protected void createUniverseList() {
        $objectMap.put("universeList", universeList = new JList());
        
        universeList.setName("universeList");
        universeList.addFocusListener(JAXXUtil.getEventListener(FocusListener.class, "focusGained", this, "doFocusGained__on__universeList"));
        universeList.addKeyListener(JAXXUtil.getEventListener(KeyListener.class, "keyPressed", this, "doKeyPressed__on__universeList"));
        universeList.addMouseListener(JAXXUtil.getEventListener(MouseListener.class, "mouseClicked", this, "doMouseClicked__on__universeList"));
    }

    protected void createUseMultiSelect() {
        $objectMap.put("useMultiSelect", useMultiSelect = false);
    }

    /*-----------------------------------------------------------------------*/
    /*------------------------ Internal jaxx methods ------------------------*/
    /*-----------------------------------------------------------------------*/

    private void $completeSetup() {
        allComponentsCreated = true;
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        // inline complete setup of $Table0
        add(beforeFilterPanel, new GridBagConstraints(0, 0, 3, 1, 0.0, 0.0, 10, 1, new Insets(3, 3, 3, 3), 0, 0));
        add(filterPanel, new GridBagConstraints(0, 1, 3, 1, 0.0, 0.0, 10, 1, new Insets(3, 3, 3, 3), 0, 0));
        add($JScrollPane0, new GridBagConstraints(0, 2, 1, 1, 0.5, 1.0, 10, 1, new Insets(3, 3, 3, 3), 0, 0));
        add($JPanel1, new GridBagConstraints(1, 2, 1, 1, 0.0, 0.0, 11, 0, new Insets(3, 3, 3, 3), 0, 0));
        add($JScrollPane1, new GridBagConstraints(2, 2, 1, 1, 0.5, 1.0, 10, 1, new Insets(3, 3, 3, 3), 0, 0));
        addChildrenToPopup();
        addChildrenToSortUp();
        addChildrenToSortDown();
        addChildrenToSelectedListPopup();
        addChildrenToFilterPanel();
        // inline complete setup of $JPanel0
        $JPanel0.add(filterFieldLabel, BorderLayout.WEST);
        $JPanel0.add(toolbarLeft, BorderLayout.EAST);
        addChildrenToToolbarLeft();
        addChildrenToToolbarRight();
        // inline complete setup of $JScrollPane0
        $JScrollPane0.getViewport().add(universeList);
        // inline complete setup of $JPanel1
        $JPanel1.add(addButton);
        $JPanel1.add(removeButton);
        // inline complete setup of $JScrollPane1
        $JScrollPane1.getViewport().add(selectedList);
        
        // apply 15 data bindings
        JAXXUtil.applyDataBinding(this, $bindings.keySet());
        
        // apply 18 property setters
        popup.setBorder(new TitledBorder(t("beandoublelist.popup.title")));
        popupSortLabel.setIcon(SwingUtil.createActionIcon("bean-sort"));
        sortUp.setIcon(SwingUtil.createActionIcon("bean-sort-up"));
        { sortUp.putClientProperty("$value", false);  Object $buttonGroup = sortUp.getClientProperty("$buttonGroup"); if ($buttonGroup instanceof JAXXButtonGroup) { ((JAXXButtonGroup) $buttonGroup).updateSelectedValue(); } }
        
        sortDown.setIcon(SwingUtil.createActionIcon("bean-sort-down"));
        { sortDown.putClientProperty("$value", true);  Object $buttonGroup = sortDown.getClientProperty("$buttonGroup"); if ($buttonGroup instanceof JAXXButtonGroup) { ((JAXXButtonGroup) $buttonGroup).updateSelectedValue(); } }
        
        selectedListMoveUpAction.setIcon(SwingUtil.createActionIcon("bean-doublelist-up"));
        selectedListMoveDownAction.setIcon(SwingUtil.createActionIcon("bean-doublelist-down"));
        selectedListRemoveAction.setIcon(SwingUtil.createActionIcon("bean-doublelist-unselect"));
        filterFieldLabel.setIcon(SwingUtil.createActionIcon("filter"));
        resetButton.setIcon(SwingUtil.createActionIcon("doublelist-reset"));
        changeDecorator.setIcon(SwingUtil.createActionIcon("doublelist-sort"));
        addButton.setIcon(SwingUtil.createActionIcon("bean-doublelist-select"));
        removeButton.setIcon(SwingUtil.createActionIcon("bean-doublelist-unselect"));
    }

    private void $initialize() {
        if (allComponentsCreated) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        $objectMap.put("$Table0", $Table0);
        createFilterable();
        createReverseSort();
        createShowDecorator();
        createShowReset();
        createUseMultiSelect();
        createShowSelectPopupEnabled();
        createHighlightFilterText();
        createBeanType();
        createProperty();
        createBean();
        createI18nPrefix();
        createModel();
        createHandler();
        createIndex();
        createIndexes();
        createSortGroup();
        createPopup();
        createPopupSortLabel();
        createSortUp();
        createSortDown();
        createPopupSeparator();
        createPopupLabel();
        // inline creation of $JSeparator0
        $objectMap.put("$JSeparator0", $JSeparator0 = new JSeparator());
        
        $JSeparator0.setName("$JSeparator0");
        createSelectedListPopup();
        createSelectedListMoveUpAction();
        createSelectedListMoveDownAction();
        createSelectedListRemoveAction();
        createBeforeFilterPanel();
        createFilterPanel();
        // inline creation of $JPanel0
        $objectMap.put("$JPanel0", $JPanel0 = new JPanel());
        
        $JPanel0.setName("$JPanel0");
        $JPanel0.setLayout(new BorderLayout());
        createFilterFieldLabel();
        createToolbarLeft();
        createResetButton();
        createFilterField();
        createToolbarRight();
        createChangeDecorator();
        // inline creation of $JScrollPane0
        $objectMap.put("$JScrollPane0", $JScrollPane0 = new JScrollPane());
        
        $JScrollPane0.setName("$JScrollPane0");
        $JScrollPane0.addFocusListener(JAXXUtil.getEventListener(FocusListener.class, "focusGained", this, "doFocusGained__on__$JScrollPane0"));
        createUniverseList();
        // inline creation of $JPanel1
        $objectMap.put("$JPanel1", $JPanel1 = new JPanel());
        
        $JPanel1.setName("$JPanel1");
        $JPanel1.setLayout(new GridLayout(0,1));
        createAddButton();
        createRemoveButton();
        // inline creation of $JScrollPane1
        $objectMap.put("$JScrollPane1", $JScrollPane1 = new JScrollPane());
        
        $JScrollPane1.setName("$JScrollPane1");
        $JScrollPane1.addFocusListener(JAXXUtil.getEventListener(FocusListener.class, "focusGained", this, "doFocusGained__on__$JScrollPane1"));
        createSelectedList();
        // inline creation of $Table0
        setName("$Table0");
        
        // registers 15 data bindings
        $registerDefaultBindings();
        $completeSetup();
    }

    private void $registerDefaultBindings() {
        // register 15 data bindings
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_SORT_UP_SELECTED, true ,"reverseSort") {
        
            @Override
            public void processDataBinding() {
                sortUp.setSelected(!isReverseSort());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_SORT_DOWN_SELECTED, true ,"reverseSort") {
        
            @Override
            public void processDataBinding() {
                sortDown.setSelected(isReverseSort());
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_SELECTED_LIST_MOVE_UP_ACTION_ENABLED, true, true) {
        
            @Override
            public void applyDataBinding() {
                addPropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().addPropertyChangeListener("selectedUpEnabled", this);
                }
            }
        
            @Override
            public void processDataBinding() {
                if (getModel() != null) {
                    selectedListMoveUpAction.setEnabled(getModel().isSelectedUpEnabled());
                }
            }
        
            @Override
            public void removeDataBinding() {
                removePropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().removePropertyChangeListener("selectedUpEnabled", this);
                }
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_SELECTED_LIST_MOVE_DOWN_ACTION_ENABLED, true, true) {
        
            @Override
            public void applyDataBinding() {
                addPropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().addPropertyChangeListener("selectedDownEnabled", this);
                }
            }
        
            @Override
            public void processDataBinding() {
                if (getModel() != null) {
                    selectedListMoveDownAction.setEnabled(getModel().isSelectedDownEnabled());
                }
            }
        
            @Override
            public void removeDataBinding() {
                removePropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().removePropertyChangeListener("selectedDownEnabled", this);
                }
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_SELECTED_LIST_REMOVE_ACTION_ENABLED, true, true) {
        
            @Override
            public void applyDataBinding() {
                addPropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().addPropertyChangeListener("removeEnabled", this);
                }
            }
        
            @Override
            public void processDataBinding() {
                if (getModel() != null) {
                    selectedListRemoveAction.setEnabled(getModel().isRemoveEnabled());
                }
            }
        
            @Override
            public void removeDataBinding() {
                removePropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().removePropertyChangeListener("removeEnabled", this);
                }
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_FILTER_PANEL_VISIBLE, true ,"filterable") {
        
            @Override
            public void processDataBinding() {
                filterPanel.setVisible(isFilterable());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_TOOLBAR_LEFT_VISIBLE, true ,"showReset") {
        
            @Override
            public void processDataBinding() {
                toolbarLeft.setVisible(isShowReset());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_RESET_BUTTON_ENABLED, true ,"showReset" ,"enabled") {
        
            @Override
            public void processDataBinding() {
                resetButton.setEnabled(isShowReset() && isEnabled());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_TOOLBAR_RIGHT_VISIBLE, true ,"showDecorator") {
        
            @Override
            public void processDataBinding() {
                toolbarRight.setVisible(isShowDecorator());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_CHANGE_DECORATOR_ENABLED, true ,"showDecorator" ,"enabled") {
        
            @Override
            public void processDataBinding() {
                changeDecorator.setEnabled(isShowDecorator() && isEnabled());
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_UNIVERSE_LIST_MODEL, true, true) {
        
            @Override
            public void applyDataBinding() {
                addPropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().addPropertyChangeListener("universeModel", this);
                }
            }
        
            @Override
            public void processDataBinding() {
                if (getModel() != null) {
                    universeList.setModel(getModel().getUniverseModel());
                }
            }
        
            @Override
            public void removeDataBinding() {
                removePropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().removePropertyChangeListener("universeModel", this);
                }
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_ADD_BUTTON_ENABLED, true, true) {
        
            @Override
            public void applyDataBinding() {
                addPropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().addPropertyChangeListener("addEnabled", this);
                }
            }
        
            @Override
            public void processDataBinding() {
                if (getModel() != null) {
                    addButton.setEnabled(getModel().isAddEnabled());
                }
            }
        
            @Override
            public void removeDataBinding() {
                removePropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().removePropertyChangeListener("addEnabled", this);
                }
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_REMOVE_BUTTON_ENABLED, true, true) {
        
            @Override
            public void applyDataBinding() {
                addPropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().addPropertyChangeListener("removeEnabled", this);
                }
            }
        
            @Override
            public void processDataBinding() {
                if (getModel() != null) {
                    removeButton.setEnabled(getModel().isRemoveEnabled());
                }
            }
        
            @Override
            public void removeDataBinding() {
                removePropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().removePropertyChangeListener("removeEnabled", this);
                }
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_SELECTED_LIST_COMPONENT_POPUP_MENU, true ,"showSelectPopupEnabled") {
        
            @Override
            public void processDataBinding() {
                selectedList.setComponentPopupMenu(handler.getSelectedListPopup(isShowSelectPopupEnabled()));
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_SELECTED_LIST_MODEL, true, true) {
        
            @Override
            public void applyDataBinding() {
                addPropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().addPropertyChangeListener("selectedModel", this);
                }
            }
        
            @Override
            public void processDataBinding() {
                if (getModel() != null) {
                    selectedList.setModel(getModel().getSelectedModel());
                }
            }
        
            @Override
            public void removeDataBinding() {
                removePropertyChangeListener("model", this);
                if (getModel() != null) {
                    getModel().removePropertyChangeListener("selectedModel", this);
                }
            }
        });
    }

}