/**
 * *##% 
 * JAXX Widgets
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.editor.config;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.editor.config.model.CallBackEntry;
import jaxx.runtime.swing.editor.config.model.CategoryModel;
import jaxx.runtime.swing.editor.config.model.ConfigUIModel;
import jaxx.runtime.swing.editor.config.model.OptionModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * La classe pour construire l'ui
 *
 * @author chemit
 */
public class ConfigUIBuilder {

    public static final Log log = LogFactory.getLog(ConfigUIBuilder.class);
    public static final String CALLBACKS_WITH_OPTIONS = "callbacksWithOptions";
    public static final String CALLBACKS = "callbacks";

    /**
     * Construire l'ui de configuration (sous forme de panel)
     *
     * @param parentContext   le context applicatif
     * @param model           le modele de l'ui de configuration
     * @param defaultCategory la categorie a selectionner
     * @return l'ui instanciate
     */
    public static ConfigUI newConfigUI(JAXXContext parentContext,
                                       final ConfigUIModel model,
                                       String defaultCategory) {
        JAXXContext tx = new JAXXInitialContext().add(parentContext).add(model);

        final ConfigUI ui = new ConfigUI(tx);
        JButton quitButton = ui.getQuit();

        // prepare quit action
        Action quitAction = new AbstractAction(quitButton.getText(),
                quitButton.getIcon()) {

            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                if (!canQuitCategory(ui)) {
                    return;
                }

                final Window parentWindow = ui.getParentContainer(Window.class);

                if (!model.isSaved() || model.isStandalone()) {

                    // just quit, no callBack can be apply here

                    ui.destroy();

                    // close the configu ui
                    parentWindow.dispose();
                    return;
                }

                Map<CallBackEntry, List<OptionModel>> forSaved;
                forSaved = model.getCallBacksForSaved();

                if (forSaved.isEmpty()) {
                    //  just quit, no callBack to call

                    ui.destroy();

                    // close the configu ui
                    parentWindow.dispose();
                    return;
                }

                forSaved = model.getCallBacksForSaved();

                // init callBackUI

                ConfigCallBackUI lastUI = new ConfigCallBackUI(
                        new JAXXInitialContext().
                                add("parent", parentWindow).
                                add(ui).
                                add(CALLBACKS_WITH_OPTIONS, forSaved).
                                add(CALLBACKS, new ArrayList<CallBackEntry>(forSaved.keySet())).
                                add(new ConfigCallBackUIHandler())
                );

                lastUI.init();
                ui.setVisible(false);
                parentWindow.remove(ui);
                parentWindow.add(lastUI);
                
                SwingUtilities.invokeLater(new Runnable() {

                    @Override
                    public void run() {
                        parentWindow.validate();
                    }
                });
//                dialog.pack();
//                SwingUtil.center(parentWindow, lastUI);

//                lastUI.setVisible(true);
            }
        };
        String tip = quitButton.getToolTipText();
        quitButton.setAction(quitAction);
        quitButton.setToolTipText(tip);

        // build categories tabs
        for (CategoryModel categoryModel : model) {
            String category = categoryModel.getCategory();
            String categoryLabel = _(categoryModel.getCategoryLabel());
            ConfigCategoryUI p = new ConfigCategoryUI(new
                    JAXXInitialContext().add(ui).add(categoryModel));
            p.getCategoryLabel().setText(categoryLabel);
            p.setName(category);
            ui.getCategories().addTab(_(category), null, p, categoryLabel);
        }

        model.setCategory(defaultCategory);
        int categoryIndex = model.getCategoryIndex(defaultCategory);
        if (log.isDebugEnabled()) {
            log.debug("index of default category (" + defaultCategory + ")  : "
                    + categoryIndex);
        }
        ui.getCategories().setSelectedIndex(categoryIndex);
        return ui;
    }

    /**
     * Affiche l'ui de configuration dans un boite de dialogue.
     *
     * @param configUI    l'ui de configuration
     * @param ui          l'ui parent de la boite de dialogue a afficher
     *                    (peut etre nulle)
     * @param undecorated un drapeau pour savoir si on affiche les decorations
     *                    de fenetre
     */
    public static void showConfigUI(final ConfigUI configUI,
                                    Frame ui,
                                    boolean undecorated) {
        JDialog f = new JDialog(ui, true);
        f.setTitle(_("config.title"));
        f.add(configUI);
        if (ui != null) {
            f.setIconImage(ui.getIconImage());
        }
        // pour savoir si l'ui est autonome
        configUI.getModel().setStandalone(ui == null);

        f.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        f.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                ActionEvent myEvent = new ActionEvent(e.getSource(), 1, "quit");
                configUI.getQuit().getAction().actionPerformed(myEvent);
            }
        });
        f.setUndecorated(undecorated);
        JRootPane rootPane = f.getRootPane();
        rootPane.setDefaultButton(configUI.getQuit());
        rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                KeyStroke.getKeyStroke("ESCAPE"), "quit");
        rootPane.getActionMap().put("quit", configUI.getQuit().getAction());
        f.pack();
        SwingUtil.center(ui, f);
        f.setVisible(true);
    }

    protected static boolean canQuitCategory(ConfigUI ui) {
        boolean canContinue = true;
        ConfigUIModel model = ui.getModel();
        CategoryModel categoryModel = model.getCategoryModel();
        String categoryName = _(categoryModel.getCategory());
        if (!categoryModel.isValid()) {

            // the category is not valid
            // get all the invalid options

            StringBuilder buffer = new StringBuilder();
            buffer.append(_("config.message.quit.invalid.category",
                    categoryName));
            buffer.append('\n');
            for (OptionModel m : categoryModel.getInvalidOptions()) {
                buffer.append("\n- ").append(m.getKey());
            }
            buffer.append('\n');
            int reponse = askUser(ui,
                    _("config.title.need.confirm"),
                    buffer.toString(),
                    JOptionPane.ERROR_MESSAGE,
                    new Object[]{
                            _("config.choice.continue"),
                            _("config.choice.cancel")},
                    0);

            switch (reponse) {
                case JOptionPane.CLOSED_OPTION:
                case 1:
                    canContinue = false;
                    break;
                case 0:
                    if (categoryModel.isModified()) {
                        // wil reset category
                        model.reset();
                    }
                    break;
            }
        } else if (categoryModel.isModified()) {

            // category was modified, ask user if wants to save

            StringBuilder buffer = new StringBuilder();
            buffer.append(_("config.message.quit.valid.and.modified.category",
                    categoryName)).append('\n');
            for (OptionModel m : categoryModel.getModifiedOptions()) {
                buffer.append("\n- ").append(m.getKey());
            }
            buffer.append('\n');

            int reponse = askUser(ui,
                    _("config.title.need.confirm"), buffer.toString(),
                    JOptionPane.WARNING_MESSAGE,
                    new Object[]{
                            _("config.choice.save"),
                            _("config.choice.doNotSave"),
                            _("config.choice.cancel")},
                    0);

            switch (reponse) {
                case JOptionPane.CLOSED_OPTION:
                case 2:
                    canContinue = false;
                    break;
                case 0:
                    // will save ui
                    model.saveModified();
                    break;
                case 1:
                    // wil reset ui
                    model.reset();
                    break;
            }

        }
        return canContinue;
    }

    public static int askUser(ConfigUI parent,
                              String title,
                              String message,
                              int typeMessage,
                              Object[] options,
                              int defaultOption) {

        int response = JOptionPane.showOptionDialog(
                parent,
                message,
                title,
                JOptionPane.DEFAULT_OPTION,
                typeMessage,
                null,
                options,
                options[defaultOption]
        );

        return response;
    }
}
