/**
/**
 * ##% Copyright (C) 2008 Code Lutin, Tony Chemit
 * This program is free software; you
 * can redistribute it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details. You
 * should have received a copy of the GNU General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 59 Temple Place
 * - Suite 330, Boston, MA 02111-1307, USA. 
 * ##%
 */
package org.nuiton.jaxx.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;

import javax.swing.event.EventListenerList;
import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;

/**
 * Factory for UI, using a cache and a provider to find ui implementations.
 *
 * @author chemit
 */
public class UIFactory {

    static protected final Log log = LogFactory.getLog(UIFactory.class);

    private final String applicationName;

    private final DialogUIDef<?,?,?>[] defs;

    private final EventListenerList listeners;

    public UIFactory(String applicationName, DialogUIDef<?,?,?>[] defs, FactoryWindowListener... listeners) {
        this.applicationName = applicationName;
        this.listeners = new EventListenerList();
        for (FactoryWindowListener listener : listeners) {
            listener.setFactory(this);
            addFactoryWindowListener(listener);
        }
        this.defs = defs;
        long t0 = System.nanoTime();
        if (log.isDebugEnabled()) {
            log.debug("start at " + new java.util.Date());
        }
        try {
            init();
        } catch (Exception e) {
            log.error(e);
            throw new RuntimeException(e);
        } finally {
            if (log.isDebugEnabled()) {
                log.info("end in " + StringUtil.convertTime(t0, System.nanoTime()));
            }
        }
    }

    public void addFactoryWindowListener(FactoryWindowListener l) {
        listeners.add(FactoryWindowListener.class, l);
        if (log.isDebugEnabled()) {
            log.debug("after added (" + listeners.getListenerCount() + ") : " + l);
        }
    }

    public void removeFactoryWindowListener(FactoryWindowListener l) {
        listeners.remove(FactoryWindowListener.class, l);
        for (DialogUIDef<?,?,?> def : getDefs()) {
            if (def.uiInstance != null) {
                def.uiInstance.removeWindowListener(l);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug(" after removed (" + listeners.getListenerCount() + ") : " + l);
        }
        if (listeners.getListenerCount(FactoryWindowListener.class) == 0) {
            // close for real factory
            close();
        }
    }

    public void close() {
        log.info(this + " at " + new java.util.Date());
        for (DialogUIDef<?, ?, ?> def : defs) {
            DialogUI<?> ui = def.uiInstance;
            if (ui != null) {
                ui.getHandler().dispose();
                def.uiInstance = null;
            }
        }
        if (listeners.getListenerCount(FactoryWindowListener.class) > 0) {
            log.warn("some listeners where not properly removed, force deletion...");
            for (FactoryWindowListener listener : listeners.getListeners(FactoryWindowListener.class)) {
                removeFactoryWindowListener(listener);
            }
        }

    }

    protected void init() {

        UIProvider[] providers = detectProviders();

        for (DialogUIDef<?, ?, ?> def : defs) {
            initDef(providers, def);
            if (def.getUiImplClass() == null) {
                throw new IllegalStateException("could not find implementation for ui def " + def);
            }
        }
    }

    protected void initDef(UIProvider[] providers, DialogUIDef<?, ?, ?> def) {
        for (UIProvider provider : providers) {
            Class<?> uiImplClass = provider.findUIImplementation(def);
            if (uiImplClass != null) {
                if (log.isDebugEnabled()) {
                    log.debug("init done for " + def);
                }
                // ui implementation was found
                break;
            }
        }
    }

    protected UIProvider[] detectProviders() {
        long t0 = System.nanoTime();
        List<UIProvider> providers = new ArrayList<UIProvider>();
        for (UIProvider provider : ServiceLoader.load(UIProvider.class)) {
            if (applicationName.equals(provider.getApplicationName())) {
                if (log.isDebugEnabled()) {
                    log.debug("provider detected [" + provider + ']');
                }
                providers.add(provider);
            }
        }
        log.info("found " + providers.size() + " ui provider(s) in " + StringUtil.convertTime(t0, System.nanoTime()) + " : " + providers);
        return providers.toArray(new UIProvider[providers.size()]);
    }

    protected DialogUIDef<?,?,?>[] getDefs() {
        return defs;
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }

    public <M extends DialogUIModel, U extends DialogUI<H>, H extends DialogUIHandler<M, U>> U getUI(DialogUIDef<M, U, H> uiType, Object... params) {
        U ui = uiType.uiInstance;
        if (ui == null) {
            try {
                ui = uiType.newUI();
                M model = uiType.newModel();
                H handler = uiType.newHandler(ui, model, params);
                registerUI(uiType, ui, handler);
            } catch (Exception e) {
                throw new IllegalStateException("could not instanciate ui handler " + uiType + " for reason : " + e.getMessage(), e);
            }
        }
        return ui;
    }

    protected <M extends DialogUIModel, U extends DialogUI<H>, H extends DialogUIHandler<M, U>> void registerUI(DialogUIDef<M, U, H> uiType, U ui, H handler) {
        ui.setHandler(handler);
        handler.init();
        uiType.setUiInstance(ui);
        for (FactoryWindowListener listener : listeners.getListeners(FactoryWindowListener.class)) {
            if (log.isDebugEnabled()) {
                log.debug("----- addFactoryWindowListener " + listener + " to " + ui);
            }
            ui.addWindowListener(listener);
        }
    }

}