/**
 * ##% Copyright (C) 2008 Code Lutin, Tony Chemit
 * This program is free software; you
 * can redistribute it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details. You
 * should have received a copy of the GNU General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 59 Temple Place
 * - Suite 330, Boston, MA 02111-1307, USA. 
 * ##%
 */
package org.nuiton.jaxx.action;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

/** @author chemit */
public class ActionProviderFromProperties<A extends MyAbstractAction> implements ActionProvider<A> {

    /** default prefix for an entryin mapping file. */
    protected static final String ACTION_KEY_PREFIX = "action.";

    protected static final String actionsFileLocation = "META-INF/jaxx-%1$s-actions.properties";

    protected String propertiesPath;

    protected static Log log = LogFactory.getLog(ActionProviderFromProperties.class);

    protected Class<A> baseClass;

    protected Map<String, Class<? extends A>> actions;


    protected ActionProviderFromProperties(Class<A> baseClass) {

        this.baseClass = baseClass;
        this.propertiesPath = "/" + String.format(actionsFileLocation, baseClass.getSimpleName());
        this.actions = initCache();
    }

    public Class<A> getBaseClass() {
        return baseClass;
    }

    public Map<String, Class<? extends A>> getClasses() {
        return actions;
    }

    @Override
    public String toString() {
        return super.toString() + "<baseClass:" + baseClass.getSimpleName() + ">";
    }

    protected void clearCache() {
        if (actions != null) {
            actions.clear();
            actions = null;
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        clearCache();
    }

    @SuppressWarnings({"unchecked"})
    protected Map<String, Class<? extends A>> initCache() {

        InputStream inputStream = null;

        Properties properties = new Properties();

        try {
            inputStream = getClass().getResourceAsStream(propertiesPath);
            if (inputStream == null) {
                //throw new NullPointerException("could not find action file " + propertiesPath);
                // actually, there is nothing to load, this is not an error
            } else {
                log.info("load " + propertiesPath);
                properties.load(inputStream);
            }
        } catch (IOException e) {
            String message = _("jaxx.error.load.actions.file", e.getMessage());
            log.warn(message);
            throw new RuntimeException(message);
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    log.warn(_("jaxx.error.close.actions.file", e.getMessage()));
                    //throw new RuntimeException(_("jaxx.error.load.actions.file", e.getMessage()));
                }
            }
        }

        Map<String, Class<? extends A>> cache = new TreeMap<String, Class<? extends A>>();
        int prefix = ACTION_KEY_PREFIX.length();
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            String key = entry.getKey() + "";
            String qfn = entry.getValue() + "";
            try {
                Class<? extends A> implCass;
                implCass = (Class<? extends A>) Class.forName(qfn);
                String actionKey = key.substring(prefix);
                if (actionKey.startsWith(":")) {
                    // this is a RuntimeActionNameProvider
                    Class<ActionNameProvider> klazz = (Class<ActionNameProvider>) Class.forName(actionKey.substring(1));
                    for (String s : klazz.newInstance().getActionCommands()) {
                        log.debug("found action <" + s + " : " + implCass + ">");
                        cache.put(s, implCass);
                    }
                    continue;
                }
                log.debug("found action <" + actionKey + " : " + implCass + ">");
                cache.put(actionKey, implCass);
            } catch (Exception e) {
                throw new RuntimeException(_("jaxx.error.load.actions.class", key, qfn), e);
            }
        }

        return cache;
    }
}
