/*
 * #%L
 * JAXX :: Runtime
 * 
 * $Id: ActionWorker.java 2225 2011-02-19 20:15:00Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.4/jaxx-runtime/src/main/java/jaxx/runtime/swing/application/ActionWorker.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.swing.application;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;

import javax.swing.SwingWorker;

/**
 * Action worker to execute a incoming action.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.1
 */
public class ActionWorker<R, P> extends SwingWorker<R, P> {

    /** Logger */
    private static final Log log = LogFactory.getLog(ActionWorker.class);

    protected final String actionLabel;

    protected Runnable target;

    protected ActionStatus status;

    protected Exception error;

    protected long startTime;

    protected long endTime;

    public ActionWorker(String actionLabel) {
        this.actionLabel = actionLabel;
    }

    public ActionWorker(String actionLabel, Runnable target) {
        this.target = target;
        this.actionLabel = actionLabel;
    }

    public Runnable getTarget() {
        return target;
    }

    public void setTarget(Runnable target) {
        this.target = target;
    }

    @Override
    protected R doInBackground() throws Exception {
        startTime = System.nanoTime();
        if (log.isDebugEnabled()) {
            log.debug("Action [" + getActionLabel() + "] is starting...");
        }
        try {
            getTarget().run();
        } catch (Exception e) {
            error = e;
        } finally {
            if (log.isDebugEnabled()) {
                log.debug("Action [" + getActionLabel() + "] is ending...");
            }
        }
        return null;
    }

    public boolean isFailed() {
        return (isDone() || isCancelled()) && error != null;
    }

    public Exception getError() {
        return error;
    }

    public ActionStatus getStatus() {
        return status;
    }

    public String getActionLabel() {
        return actionLabel;
    }

    @Override
    protected void done() {
        super.done();
        endTime = System.nanoTime();
        if (error != null) {
            status = ActionStatus.FAIL;
        } else if (isCancelled()) {
            status = ActionStatus.CANCEL;
        } else {
            status = ActionStatus.OK;
        }
        if (log.isDebugEnabled()) {
            log.debug("Action [" + getActionLabel() + "] ends with status : " + status + " in " + getTime());
        }
    }

    public String getTime() {
        return StringUtil.convertTime(endTime - startTime);
    }

    public long getStartTime() {
        return startTime;
    }

    public long getEndTime() {
        return endTime;
    }

    /** State of a running action */
    public enum ActionStatus {
        OK,
        CANCEL,
        FAIL
    }

}
