package jaxx.reflect;

import jaxx.runtime.JAXXObjectDescriptor;

import java.util.Arrays;

/**
 * Mirrors the class <code>java.lang.Class</code>.  JAXX uses <code>ClassDescriptor</code> instead of <code>Class</code>
 * almost everywhere so that it can handle circular dependencies (there can't be a <code>Class</code> object for an uncompiled
 * JAXX or Java source file, and a compiler must be allow references to symbols in uncompiled source files in order to handle
 * circular dependencies).
 */
public abstract class ClassDescriptor {
    private String name;
    private String packageName;
    private String superclass;
    private String[] interfaces;
    private boolean isInterface;
    private boolean isArray;
    private String componentType;
    private JAXXObjectDescriptor jaxxObjectDescriptor;
    private ClassLoader classLoader;
    private MethodDescriptor[] methodDescriptors;
    private FieldDescriptor[] fieldDescriptors;


    ClassDescriptor(String name, String packageName, String superclass, String[] interfaces, boolean isInterface,
                    boolean isArray, String componentType, JAXXObjectDescriptor jaxxObjectDescriptor,
                    ClassLoader classLoader, MethodDescriptor[] methodDescriptors, FieldDescriptor[] fieldDescriptors) {
        this.name = name;
        this.packageName = packageName;
        this.superclass = superclass;
        this.interfaces = interfaces;
        this.isInterface = isInterface;
        this.isArray = isArray;
        this.componentType = componentType;
        this.jaxxObjectDescriptor = jaxxObjectDescriptor;
        this.classLoader = classLoader;
        this.methodDescriptors = methodDescriptors;
        this.fieldDescriptors = fieldDescriptors;
    }


    public String getName() {
        return name;
    }


    public String getPackageName() {
        return packageName;
    }


    public ClassDescriptor getSuperclass() {
        try {
            return superclass != null ? ClassDescriptorLoader.getClassDescriptor(superclass, getClassLoader()) : null;
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }


    public ClassDescriptor[] getInterfaces() {
        try {
            ClassDescriptor[] result = new ClassDescriptor[interfaces.length];
            for (int i = 0; i < result.length; i++) {
                result[i] = ClassDescriptorLoader.getClassDescriptor(interfaces[i], getClassLoader());
            }
            return result;
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }


    public boolean isInterface() {
        return isInterface;
    }


    public boolean isArray() {
        return isArray;
    }


    public ClassDescriptor getComponentType() {
        try {
            return componentType != null ? ClassDescriptorLoader.getClassDescriptor(componentType, getClassLoader()) : null;
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }


    public ClassLoader getClassLoader() {
        return classLoader;
    }


    public MethodDescriptor[] getMethodDescriptors() {
        return methodDescriptors;
    }


    public MethodDescriptor getMethodDescriptor(String name, ClassDescriptor... parameterTypes) throws NoSuchMethodException {
        for (MethodDescriptor methodDescriptor : methodDescriptors) {
            if (methodDescriptor.getName().equals(name)
                    && methodDescriptor.getParameterTypes().length == parameterTypes.length
                    && Arrays.equals(methodDescriptor.getParameterTypes(), parameterTypes)) {
                return methodDescriptor;
            }
        }
        throw new NoSuchMethodException("Could not find method " + name + "(" + Arrays.asList(parameterTypes) + ") in " + getName());
    }


    public abstract MethodDescriptor getDeclaredMethodDescriptor(String name, ClassDescriptor... parameterTypes) throws NoSuchMethodException;


    public FieldDescriptor[] getFieldDescriptors() {
        return fieldDescriptors;
    }


    public FieldDescriptor getFieldDescriptor(String name) throws NoSuchFieldException {
        for (FieldDescriptor fieldDescriptor : fieldDescriptors) {
            if (fieldDescriptor.getName().equals(name)) {
                return fieldDescriptor;
            }
        }
        throw new NoSuchFieldException("Could not find field " + name + " in " + getName());
    }


    public abstract FieldDescriptor getDeclaredFieldDescriptor(String name) throws NoSuchFieldException;


    public JAXXObjectDescriptor getJAXXObjectDescriptor() {
        return jaxxObjectDescriptor;
    }


    public boolean isAssignableFrom(ClassDescriptor descriptor) {
        while (descriptor != null) {
            if (descriptor == this) {
                return true;
            }
            ClassDescriptor[] interfaces = descriptor.getInterfaces();
            for (ClassDescriptor anInterface : interfaces) {
                if (anInterface == this) {
                    return true;
                }
            }
            descriptor = descriptor.getSuperclass();
        }
        return false;
    }

    @Override
    public String toString() {
        return "ClassDescriptor[" + getName() + "]";
    }
}