package jaxx.compiler;

import jaxx.CompilerException;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.spi.Initializer;
import jaxx.tags.DefaultObjectHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;

/** @author chemit */
public class JAXXCompilerLaunchor {

    /** log */
    protected static final Log log = LogFactory.getLog(JAXXCompilerLaunchor.class);

    protected enum LifeCycle {

        init,// state before compilation
        compile_first_pass, // state when first pass of compilation
        compile_second_pass, // state when second pass of compilation
        stylesheet_pass, // state when applygin stylesheet phase after compilation
        generate_pass, // state when generation phase
        profile_pass // state when profile
    }
    /** shared instance of unique launchor at a givne time. */
    protected static JAXXCompilerLaunchor singleton;

    /**
     * Create a new empty launchor and set it as current launchor accessible via method {@link #get()}
     *
     * @return the new instanciated launchor
     */
    public static synchronized JAXXCompilerLaunchor newLaunchor() {
        return newLaunchor((File[]) null, null, null);
    }

    /**
     * Create a new launchor and set it as current launchor accessible via method {@link #get()}.
     * <p/>
     * The launchor will be prepared to compile a set of files, expressed as paths relative to a base directory.
     * The class names of the compiled files are derived from the relative path strings
     * (e.g. "example/Foo.jaxx" compiles into a class named "example.Foo").
     *
     * @param base          the directory against which to resolve relative paths
     * @param relativePaths a list of relative paths to .jaxx files being compiled
     * @param options       the compiler options to use
     * @return the new instanciated launchor
     */
    public static synchronized JAXXCompilerLaunchor newLaunchor(File base, String[] relativePaths, CompilerOptions options) {
        File[] files = new File[relativePaths.length];
        String[] classNames = new String[relativePaths.length];
        for (int i = 0; i < files.length; i++) {
            files[i] = new File(base, relativePaths[i]);
            classNames[i] = relativePaths[i].substring(0, relativePaths[i].lastIndexOf("."));
            classNames[i] = classNames[i].replace(File.separatorChar, '.');
            classNames[i] = classNames[i].replace('/', '.');
            classNames[i] = classNames[i].replace('\\', '.');
            classNames[i] = classNames[i].replace(':', '.');
        }
        return newLaunchor(files, classNames, options);
    }

    /**
     * Create a new launchor and set it as current launchor accessible via method {@link #get()}.
     * <p/>
     * The launchor will be prepared to compile a set of files, with the class names specified explicitly.
     * The class compiled from files[i] will be named classNames[i].
     *
     * @param files      the .jaxx files to compile
     * @param classNames the names of the classes being compiled
     * @param options    the compiler options to use
     * @return the new instanciated launchor
     */
    public static synchronized JAXXCompilerLaunchor newLaunchor(File[] files, String[] classNames, CompilerOptions options) {
        if (singleton != null) {
            singleton.reset();
        }
        singleton = new JAXXCompilerLaunchor(files, classNames, options);
        return singleton;
    }

    /**
     * @return the current launchor
     * @throws NullPointerException if no launchor was registred via a <code>newLaunchor-like</code> method.
     */
    public static JAXXCompilerLaunchor get() throws NullPointerException {
        if (singleton == null) {
            throw new NullPointerException("no launchor was registred via newLaunchor method");
        }
        return singleton;
    }

    /** @return <code> if there is a launchor registred, <code>false</code> otherwise. */
    public static boolean isRegistred() {
        return singleton != null;
    }

    /**
     * Load the {@link jaxx.spi.Initializer} services found via the{@link ServiceLoader} mecanism.
     *
     * @param verbose <ocde>true</code> to print initializers
     */
    public static void loadLibraries(boolean verbose) {
        //BeanInfoUtil.reset();
        ClassLoader classloader = Thread.currentThread().getContextClassLoader();
        if (verbose) {
            log.info("with cl " + classloader);
        }
        ServiceLoader<Initializer> loader = ServiceLoader.load(Initializer.class, classloader);
        for (Initializer initializer : loader) {
            if (verbose) {
                log.info("load initializer " + initializer);
            }
            initializer.initialize();
        }
    }
    /** options of the launchor and underlines compilers */
    protected CompilerOptions options;
    /** original list of files to compile */
    protected final File[] files;
    /** original list of classes to compile */
    protected final String[] classNames;
    /** Files to be treated while compilation. */
    protected List<File> jaxxFiles = new ArrayList<File>();
    /** Class names corresponding to the files in the jaxxFiles list. */
    protected List<String> jaxxFileClassNames = new ArrayList<String>();
    /** Maps the names of classes being compiled to the compiler instance handling the compilation. */
    protected Map<String, JAXXCompiler> compilers = new HashMap<String, JAXXCompiler>();
    /** Maps the names of classes being compiled to their symbol tables (created after the first compiler pass). */
    protected Map<File, SymbolTable> symbolTables = new HashMap<File, SymbolTable>();
    protected LifeCycle currentPass;
    protected int errorCount;
    protected int warningCount;
    protected int compilerCount;
    protected JAXXProfile profiler;

    protected JAXXCompilerLaunchor(File[] files, String[] classNames, CompilerOptions options) {
        this.options = options == null ? new CompilerOptions() : options;
        this.files = files;
        this.classNames = classNames;
        if (this.options.isVerbose()) {
            log.info("files : " + Arrays.toString(files));
        }
        if (this.options.isProfile()) {
            profiler = new JAXXProfile();
        }
    }

    public void init() {
        // forces static initializer to run if it hasn't yet
    }

    /** Resets all state in preparation for a new compilation session. */
    protected void reset() {
        errorCount = warningCount = 0;
        jaxxFiles.clear();
        jaxxFileClassNames.clear();
        symbolTables.clear();
        compilers.clear();
        if (profiler != null) {
            profiler.clear();
        }
    }

    public String getVersion() {
        return "1.0.4";
    }

    /**
     * Creates a dummy Compiler for use in unit testing.
     *
     * @return the compiler
     */
    public static JAXXCompiler createDummyCompiler() {
        return createDummyCompiler(JAXXCompiler.class.getClassLoader());
    }

    /**
     * Creates a dummy Compiler for use in unit testing.
     *
     * @param classLoader class loader to use
     * @return the compiler
     */
    public static JAXXCompiler createDummyCompiler(ClassLoader classLoader) {
        return new JAXXCompiler(classLoader, new DefaultObjectHandler(ClassDescriptorLoader.getClassDescriptor(Object.class))) {
        };
    }

    /**
     * @param className the name of the class to use
     * @return the compiler instance which is processing the specified JAXX class.  Each class is compiled by a
     *         different compiler instance.
     */
    public JAXXCompiler getJAXXCompiler(String className) {
        return compilers != null ? compilers.get(className) : null;
    }

    /**
     * @param className the name of the class to use
     * @return the symbol table for the specified JAXX class.  Must be called during the second compiler pass.
     *         Returns <code>null</code> if no such symbol table could be found.
     */
    public SymbolTable getSymbolTable(String className) {
        JAXXCompiler compiler = getJAXXCompiler(className);
        if (compiler == null) {
            return null;
        }
        return compiler.getSymbolTable();
    }

    /**
     * Returns the system line separator string.
     *
     * @return the string used to separate lines
     */
    public String getLineSeparator() {
        return System.getProperty("line.separator", "\n");
    }

    /**
     * Compiled a set of files.
     *
     * @return <code>true</code> if compilation succeeds, <code>false</code> otherwise
     */
    public synchronized boolean compile() {
        //reset(); // just to be safe...
        compilerCount = 0;
        jaxxFiles.addAll(Arrays.asList(files));
        jaxxFileClassNames.addAll(Arrays.asList(classNames));
        try {
            boolean success = true;

            // pass 1
            if (!nextStep(LifeCycle.compile_first_pass, success)) {
                return false;
            }
            boolean compiled;
            do {
                compiled = false;
                assert jaxxFiles.size() == jaxxFileClassNames.size();
                java.util.Iterator<File> filesIterator = new ArrayList<File>(jaxxFiles).iterator(); // clone it so it can safely be modified while we're iterating
                java.util.Iterator<String> classNamesIterator = new ArrayList<String>(jaxxFileClassNames).iterator();
                while (filesIterator.hasNext()) {
                    File file = filesIterator.next();
                    String className = classNamesIterator.next();
                    if (log.isDebugEnabled()) {
                        log.debug("compile first pass for " + className);
                    }
                    if (symbolTables.get(file) == null) {
                        compiled = true;
                        if (compilers.containsKey(className)) {
                            throw new CompilerException("Internal error: " + className + " is already being compiled, attempting to compile it again");
                        }

                        File destDir = options.getTargetDirectory();
                        if (destDir != null) {
                            int dotPos = className.lastIndexOf(".");
                            if (dotPos != -1) {
                                destDir = new File(destDir, className.substring(0, dotPos).replace('.', File.separatorChar));
                            }
                            if (!destDir.exists() && !destDir.mkdirs()) {
                                log.warn("could not create directory " + destDir);
                                continue;
                            }
                        } else {
                            //destDir = file.getParentFile();
                        }
                        JAXXCompiler compiler = newCompiler(file.getParentFile(), file, className);
                        addProfileTime(compiler, currentPass.name() + "_start");
                        compilers.put(className, compiler);
                        compiler.compileFirstPass();
                        addProfileTime(compiler, currentPass.name() + "_end");
                        assert !symbolTables.values().contains(compiler.getSymbolTable()) : "symbolTable is already registered";
                        symbolTables.put(file, compiler.getSymbolTable());
                        if (compiler.isFailed()) {
                            success = false;
                        }
                    }
                }

            } while (compiled);

            // pass 2
            if (!nextStep(LifeCycle.compile_second_pass, success)) {
                return false;
            }

            assert jaxxFiles.size() == jaxxFileClassNames.size();
            List<File> jaxxFilesClone = new ArrayList<File>(jaxxFiles);
            for (String className : jaxxFileClassNames) {
                JAXXCompiler compiler = getCompiler(className, "Internal error: could not find compiler for " + className + " during second pass");
                addProfileTime(compiler, currentPass.name() + "_start");
                if (log.isDebugEnabled()) {
                    log.debug("runInitializers for " + className);
                }
                if (!compiler.isFailed()) {
                    compiler.runInitializers();
                }
                if (log.isDebugEnabled()) {
                    log.debug("compile second pass for " + className);
                }
                compiler.compileSecondPass();
                addProfileTime(compiler, currentPass.name() + "_end");
                if (log.isDebugEnabled()) {
                    log.debug("done with result [" + !compiler.isFailed() + "] for " + className);
                }
                if (compiler.isFailed()) {
                    success = false;
                }
            }
            if (!jaxxFilesClone.equals(jaxxFiles)) {
                throw new AssertionError("Internal error: compilation set altered during pass 2 (was " + jaxxFilesClone + ", modified to " + jaxxFiles + ")");
            }

            // stylesheet application
            if (!nextStep(LifeCycle.stylesheet_pass, success)) {
                return false;
            }
            assert jaxxFiles.size() == jaxxFileClassNames.size();
            for (String className : jaxxFileClassNames) {
                JAXXCompiler compiler = getCompiler(className, "Internal error: could not find compiler for " + className + " during stylesheet application");
                addProfileTime(compiler, currentPass.name() + "_start");
                compiler.applyStylesheets();
                addProfileTime(compiler, currentPass.name() + "_end");
                if (compiler.isFailed()) {
                    success = false;
                }
            }

            // code generation
            if (!nextStep(LifeCycle.generate_pass, success)) {
                return false;
            }
            assert jaxxFiles.size() == jaxxFileClassNames.size();
            List<Generator> generators = new ArrayList<Generator>();
            for (Generator generator : ServiceLoader.load(Generator.class)) {
                generators.add(generator);
            }
            for (String className : jaxxFileClassNames) {
                JAXXCompiler compiler = getCompiler(className, "Internal error: could not find compiler for " + className + " during code generation");
                addProfileTime(compiler, currentPass.name() + "_start");
                compiler.generateCode(generators);
                addProfileTime(compiler, currentPass.name() + "_end");
                //compiler.generateCode();
                if (compiler.isFailed()) {
                    success = false;
                }
            }

            if (options.isProfile()) {
                // profile pass (only if succes compile)
                if (!nextStep(LifeCycle.profile_pass, success)) {
                    return false;
                }
                StringBuilder buffer = profiler.computeProfileReport();
                log.info(buffer.toString());
            }

            return report(success);

            //FIXME : deal better  the exception treatment...
        } catch (CompilerException e) {
            System.err.println(e.getMessage());
            e.printStackTrace();
            return false;
        } catch (Throwable e) {
            e.printStackTrace();
            return false;
        } finally {
            compilerCount = compilers.size();
            //TC - 20081018 only reset when no error was detected
            if (options.isResetAfterCompile() && errorCount == 0) {
                reset();
            }
        }
    }

    public int getCompilerCount() {
        return compilerCount;
    }

    protected JAXXCompiler getCompiler(String className, String message) {
        JAXXCompiler compiler = compilers.get(className);
        if (compiler == null) {
            throw new CompilerException(message);
        }
        return compiler;
    }

    protected boolean nextStep(LifeCycle nextCycle, boolean success) {
        if (!success) {
            return report(false);
        }
        currentPass = nextCycle;
        return true;
    }

    protected boolean report(boolean success) {
        if (warningCount == 1) {
            System.err.println("1 warning");
        } else if (warningCount > 0) {
            System.err.println(warningCount + " warnings");
        }
        if (errorCount == 1) {
            System.err.println("1 error");
        } else if (errorCount > 0) {
            System.err.println(errorCount + " errors");
        }
        return success;
    }

    protected JAXXCompiler newCompiler(File parentFile, File file, String className) throws InvocationTargetException, IllegalAccessException, InstantiationException, NoSuchMethodException {
        Constructor<? extends JAXXCompiler> cons = options.getCompilerClass().getConstructor(File.class, File.class, String.class, CompilerOptions.class);
        return cons.newInstance(parentFile, file, className, options);
    }

    public static void addProfileTime(JAXXCompiler compiler, String key) {
        JAXXProfile p = JAXXCompilerLaunchor.get().profiler;
        if (p != null) {
            p.addTime(compiler, key);
        }
    }

    protected static void showUsage() {
        System.out.println("Usage: jaxxc <options> <source files>");
        System.out.println();
        System.out.println("Source files must end in extension .jaxx");
        System.out.println("Use JAXX_OPTS environment variable to pass arguments to Java runtime");
        System.out.println();
        System.out.println("Supported options include:");
        System.out.println("  -classpath <paths>  paths to search for user classes");
        System.out.println("  -cp <paths>         same as -classpath");
        System.out.println("  -d <directory>      target directory for generated class files");
        System.out.println("  -java or -j         produce .java files, but do not compile them");
        System.out.println("  -keep or -k         preserve generated .java files after compilation");
        System.out.println("  -optimize or -o     optimize during compilation");
        System.out.println("  -version            display version information");
        System.out.println();
        System.out.println("See http://www.jaxxframework.org/ for full documentation.");
    }
}
