/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: ParserJavaMojo.java 1848 2011-01-20 07:33:24Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.2/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/parser/impl/ParserJavaMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n.plugin.parser.impl;

import org.apache.maven.plugin.logging.Log;
import org.nuiton.i18n.plugin.parser.AbstractFileParser;
import org.nuiton.i18n.plugin.parser.AbstractI18nParserMojo;
import org.nuiton.i18n.plugin.parser.FileParser;
import org.nuiton.i18n.plugin.parser.I18nSourceEntry;
import org.nuiton.i18n.plugin.parser.ParserException;
import org.nuiton.i18n.plugin.parser.SourceEntry;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.FileUpdaterHelper;
import org.nuiton.io.SortedProperties;
import org.nuiton.processor.filters.DefaultFilter;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * To parse java files to detect new i18n keys.
 * <p/>
 * <b>Note: </b> this goal must always be invoked before the {@code process-resources}
 * phase, otherwise all files will be considered as uptodate.
 *
 * @author jruchaud <ruchaud@codelutin.com>
 * @author tchemit <tchemit@codelutin.com>
 * @goal parserJava
 * @phase generate-resources
 */
public class ParserJavaMojo extends AbstractI18nParserMojo {

    /**
     * Root directory of the default entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/java"
     */
    protected File defaultBasedir;

    /**
     * Default included files to process (ant-like expression).
     *
     * @parameter expression="${i18n.defaultIncludes}" default-value="**\/*.java"
     */
    protected String defaultIncludes;

    /**
     * Defines the file name of the getter where to put detected i18n keys
     * while getter phase.
     *
     * @parameter expression="${i18n.outputGetter}" default-value="java.getter"
     * @since 2.0
     */
    protected String outputGetter;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return FileUpdaterHelper.newJavaFileUpdater(entry.getBasedir(), cp);
    }

    @Override
    @Deprecated
    protected String getKeyModifierStart() {
        return "_\\(\\s*\"";
    }

    @Override
    @Deprecated
    protected String getKeyModifierEnd() {
        return "\"\\s*(\\)|,|\\+|$)";
    }

    @Override
    protected String getOutGetter() {
        return outputGetter;
    }

    @Override
    public FileParser newFileParser() {

        return new JavaFileParser(getLog(),
                                  getEncoding(),
                                  oldParser,
                                  isShowTouchedFiles()
        );
    }

    protected static class JavaFileParser extends AbstractFileParser {

        protected final I18nFilter filter;

        public JavaFileParser(Log log,
                              String encoding,
                              SortedProperties oldParser,
                              boolean showTouchedFiles) {
            super(log, encoding, oldParser, showTouchedFiles);
            filter = new I18nFilter(log);
        }

        @Override
        public void parseFile(File file) throws IOException {
            String line = null;
            LineNumberReader lnr = new LineNumberReader(new InputStreamReader(
                    new FileInputStream(file), getEncoding()));
            try {
                while ((line = lnr.readLine()) != null) {
                    parseLine(file, line);
                }
            } catch (Exception e) {
                if (line != null) {
                    getLog().error(
                            "could not parse line (" + lnr.getLineNumber() + ") '"
                            + line + "' of file " + file);
                }
                throw new ParserException(e);
            } finally {
                lnr.close();
            }
        }

        @Override
        public void parseLine(File file, String line) throws IOException {

            String keysSet = filter.parse(line);

            if (keysSet.equals(I18nFilter.EMPTY_STRING)) {
                // no key detected on this line
                return;
            }

            // one key found in file, so file is marked as touched
            setTouched(true);
            // Found a set of i18n Strings, split it.
            String[] keys = keysSet.split("=");
            for (String key : keys) {
                if (getLog().isDebugEnabled()) {
                    getLog().debug(file.getName() + " detected key = " + key);
                }
                registerKey(key);
            }
        }
    }

    /**
     * Pour filtrer les clefs i18n dans un fichier java.
     *
     * @since 2.1
     */
    public static class I18nFilter extends DefaultFilter {

        /** Instance logger */
        private final Log log;

        //        private String header = "_\\(\\s*\"";
        private String header = "^_\\(\\s*\\\"|[^l]_\\(\\s*\\\"|l_\\([^,]+\\s*,\\s*\\\"";

        private String footer = "\"\\s*(\\)|,|\\+|$)";

        private Pattern headerPattern = Pattern.compile(getHeader());

        private Pattern footerPattern = Pattern.compile(getFooter());

        private Matcher matcher;

        public I18nFilter(Log log) {
            this.log = log;
        }

        protected void setFooter(String footer) {
            this.footer = footer;
        }

        protected void setHeader(String header) {
            this.header = header;
        }

        @Override
        protected String getHeader() {
            return header;
        }

        @Override
        protected String getFooter() {
            return footer;
        }

        @Override
        public int getMatchIndexFor(String input, String sequence) {
            int index = NOT_FOUND;

            setMatcher(null);
            if (sequence.equals(getHeader())) {
                setMatcher(getHeaderPattern().matcher(input));
            } else if (sequence.equals(getFooter())) {
                setMatcher(getFooterPattern().matcher(input));
            }
            if (getMatcher() != null) {
                try {
                    getMatcher().find();
                    index = getMatcher().start();
                } catch (RuntimeException e) {
                    if (log.isDebugEnabled()) {
                        log.debug("Could not match with " + getMatcher() + " input " + input);
                    }
                }
            }

            return index;
        }

        @Override
        public int getMatchLengthFor(String sequence) {
            int length = NOT_FOUND;

            try {
                length = getMatcher().end() - getMatcher().start();
            } catch (RuntimeException e) {
                if (log.isDebugEnabled()) {
                    log.debug("Could not match with " + getMatcher() + " input " + sequence);
                }
            }

            return length;
        }

        /**
         * methode appele lorsqu'on a la chaine entiere entre le header et le
         * footer.
         *
         * @param ch la chaine trouve
         * @return ce qu'il faut ecrire dans le fichier de sortie
         */
        @Override
        protected String performInFilter(String ch) {
            return ch.replaceAll("\"\\s*\\+\\s*\"", "") + "=";
        }

        @Override
        public String performHeaderFooterFilter(String ch) {
            return ch.substring(ch.indexOf('"') + 1, ch.lastIndexOf('"'));
        }

        /**
         * methode appele lorsqu'on a la chaine entiere a l'exterieur du
         * header/footer
         *
         * @param ch la chaine trouve
         * @return ce qu'il faut ecrire dans le fichier de sortie
         */
        @Override
        protected String performOutFilter(String ch) {
            return EMPTY_STRING;
        }

        /** @return Returns the footerPattern. */
        protected Pattern getFooterPattern() {
            return footerPattern;
        }

        /** @return Returns the headerPattern. */
        protected Pattern getHeaderPattern() {
            return headerPattern;
        }

        /** @return Returns the matcher. */
        protected Matcher getMatcher() {
            return matcher;
        }

        /** @param matcher The matcher to set. */
        protected void setMatcher(Matcher matcher) {
            this.matcher = matcher;
        }
    }
}
