package org.nuiton.i18n.plugin.parser;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.io.SortedProperties;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;

/**
 * A thread to launch parser on files.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.2
 */
public class ParserThread extends Thread {

    /** Logger */
    private static final Log log = LogFactory.getLog(ParserThread.class);

    class ParserTask implements Runnable {

        protected final FileParser parser;

        protected final File file;

        protected long startingTime;

        protected long endingTime;

        ParserTask(FileParser parser, File file) {
            this.parser = parser;
            this.file = file;
        }

        @Override
        public void run() {
            startingTime = System.nanoTime();
            if (log.isDebugEnabled()) {
                log.debug("starting action for " + file);
            }
            try {

                parser.parseFile(file);

            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("could not parse file " + file, e);
                }
            } finally {

                if (log.isDebugEnabled()) {
                    log.debug("ending action for " + file);
                }
                endingTime = System.nanoTime();
            }
        }

        @Override
        public String toString() {
            return super.toString() + " - " + file;
        }

        protected File getFile() {
            return file;
        }

        protected long getDelay() {
            return endingTime - startingTime;
        }

        protected void destroy() {
            parser.destroy();
        }

        @Override
        protected void finalize() throws Throwable {
            super.finalize();
            destroy();
        }

        protected synchronized void registerResult(List<File> treatedFiles,
                                                   List<File> touchedFiles,
                                                   SortedProperties result) {

            try {
                treatedFiles.add(file);
                if (log.isDebugEnabled()) {
                    String delay = PluginHelper.convertTime(getDelay());
                    log.debug("[" + treatedFiles.size() + "] " + file +
                              " in " + delay);
                }

                if (parser.isTouched()) {

                    // mark file as touched
                    touchedFiles.add(file);

                    if (parser.isShowTouchedFiles()) {
                        parser.getLog().info("touch " + file);
                    }
                    if (log.isInfoEnabled()) {
                        String delay = PluginHelper.convertTime(getDelay());
                        log.info("[" + treatedFiles.size() + "] touchs " +
                                 file + " in " + delay);
                    }

                    // merge file result with
                    // merge result
                    result.putAll(parser.getResult());
                }
            } finally {
                // destroy runner
                destroy();
            }
        }
    }

    protected final BlockingQueue<ParserTask> tasks;

    /** list of files consumed */
    protected final List<File> treatedFiles;

    /** list of files touched (says having at least one i18n key) */
    protected final List<File> touchedFiles;

    /** the shared result where to merge atomic results */
    protected final SortedProperties result;

    /**
     * object used to lock method {@link #terminatesAndWaits()} until all tasks
     * are consumed.
     */
    private final Object lock = new Object();

    /** number of files registred to consume */
    protected int nbFiles;

    public ParserThread(SortedProperties result) {
        super(ParserThread.class.getSimpleName());
        this.result = result;
        tasks = new LinkedBlockingQueue<ParserTask>();
        touchedFiles = new ArrayList<File>();
        treatedFiles = new ArrayList<File>();
    }

    public List<File> getTreatedFiles() {
        return treatedFiles;
    }

    public List<File> getTouchedFiles() {
        return touchedFiles;
    }

    public void addFile(FileParser parser, File... files) {

        synchronized (tasks) {

            for (File f : files) {

                nbFiles++;
                if (log.isInfoEnabled()) {
                    log.info("[" + nbFiles + "] " + f);
                }
                tasks.offer(new ParserTask(parser, f));
            }
        }
    }

    public synchronized void terminatesAndWaits() {
        synchronized (tasks) {
            if (log.isDebugEnabled()) {
                log.debug("add terminate runner");
            }
            tasks.offer(new ParserTask(null, null));
        }

        // block until end of service

        if (log.isInfoEnabled()) {
            log.info("block until ends...");
        }

        synchronized (lock) {
            try {
                lock.wait();
            } catch (InterruptedException e) {
                if (log.isErrorEnabled()) {
                    log.error(e);
                }
            }
        }
        if (nbFiles != treatedFiles.size()) {
            throw new IllegalStateException(
                    "should have " + nbFiles + " files treated, but found " +
                    treatedFiles.size());
        }
        if (log.isDebugEnabled()) {
            log.debug("thread is terminated.");
        }
    }

    public void clear() {
        treatedFiles.clear();
        touchedFiles.clear();
        nbFiles = 0;
    }

    @Override
    public void run() {

        // creates thread factory
        ThreadFactory factory =
                new ThreadFactory() {
                    final ThreadFactory defaultFactory =
                            Executors.defaultThreadFactory();

                    public Thread newThread(Runnable r) {
                        Thread thread =
                                defaultFactory.newThread(r);
                        thread.setName(ParserThread.this + "-" + thread.getName());
                        return thread;
                    }
                };

        // create pool executor
        ExecutorService service = new ThreadPoolExecutor(
                8, 10,
                1L, TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                factory) {

            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                super.afterExecute(r, t);
                ParserTask i18n = (ParserTask) r;
                i18n.registerResult(treatedFiles, touchedFiles, result);
            }
        };

        while (true) {

            // waiting for a new file to treate
            ParserTask runner;
            try {
                runner = tasks.take();
            } catch (InterruptedException e) {
                if (log.isErrorEnabled()) {
                    log.error(e);
                }
                continue;
            }

            if (runner.getFile() == null) {
                // ask to quit the tread
                if (log.isInfoEnabled()) {
                    log.info("no more files to treate, wait termination...");
                }
                service.shutdown();
                break;
            }

            if (log.isDebugEnabled()) {
                log.debug("consume " + runner);
            }
            service.execute(runner);

            if (log.isDebugEnabled()) {
                log.debug("after consume " + runner);
            }
        }

        if (log.isInfoEnabled()) {
            log.info("will waits until ends of all " + nbFiles + " runner(s)!");
        }

        // waits end of service

        while (true) {

            if (service.isTerminated()) {
                if (log.isInfoEnabled()) {
                    log.info("all runners are terminated. [" + treatedFiles.size() + "]");
                }

                // release lock
                synchronized (lock) {
                    lock.notifyAll();
                }
                break;
            }

            if (log.isDebugEnabled()) {
                log.debug("waiting end...");
            }

            try {
                sleep(200);
            } catch (InterruptedException e) {
                if (log.isErrorEnabled()) {
                    log.error(e);
                }
            }
        }
    }

}
