/*
 * *##% Plugin maven pour i18n
 * Copyright (C) 2007 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */

package org.nuiton.i18n.plugin.parser.impl;

import org.apache.maven.project.MavenProject;
import org.nuiton.util.SourceEntry;
import org.nuiton.i18n.plugin.parser.AbstractI18nParser;
import org.nuiton.i18n.plugin.parser.ParserEvent;
import org.nuiton.i18n.plugin.parser.ParserException;
import org.nuiton.util.FileUpdater;
import org.nuiton.util.FileUpdaterHelper;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import org.nuiton.processor.filters.I18nFilter;

/**
 * Récupération des chaine a traduire depuis les fichiers java.
 *
 * @author julien
 * @goal parserJava
 * @phase generate-resources
 */
public class ParserJava extends AbstractI18nParser {

    /**
     * Source entries (src+includes+excludes) .
     *
     * @parameter expression="${i18n.defaultIncludes}" default-value="**\/*.java"
     */
    protected String defaultIncludes;

    /**
     * default src for an entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/java"
     */
    protected File defaultBasedir;

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @readonly
     */
    protected MavenProject project;

    /**
     * Repertoire sources des fichiers i18n.
     *
     * @parameter expression="${i18n.cp}" default-value="${basedir}/target/classes"
     * @required
     */
    protected File cp;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return new String[]{};
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return FileUpdaterHelper.newJavaFileUpdater(entry.getBasedir(), cp);
    }

    @Override
    protected String getKeyModifierStart() {
        return "_\\(\\s*\"";
    }

    @Override
    protected String getKeyModifierEnd() {
        return "\"\\s*(\\)|,|\\+|$)";
    }

    @Override
    protected String getOutGetter() {
        return "java.getter";
    }

    protected I18nFilter filter;

    @Override
    public void init() {
        super.init();
        filter = new I18nFilter();
    }

    @Override
    public void parseFile(File srcFile) {
        LineNumberReader lnr=null;
        String line=null;
        try {
            lnr = new LineNumberReader(new InputStreamReader(new FileInputStream(srcFile)));

            while (lnr.ready()) {
                line = lnr.readLine();
                parseLine(srcFile, line);
            }

        } catch (Exception e) {
            if (line!=null) {
                getLog().error("could not parse line "+line);
            }
            throw new ParserException(e);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.codelutin.i18n.plugin.core.Parser#parseLine(java.io.File, java.lang.String)
     */
    @Override
    public void parseLine(File srcFile, String line) {
        String keysSet = filter.parse(line);

        if (!keysSet.equals(I18nFilter.EMPTY_STRING)) {
            touchFile = true;
            // Found a set of i18n Strings, split it.
            String[] keys = keysSet.split("=");
            for (String key : keys) {
                String keyModified = key;
                for (ParserEvent event : events) {
                    event.eventChangeKey(key, !oldLanguage.containsKey(key));
                    keyModified = event.eventGetRealKey();
                }
                if (oldParser.containsKey(key)) {
                    result.put(keyModified, oldParser.get(key));
                } else {
                    result.put(keyModified, key);
                }
            }
        }
    }

}
