/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.generator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.nuiton.guix.model.GuixModelObject;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Abstract class of all the generator which generate java files
 *
 * @author kmorin
 *
 */
public abstract class JavaFileGenerator {

    /**
     * Generated methods
     */
    public enum Method {
        COMPONENTS_CREATION, COMPONENTS_SETTINGS, COMPONENTS_TREE, DATABINDING_INIT,
        BEFORE_CREATION, BEFORE_TREE, BEFORE_SETTINGS, BEFORE_BINDING, IN_THE_END
    }
    /** log */
    protected Log log = LogFactory.getLog(JavaFileGenerator.class);
    /** List of the class names to import */
    protected List<String> imports = new ArrayList<String>();
    /** List of the classes to generate */
    protected Map<String, JavaFile> classes = null;
    /** GuixModelObject which represents the class to generate */
    protected GuixModelObject gmo;
    /** Java file generated */
    protected JavaFile jf;

    /**
     * Constructor
     *
     * @param gmo GuixModelObject which represents the class to generate
     * @param classes List of the classes to generate
     */
    public JavaFileGenerator(GuixModelObject gmo, Map<String, JavaFile> classes) {
        this.gmo = gmo;
        this.classes = classes;
    }

    /**
     * Generates the JavaFile repesnting the class described by the GuixModelObject gmo
     *
     * @return the generated JavaFile
     */
    public abstract JavaFile generate();

    /**
     * Add imports to the generated file
     * 
     */
    protected abstract void addImports();

    /**
     * Fills and saves the generated file
     * @param out the file to record
     */
    public void saveFile(File out) {
        FileWriter fw = null;
        try {
            fw = new FileWriter(out);
            fw.write(jf.toString("\n"));
            fw.close();
        }
        catch (IOException ex) {
            if (log.isErrorEnabled()) {
                log.error(ex);
            }
        }
        finally {
            try {
                fw.close();
            }
            catch (IOException ex) {
                log.error(ex);
            }
        }
    }

    /**
     * Get the name of the generated method m
     *
     * @param m the generated method we would like to know the name
     * @return the name of the method m
     */
    protected String getMethodName(Method m) {
        if (m == Method.BEFORE_BINDING) {
            return "beforeBinding";
        }
        else if (m == Method.DATABINDING_INIT) {
            return "initDataBinding";
        }
        else if (m == Method.BEFORE_TREE) {
            return "beforeTree";
        }
        else if (m == Method.COMPONENTS_CREATION) {
            return "componentsCreation";
        }
        else if (m == Method.BEFORE_SETTINGS) {
            return "beforeSetting";
        }
        else if (m == Method.COMPONENTS_SETTINGS) {
            return "componentsSetting";
        }
        else if (m == Method.BEFORE_CREATION) {
            return "beforeCreation";
        }
        else if (m == Method.COMPONENTS_TREE) {
            return "componentsTree";
        }
        else if (m == Method.IN_THE_END) {
            return "inTheEnd";
        }
        else {
            return null;
        }
    }
}

