/**
 * *##% guix-compiler-swing
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.tags.swing;

import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.guix.model.AttributeDescriptor;
import org.nuiton.guix.model.GuixModelObject;

/**
 * Represents a table
 *
 * @author kmorin
 */
public class TableHandler extends PanelHandler {

    private static Log log = LogFactory.getLog(TableHandler.class);

    public static String DEFAULT_ANCHOR = "java.awt.GridBagConstraints.FIRST_LINE_START";

    protected enum VerticalAlignment {

        ALIGN_TOP, ALIGN_MIDDLE, ALIGN_BOTTOM
    }
    protected enum HorizontalAlignment {

        ALIGN_LEFT, ALIGN_CENTER, ALIGN_RIGHT, ALIGN_DEFAULT
    }
    public static final String DEFAULT_INSETS = "3, 3, 3, 3";
    public static final HorizontalAlignment DEFAULT_HORIZONTAL_ALIGNMENT = HorizontalAlignment.ALIGN_DEFAULT;
    public static final VerticalAlignment DEFAULT_VERTICAL_ALIGNMENT = VerticalAlignment.ALIGN_TOP;
    public static final String DEFAULT_FILL = "both";
    public static final Integer DEFAULT_IPADX = 1;
    public static final Integer DEFAULT_IPADY = 1;
    public static final Double DEFAULT_WEIGHTX = 0.0;
    public static final Double DEFAULT_WEIGHTY = 0.0;
    protected Double weightx = null;
    protected Double weighty = null;
    protected Integer ipadx = null;
    protected Integer ipady = null;
    protected HorizontalAlignment horizontalAlignment = null;
    protected VerticalAlignment verticalAlignment = null;
    protected String fill = null;
    protected String insets = null;

    /**
     * Constructor
     */
    public TableHandler() {
    }

    /**
     * Constructor
     *
     * @param table the GuixModelObject which represents the table
     */
    public TableHandler(GuixModelObject table) {

        for (AttributeDescriptor attr : table.getAttributeDescriptors()) {
            if (attr.getName().equalsIgnoreCase("weightx")) {
                weightx = Double.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("weighty")) {
                weighty = Double.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("ipadx")) {
                ipadx = Integer.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("ipady")) {
                ipady = Integer.valueOf(attr.getValue());
            }
            else if (attr.getName().equalsIgnoreCase("horizontalAlignment")) {
                horizontalAlignment = getHalign(attr.getValue());
                if (horizontalAlignment == null) {
                    if (log.isErrorEnabled()) {
                        log.error(table.getId() + "'s horizontalAlignment attribute has a wrong value : " + attr.getValue());
                    }
                    horizontalAlignment = DEFAULT_HORIZONTAL_ALIGNMENT;
                }
            }
            else if (attr.getName().equalsIgnoreCase("verticalAlignment")) {
                verticalAlignment = getValign(attr.getValue());
                if (verticalAlignment == null) {
                    if (log.isErrorEnabled()) {
                        log.error(table.getId() + "'s verticalAlignment attribute has a wrong value : " + attr.getValue());
                    }
                    verticalAlignment = DEFAULT_VERTICAL_ALIGNMENT;
                }
            }
            else if (attr.getName().equalsIgnoreCase("fill")) {
                fill = attr.getValue();                
            }
            else if (attr.getName().equalsIgnoreCase("insets")) {
                insets = attr.getValue();
            }
        }
    }

    public HorizontalAlignment getHorizontalAlignment() {
        return horizontalAlignment;
    }

    public void setHorizontalAlignment(HorizontalAlignment horizontalAlignment) {
        this.horizontalAlignment = horizontalAlignment;
    }

    public VerticalAlignment getVerticalAlignment() {
        return verticalAlignment;
    }

    public void setVerticalAlignment(VerticalAlignment verticalAlignment) {
        this.verticalAlignment = verticalAlignment;
    }

    public String getFill() {
        return fill;
    }

    public void setFill(String fill) {
        this.fill = fill;
    }

    public String getInsets() {
        return insets;
    }

    public void setInsets(String insets) {
        this.insets = insets;
    }

    public Integer getIpadx() {
        return ipadx;
    }

    public void setIpadx(Integer ipadx) {
        this.ipadx = ipadx;
    }

    public Integer getIpady() {
        return ipady;
    }

    public void setIpady(Integer ipady) {
        this.ipady = ipady;
    }

    public Double getWeightx() {
        return weightx;
    }

    public void setWeightx(Double weightx) {
        this.weightx = weightx;
    }

    public Double getWeighty() {
        return weighty;
    }

    public void setWeighty(Double weighty) {
        this.weighty = weighty;
    }

    public HorizontalAlignment getHalign(String halign) {
        if (halign.equalsIgnoreCase("left")) {
            return HorizontalAlignment.ALIGN_LEFT;
        }
        if (halign.equalsIgnoreCase("right")) {
            return HorizontalAlignment.ALIGN_RIGHT;
        }
        if (halign.equalsIgnoreCase("center")) {
            return HorizontalAlignment.ALIGN_CENTER;
        }
        return null;
    }

    public VerticalAlignment getValign(String valign) {
        if (valign.equalsIgnoreCase("top")) {
            return VerticalAlignment.ALIGN_TOP;
        }
        if (valign.equalsIgnoreCase("bottom")) {
            return VerticalAlignment.ALIGN_BOTTOM;
        }
        if (valign.equalsIgnoreCase("middle")) {
            return VerticalAlignment.ALIGN_MIDDLE;
        }
        return null;
    }
    
    /**
     * Creates a cell with the right column and generates the code to add it to the table
     *
     * @param parentId the table id
     * @param cl the CellHandler containing the infos
     * @param layout the matrix representing the free spaces in the table
     * @param iC the last column where a cell was added in the row
     * @param r the row of the cell
     * @return the code to add the cell in the table
     */
    public static String createCell(String parentId, CellHandler cl, List<List<Boolean>> layout, int iC, int r) {
        StringBuffer result = new StringBuffer();
        //is the column defined by the user ?
        boolean columnSetByUser = true;
        boolean rowSetByUser = true;
        //FIXME
        if (cl.getColumn() == -1) {
            cl.setColumn(iC);
            columnSetByUser = false;
        }
        else if (cl.getColumn() < iC) {
            if (log.isErrorEnabled()) {
                log.error("Error cell column number");
            }
        }
        if(cl.getRow() == -1) {
            cl.setRow(r);
            rowSetByUser = false;
        }
        else if(rowSetByUser && cl.getRow() < r) {
            if(log.isErrorEnabled()) {
                log.error("Error cell row number");
            }
        }

        //if the row has not been added yet to layout
        if (r >= layout.size()) {
            for (int i = layout.size(); i <= r; i++) {
                layout.add(new ArrayList<Boolean>());
            }
        }
        //is there enough place to put the cell at column iC ?
        boolean enoughPlace = false;
        while (!enoughPlace) {
            enoughPlace = true;
            //while the column iC is occupied, go to next column
            while (layout.get(r).size() > iC && layout.get(r).get(iC)) {
                if (columnSetByUser && cl.getColumn() == iC) {
                    if (log.isErrorEnabled()) {
                        log.error("This column is already occupied.");
                    }
                }
                iC++;
            }
            //if there is a free space
            if (layout.get(r).size() > iC) {
                int j = 0;
                //check if the n-1 next columns are free (n = number of columns of the cell)
                for (int i = 0; i < cl.getColumns(); i++) {
                    //if a column is occupied, there is not enough place to put the cell
                    if (layout.get(r).size() > iC + i && layout.get(r).get(iC + i)) {
                        enoughPlace = false;
                        // j = the last occupied column met
                        j = iC + i;
                    }
                }
                if (!enoughPlace) {
                    iC = j + 1;
                    //iC is now the column just after the last occupied column met
                }
            }
        }
        cl.setColumn(iC);
        //for each row that must contain the cell
        for (int i = 0; i < cl.getRows(); i++) {
            //if the row has not been created in the layout, add it
            if (layout.size() <= r + i) {
                for (int j = layout.size(); j <= r + i; j++) {
                    layout.add(new ArrayList<Boolean>());
                }
            }
            //for each column that must contain the cell
            for (int j = 0; j < cl.getColumns(); j++) {
                //if the column has not been created yet for the row in the layout
                if (layout.get(r + i).size() < iC + j) {
                    //add false to all the column between the last column and the right column
                    for (int k = layout.get(r + i).size(); k < iC + j; k++) {
                        layout.get(r + i).add(false);
                    }
                }
                //if the column exists, replace false by true
                if((iC + j) < layout.get(r + i).size())
                    layout.get(r + i).set(iC + j, true);
                //else add true
                else
                    layout.get(r + i).add(true);
            }
        }
        String anchor = getAnchor(cl.getHorizontalAlignment(), cl.getVerticalAlignment());
        //generates the code
        result.append(parentId).append(".add(")
                .append(cl.getId()).append(", new java.awt.GridBagConstraints (")
                .append(cl.getColumn()).append(",")
                .append(cl.getRow()).append(",")
                .append(cl.getColumns()).append(",")
                .append(cl.getRows()).append(",")
                .append(cl.getWeightx()).append(",")
                .append(cl.getWeighty()).append(",")
                .append(anchor != null ? anchor : DEFAULT_ANCHOR).append(",")
                .append("GridBagConstraints.").append(cl.getFill().toUpperCase()).append(", new java.awt.Insets(")
                .append(cl.getInsets()).append("),")
                .append(cl.getIpadx()).append(",")
                .append(cl.getIpady()).append("));\n");

        return result.toString();
    }

    private static String getAnchor(HorizontalAlignment halign, VerticalAlignment valign) {
        if(halign == HorizontalAlignment.ALIGN_LEFT) {
            if(valign == VerticalAlignment.ALIGN_TOP) {
                return "java.awt.GridBagConstraints.FIRST_LINE_START";
            }
            if(valign == VerticalAlignment.ALIGN_MIDDLE) {
                return "java.awt.GridBagConstraints.PAGE_START";
            }
            if(valign == VerticalAlignment.ALIGN_BOTTOM) {
                return "java.awt.GridBagConstraints.FIRST_LINE_END";
            }
            return null;
        }
        if(halign == HorizontalAlignment.ALIGN_CENTER) {
            if(valign == VerticalAlignment.ALIGN_TOP) {
                return "java.awt.GridBagConstraints.LINE_START";
            }
            if(valign == VerticalAlignment.ALIGN_MIDDLE) {
                return "java.awt.GridBagConstraints.CENTER";
            }
            if(valign == VerticalAlignment.ALIGN_BOTTOM) {
                return "java.awt.GridBagConstraints.LINE_END";
            }
            return null;
        }
        if(halign == HorizontalAlignment.ALIGN_RIGHT) {
            if(valign == VerticalAlignment.ALIGN_TOP) {
                return "java.awt.GridBagConstraints.LAST_LINE_START";
            }
            if(valign == VerticalAlignment.ALIGN_MIDDLE) {
                return "java.awt.GridBagConstraints.PAGE_END";
            }
            if(valign == VerticalAlignment.ALIGN_BOTTOM) {
                return "java.awt.GridBagConstraints.LAST_LINE_END";
            }
            return null;
        }
        return null;
    }

}
