/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.test.common;

import com.google.common.annotations.VisibleForTesting;
import io.quarkus.test.common.IntegrationTestStartedNotifier;
import io.quarkus.test.common.ListeningAddress;
import io.quarkus.test.common.http.TestHTTPResourceManager;
import io.smallrye.common.os.OS;
import io.smallrye.config.SmallRyeConfig;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.microprofile.config.ConfigProvider;

public final class LauncherUtil {
    public static final int LOG_CHECK_INTERVAL = 50;

    private LauncherUtil() {
    }

    static String generateTestUrl() {
        SmallRyeConfig config = (SmallRyeConfig)ConfigProvider.getConfig().unwrap(SmallRyeConfig.class);
        String host = TestHTTPResourceManager.host(config, "quarkus.http.host");
        String rootPath = TestHTTPResourceManager.rootPath(config, new String[0]);
        if (rootPath.endsWith("/")) {
            rootPath = rootPath.substring(0, rootPath.length() - 1);
        }
        return "http://" + host + ":${quarkus.http.test-port:8081}" + rootPath;
    }

    static Process launchProcessAndDrainIO(List<String> args, Map<String, String> env) throws IOException {
        ProcessBuilder pb = new ProcessBuilder(args).redirectInput(ProcessBuilder.Redirect.INHERIT);
        pb.environment().putAll(env);
        Process process = pb.start();
        new Thread(new ProcessReader(process)).start();
        return process;
    }

    static Process launchProcess(List<String> args, Map<String, String> env) throws IOException {
        ProcessBuilder pb = new ProcessBuilder(args);
        pb.environment().putAll(env);
        return pb.start();
    }

    static Optional<ListeningAddress> waitForCapturedListeningData(Process quarkusProcess, Path logFile, long waitTimeSeconds) {
        LauncherUtil.ensureProcessIsAlive(quarkusProcess);
        CountDownLatch signal = new CountDownLatch(1);
        AtomicReference<ListeningAddress> resultReference = new AtomicReference<ListeningAddress>();
        CaptureListeningDataReader captureListeningDataReader = new CaptureListeningDataReader(logFile, Duration.ofSeconds(waitTimeSeconds), signal, resultReference);
        new Thread((Runnable)captureListeningDataReader, "capture-listening-data").start();
        try {
            signal.await(waitTimeSeconds + 2L, TimeUnit.SECONDS);
            ListeningAddress result = resultReference.get();
            if (result != null) {
                return result.port() != null && result.protocol() != null ? Optional.of(result) : Optional.empty();
            }
            LauncherUtil.destroyProcess(quarkusProcess);
            throw new IllegalStateException("Unable to determine the status of the running process. See the above logs for details");
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while waiting to capture listening process port and protocol");
        }
    }

    private static void ensureProcessIsAlive(Process quarkusProcess) {
        try {
            Thread.sleep(100L);
        }
        catch (InterruptedException ignored) {
            throw new RuntimeException("Interrupted while waiting to determine the status of process '" + quarkusProcess.pid() + "'.");
        }
        if (!quarkusProcess.isAlive()) {
            int exit = quarkusProcess.exitValue();
            String message = "Unable to successfully launch process '" + quarkusProcess.pid() + "'. Exit code is: '" + exit + "'.";
            if (OS.current() == OS.MAC && exit == 126) {
                message = message + System.lineSeparator() + "This may be caused by building the native binary in a Linux container while the host is macOS.";
            }
            throw new RuntimeException(message);
        }
    }

    static void destroyProcess(Process quarkusProcess) {
        quarkusProcess.destroy();
        int i = 0;
        while (i++ < 200) {
            try {
                Thread.sleep(50L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (quarkusProcess.isAlive()) continue;
        }
        if (quarkusProcess.isAlive()) {
            quarkusProcess.destroyForcibly();
        }
    }

    static void destroyProcess(ProcessHandle quarkusProcess) {
        try {
            CompletableFuture<ProcessHandle> exit = quarkusProcess.onExit();
            if (!quarkusProcess.destroy()) {
                quarkusProcess.destroyForcibly();
                return;
            }
            exit.get(500L, TimeUnit.MILLISECONDS);
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (quarkusProcess.isAlive()) {
            quarkusProcess.destroyForcibly();
        }
    }

    static void destroyProcess(Process process, boolean children) {
        if (!children) {
            LauncherUtil.destroyProcess(process);
            return;
        }
        process.descendants().forEach(p -> LauncherUtil.destroyProcess(p));
        LauncherUtil.destroyProcess(process);
    }

    static Function<IntegrationTestStartedNotifier.Context, IntegrationTestStartedNotifier.Result> createStartedFunction() {
        ArrayList<IntegrationTestStartedNotifier> startedNotifiers = new ArrayList<IntegrationTestStartedNotifier>();
        for (IntegrationTestStartedNotifier i : ServiceLoader.load(IntegrationTestStartedNotifier.class)) {
            startedNotifiers.add(i);
        }
        if (startedNotifiers.isEmpty()) {
            return null;
        }
        return ctx -> {
            for (IntegrationTestStartedNotifier startedNotifier : startedNotifiers) {
                IntegrationTestStartedNotifier.Result result = startedNotifier.check((IntegrationTestStartedNotifier.Context)ctx);
                if (!result.isStarted()) continue;
                return result;
            }
            return IntegrationTestStartedNotifier.Result.NotStarted.INSTANCE;
        };
    }

    static IntegrationTestStartedNotifier.Result waitForStartedFunction(Function<IntegrationTestStartedNotifier.Context, IntegrationTestStartedNotifier.Result> startedFunction, Process quarkusProcess, long waitTimeSeconds, Path logFile) {
        long bailout = System.currentTimeMillis() + waitTimeSeconds * 1000L;
        IntegrationTestStartedNotifier.Result result = null;
        SimpleContext context = new SimpleContext(logFile);
        while (System.currentTimeMillis() < bailout) {
            if (!quarkusProcess.isAlive()) {
                throw new RuntimeException("Failed to start target quarkus application, process has exited");
            }
            try {
                Thread.sleep(100L);
                result = startedFunction.apply(context);
                if (!result.isStarted()) continue;
                break;
            }
            catch (Exception exception) {
            }
        }
        if (result == null) {
            LauncherUtil.destroyProcess(quarkusProcess);
            throw new RuntimeException("Unable to start target quarkus application " + waitTimeSeconds + "s");
        }
        return result;
    }

    static class ProcessReader
    implements Runnable {
        private final Process process;
        private final ByteArrayOutputStream stdoutBuffer = new ByteArrayOutputStream();
        private final ByteArrayOutputStream stderrBuffer = new ByteArrayOutputStream();
        private final byte[] readBuffer = new byte[100];
        private final OutputStream outTo;
        private final OutputStream errTo;

        ProcessReader(Process process) {
            this(process, System.out, System.err);
        }

        @VisibleForTesting
        ProcessReader(Process process, OutputStream outTo, OutputStream errTo) {
            this.process = process;
            this.outTo = outTo;
            this.errTo = errTo;
        }

        @Override
        public void run() {
            try (InputStream stdout = this.process.getInputStream();
                 InputStream stderr = this.process.getErrorStream();){
                int exited = -1;
                while (true) {
                    exited = this.checkExited(exited);
                    boolean anyData = this.flush(stderr, this.stderrBuffer, this.errTo);
                    if (exited != -1 && !(anyData |= this.flush(stdout, this.stdoutBuffer, this.outTo))) {
                        if (exited == 0) break;
                        System.err.println("Process exited with non-zero status: " + exited);
                        break;
                    }
                    Thread.sleep(1L);
                }
                this.stderrBuffer.writeTo(this.errTo);
                this.stdoutBuffer.writeTo(this.outTo);
            }
            catch (IOException | InterruptedException exception) {
                // empty catch block
            }
        }

        @VisibleForTesting
        int checkExited(int exited) {
            if (exited == -1) {
                try {
                    exited = this.process.exitValue();
                }
                catch (IllegalThreadStateException illegalThreadStateException) {
                    // empty catch block
                }
            }
            return exited;
        }

        private boolean flush(InputStream processStream, ByteArrayOutputStream buffer, OutputStream to) throws IOException {
            int read;
            boolean any = false;
            for (int available = processStream.available(); available > 0 && (read = processStream.read(this.readBuffer, 0, Math.min(available, this.readBuffer.length))) >= 0; available -= read) {
                for (int i = 0; i < read; ++i) {
                    byte b = this.readBuffer[i];
                    buffer.write(b);
                    if (b != 10) continue;
                    buffer.writeTo(to);
                    buffer.reset();
                }
                any = true;
            }
            return any;
        }
    }

    private static class CaptureListeningDataReader
    implements Runnable {
        private final Path processOutput;
        private final Duration waitTime;
        private final CountDownLatch signal;
        private final AtomicReference<ListeningAddress> resultReference;
        private final Pattern listeningRegex = Pattern.compile("Listening on:\\s+(https?)://[^:]*:(\\d+)");
        private final Pattern startedRegex = Pattern.compile(".*Quarkus .* started in \\d+.*s.*");

        public CaptureListeningDataReader(Path processOutput, Duration waitTime, CountDownLatch signal, AtomicReference<ListeningAddress> resultReference) {
            this.processOutput = processOutput;
            this.waitTime = waitTime;
            this.signal = signal;
            this.resultReference = resultReference;
        }

        @Override
        public void run() {
            if (!this.ensureProcessOutputFileExists()) {
                this.unableToDetermineData("Log file '" + String.valueOf(this.processOutput.toAbsolutePath()) + "' was not created. Check if the options quarkus.log.level and quarkus.log.file.level are at least INFO (or more verbose).");
                return;
            }
            long bailoutTime = System.currentTimeMillis() + this.waitTime.toMillis();
            try (BufferedReader reader = new BufferedReader(new FileReader(this.processOutput.toFile()));){
                long timeStarted = Long.MAX_VALUE;
                boolean started = false;
                while (true) {
                    if (reader.ready()) {
                        Matcher regexMatcher;
                        String line = reader.readLine();
                        if (this.startedRegex.matcher(line).matches()) {
                            timeStarted = System.currentTimeMillis();
                            started = true;
                        }
                        if ((regexMatcher = this.listeningRegex.matcher(line)).find()) {
                            this.dataDetermined(regexMatcher.group(1), Integer.valueOf(regexMatcher.group(2)));
                            return;
                        }
                        if (!line.contains("Failed to start application (with profile")) continue;
                        this.unableToDetermineData("Application was not started: " + line);
                        return;
                    }
                    long now = System.currentTimeMillis();
                    if (now + 50L > bailoutTime || now - 100L > timeStarted) {
                        if (started) {
                            this.dataDetermined(null, null);
                        } else {
                            this.unableToDetermineData("Waited " + this.waitTime.getSeconds() + " seconds for " + String.valueOf(this.processOutput) + " to contain info about the listening port and protocol but no such info was found. Check if the options quarkus.log.level and quarkus.log.file.level are at least INFO (or more verbose).");
                        }
                        return;
                    }
                    try {
                        Thread.sleep(50L);
                    }
                    catch (InterruptedException e) {
                        this.unableToDetermineData("Thread interrupted while waiting for more data to become available in process output file: " + String.valueOf(this.processOutput.toAbsolutePath()));
                        reader.close();
                        return;
                    }
                }
            }
            catch (Exception e) {
                this.unableToDetermineData("Exception occurred while reading process output from file " + String.valueOf(this.processOutput));
                e.printStackTrace();
                return;
            }
        }

        private boolean ensureProcessOutputFileExists() {
            long bailoutTime = System.currentTimeMillis() + this.waitTime.toMillis();
            while (System.currentTimeMillis() < bailoutTime) {
                if (Files.exists(this.processOutput, new LinkOption[0])) {
                    return true;
                }
                try {
                    Thread.sleep(200L);
                }
                catch (InterruptedException e) {
                    this.unableToDetermineData("Thread interrupted while waiting for process output file to be created");
                    return false;
                }
            }
            return false;
        }

        private void dataDetermined(String protocolValue, Integer portValue) {
            this.resultReference.set(new ListeningAddress(portValue, protocolValue));
            this.signal.countDown();
        }

        private void unableToDetermineData(String errorMessage) {
            System.err.println(errorMessage);
            this.resultReference.set(null);
            this.signal.countDown();
        }
    }

    private static class SimpleContext
    implements IntegrationTestStartedNotifier.Context {
        private final Path logFile;

        public SimpleContext(Path logFile) {
            this.logFile = logFile;
        }

        @Override
        public Path logFile() {
            return this.logFile;
        }
    }
}

