/*
 * #%L
 * Wao :: Web Interface
 * %%
 * Copyright (C) 2009 - 2011 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ContactFilterImpl;
import fr.ifremer.wao.bean.ContactState;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.ObservationType;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.bean.ValidationResult;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.ContactStateMotif;
import fr.ifremer.wao.entity.ObsDebCode;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.entity.TerrestrialLocation;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceContact;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.GenericSelectModel;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.Environmental;
import org.apache.tapestry5.annotations.Import;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.services.javascript.JavaScriptSupport;
import org.apache.tapestry5.util.EnumSelectModel;
import org.chenillekit.tapestry.core.components.DateTimeField;
import org.slf4j.Logger;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;


/**
 * In this page, user will be able to edit a contact. Fields depends on right
 * level and validations.
 *
 * 
 */
@RequiresAuthentication(allowedRoles = {UserRole.ADMIN, UserRole.COORDINATOR, UserRole.OBSERVER}, readOnlyAllowed = false)
@Import(stylesheet = {"context:css/common.css", "context:css/contactForm.css"}, library = "context:js/contactForm.js")
public class ContactForm {

    @Inject
    private PropertyAccess propertyAccess;

    @Inject
    private Logger logger;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private ConnectedUser connectedUser;

    @Inject
    private WaoManager manager;

    @Inject
    private ServiceContact serviceContact;

    @Inject
    private ServiceUser serviceUser;

    @Environmental
    private JavaScriptSupport javascriptSupport;

    @InjectComponent
    private Zone globalZone;

    public String getDateTimeFormatPattern() {
        return "dd/MM/yyyy HH:mm";
    }

    public DateFormat getDateTimeFormat() {
        return new SimpleDateFormat(getDateTimeFormatPattern());
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }

    public DateFormat getMonthYearFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    void onActivate(EventContext eventContext) {
        if (eventContext.getCount() > 0) {
            contactId = eventContext.get(String.class, 0);
        }
    }

    String onPassivate() {
        return contactId;
    }

    void setupRender() throws WaoException {
        contact = null; // reset
        getContact();
    }

    @Log
    void afterRender() {
        addSendEmailScript();
    }

    protected void addSendEmailScript() {
        // Ask user to send an email if not already sent
        if ( ObsProgram.OBSMER == getContact().getObsProgram() // mail is only for ObsMer
             && ! getContact().getEmailSent()) {

            String confirmMessage =
                    "Souhaitez vous envoyer un email de demande d\\'ajout " +
                    "du navire à votre portefeuille Allegro afin de pouvoir " +
                    "saisir les données ?";

            javascriptSupport.addScript("new ContactSendEmail('%s', '%s');",
                    confirmMessage,
                    // Check state OBSERVATION_DONE
                    ContactState.OBSERVATION_DONE.name());

            javascriptSupport.addScript("new ContactSendEmail('%s', '%s');",
                    confirmMessage,
                    // Check state BOARDING_EXPECTED
                    ContactState.OBSERVATION_EXPECTED.name());
        }
    }

    @Persist
    private String contactId;

    @Persist
    private Contact contact;

    @Log
    public Contact getContact() {
        if (contact == null) {
            if (StringUtils.isEmpty(contactId)) {
                throw new IllegalStateException();
            } else {
                contact = serviceContact.getContact(contactId);
            }
        }
        return contact;
    }

    @InjectComponent
    private Form contactForm;

    /****************************** INFORMATIONS ******************************/

    public String getBoatDescription() {
        return manager.getTooltipBoat(contact.getBoat());
    }

    /****************************** OBSERVERS *********************************/

    @Inject
    private Block viewObservers;

    @Inject
    private Block editObservers;

    public Block getViewOrEditObservers() {
        Block result = viewObservers;
        if (connectedUser.isCoordinatorOrObserver() && contact.getValidationCompany() == null) {
            result = editObservers;
        }
        return result;
    }

    @Property
    private WaoUser observer; // for loop

    public List<WaoUser> getObservers() {
        List<WaoUser> observers = serviceUser.getObservers(connectedUser.getCompany(), true);
        return observers;
    }

    public boolean showSecondaryObservers() {
        boolean showSecondaryObservers = true;
        if (contact.getSampleRow().isPhoneCall()) {
            // when phoning, we are always single
            showSecondaryObservers = false;
        }
        return showSecondaryObservers;
    }

    /************************ OBSERVATION REPORT ******************************/

    @Component
    private DateTimeField beginDate, endDate, landingDate;

    @Inject
    private Block viewObservationReport;

    @Inject
    private Block editObservationReport;

    public Block getViewOrEditObservationReport() {
        Block result = viewObservationReport;
        if (connectedUser.isCoordinatorOrObserver() && contact.getValidationCompany() == null) {
            result = editObservationReport;
        }
        return result;
    }

    public SelectModel getObservationTypeSelectModel() {
        EnumSelectModel selectModel = null;
        if (getContact().getSampleRow().isFieldWorkObservation()) {
            ObservationType[] allowedObservationTypes = {
                                         ObservationType.FIELD_WORK_OBSERVATION,
                                         ObservationType.FIELD_WORK_SURVEY };
            selectModel = new EnumSelectModel(ObservationType.class, messages,
                                              allowedObservationTypes);
        }
        return selectModel;
    }

    @Inject
    private ServiceReferential serviceReferential;

    public List<TerrestrialLocation> getTerrestrialLocations() {
        List<TerrestrialLocation> locations = serviceContact.
                getPossibleTerrestrialLocations(getContact());
        return locations;
    }

    public List<ObsDebCode> getObsDebCodes() {
        List<ObsDebCode> obsDebCodes = serviceContact.
                                         getPossibleObsDebCodes(getContact());
        return obsDebCodes;
    }

    public SelectModel getContactStateSelectModel() {
        List<ContactState> allowedStates = ContactState.getAllowedStates(contact.getObsProgram());
        return new EnumSelectModel(ContactState.class, messages,
                                 // just convert list to array
                                 allowedStates.toArray(new ContactState[allowedStates.size()]));
    }

    public List<ContactStateMotif> getContactStateMotives() {
        List<ContactStateMotif> motives = serviceReferential.getAllContactStateMotifs(null);
        return motives;
    }

    /************************ ADMIN FIELDS ************************************/


    public boolean showDataReliability() {
        // ObsMer is the single program where data reliability is evaluated
        return getContact().getObsProgram() == ObsProgram.OBSMER &&
               (connectedUser.isAdmin() || connectedUser.isCoordinator());
    }

    public boolean canEditDataReliability() {
        boolean result = connectedUser.isAdmin() // coordinator can see the value
                      && BooleanUtils.isTrue(getContact().getValidationCompany())
                      && getContact().getValidationProgram() == null;
        return result;
    }


    /************** VALIDATION AND SUBMISSION *********************************/

    @Persist
    private boolean saveContact; // true if user selected save button, false if revert

    @Log
    public void onSelectedFromRevertContact() {
        saveContact = false;
    }

    @Log
    public void onSelectedFromSaveContact() {
        saveContact = true;
    }

    @Log
    void onValidateFormFromContactForm() {
        contactForm.clearErrors();

        ValidationResult validationResult = serviceContact.validateContact(contact);
        if ( ! validationResult.isSuccess()) {
            contactForm.recordError(validationResult.getMessage());
        }
    }

    @Log
    Object onFailureFromContactForm() {
        // refresh zone to show errors
        //return globalZone;
        return contactForm;
    }

    @Inject
    private Messages messages;

    @Property
    private boolean sendEmail;

    @InjectPage
    private Contacts contacts;

    @Log
    Object onSuccessFromContactForm() {

        if (saveContact) {

            if (logger.isDebugEnabled()) {
                logger.debug("Contact save : " + contact);
                logger.debug("secondary observers : " + contact.getSecondaryObservers());
                logger.debug("Contact sendEmail : " + sendEmail);
            }
            serviceContact.saveContact(connectedUser, contact, false);

            try {
                if (sendEmail && serviceContact.sendContactDoneEmail(contact)) {
                    layout.addInfo("Un email a été envoyé pour l'ajout du navire au portefeuille ALLEGRO.");
                }
            } catch (WaoBusinessException eee) {
                String message = manager.getErrorMessage(eee, messages, logger);
                layout.addError(message);
            }
        } else {
            // reset
            contact = null;
        }

        if (saveContact) {
            // make the modified contact highlighted in contacts grid
            contacts.setContactSelectedId(contact.getTopiaId());
            return contacts;
        } else {
            return this;
        }
    }

    public String getLabelForEnum(Enum value) {
        return messages.get(value.getDeclaringClass().getSimpleName() + "." + value.name());
    }

    public String getMammalsInfoBoxStyle() {
        if (!contact.isMammalsCapture()) {
            return "display: none;";
        }
        return StringUtils.EMPTY;
    }
}
