/*
 * #%L
 * $Id: WelcomeHandler.java 4233 2015-05-22 16:10:26Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/ui/WelcomeHandler.java $
 * %%
 * Copyright (C) 2011 - 2015 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui;

import static org.nuiton.i18n.I18n.t;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.ToolTipManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.widget.AboutFrame;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.ui.config.RConfigUI;
import fr.ifremer.isisfish.ui.config.SSHLauncherConfigUI;
import fr.ifremer.isisfish.ui.input.InputContext;
import fr.ifremer.isisfish.ui.input.InputUI;
import fr.ifremer.isisfish.ui.queue.QueueUI;
import fr.ifremer.isisfish.ui.result.ResultUI;
import fr.ifremer.isisfish.ui.script.ScriptUI;
import fr.ifremer.isisfish.ui.sensitivity.SensitivityContext;
import fr.ifremer.isisfish.ui.sensitivity.SensitivityUI;
import fr.ifremer.isisfish.ui.simulator.SimulUI;
import fr.ifremer.isisfish.ui.simulator.SimulatorContext;
import fr.ifremer.isisfish.ui.vcs.VCSConfigUI;
import jaxx.runtime.swing.config.ConfigUI;
import jaxx.runtime.swing.config.ConfigUIHelper;

/**
 * Welcome related ui handler.
 * 
 * @author chatellier
 * @version $Revision: 4233 $
 * 
 * Last update : $Date: 2015-05-22 18:10:26 +0200 (ven., 22 mai 2015) $
 * By : $Author: echatellier $
 */
public class WelcomeHandler extends CommonHandler {

    /** Class logger. */
    private static Log log = LogFactory.getLog(WelcomeHandler.class);

    protected WelcomeUI welcomeUI;

    protected Map<JFrame, WelcomePanelUI> allFrameOpened;
    
    // URL alias
    protected static final Map<String, String> URLSALIAS = new HashMap<String, String>();

    static {
        URLSALIAS.put("ISISFISH", "http://www.isis-fish.org/");
        URLSALIAS.put("JAVA_API", "https://docs.oracle.com/javase/8/docs/api/");
        URLSALIAS.put("ISIS_API", "http://www.isis-fish.org/apidocs/index.html");
        URLSALIAS.put("MATRIX_API", "http://maven-site.nuiton.org/nuiton-matrix/apidocs/index.html");
        URLSALIAS.put("TOPIA_API", "http://maven-site.nuiton.org/topia/topia-persistence/apidocs/index.html");
    }

    public WelcomeHandler(WelcomeUI welcomeUI) {
        this.welcomeUI = welcomeUI;
    }

    public void afterInit() {
        allFrameOpened = new HashMap<JFrame, WelcomePanelUI>();
        welcomeUI.getWelcomePanelUI().setContent(new WelcomeTabUI(welcomeUI));
    
        // increase tooltip display time
        ToolTipManager toolTipManager = ToolTipManager.sharedInstance();
        toolTipManager.setInitialDelay(0);
        toolTipManager.setDismissDelay(60000);
    }

    protected void openFrame(WelcomeUI welcomeUI, Component c, String title) {

        WelcomePanelUI welcome = new WelcomePanelUI(welcomeUI);
        welcome.setContent(c);
        
        JFrame frame = new JFrame(title);
        frame.setLayout(new BorderLayout());
        frame.add(welcome, BorderLayout.CENTER);
        frame.setSize(new Dimension(800, 600));
        frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

        allFrameOpened.put(frame, welcome);
        frame.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                JFrame f = (JFrame) e.getSource();
                allFrameOpened.get(f).close(f);
            }
        });
        frame.setLocationRelativeTo(welcomeUI);
        frame.setVisible(true);
    }

    /**
     * 
     * @param welcomeUI
     */
    public void newSimulationFrame(WelcomeUI welcomeUI) {
        SimulatorContext simulatorContext = new SimulatorContext(welcomeUI);
        SimulUI simulUI = new SimulUI(simulatorContext);
        openFrame(welcomeUI, simulUI, t("isisfish.simulation.title"));
    }
    
    /**
     * 
     * @param welcomeUI
     */
    public void newResultFrame(WelcomeUI welcomeUI) {
        openFrame(welcomeUI, new ResultUI(), t("isisfish.result.title"));
    }
    
    /**
     * 
     * @param welcomeUI
     */
    public void newInputFrame(WelcomeUI welcomeUI) {
        InputContext inputContext = new InputContext(welcomeUI);
        InputUI inputUI = new InputUI(inputContext);
        openFrame(welcomeUI, inputUI, t("isisfish.input.title"));
    }
    
    /**
     * 
     * @param welcomeUI
     */
    public void newScriptFrame(WelcomeUI welcomeUI) {
        openFrame(welcomeUI, new ScriptUI(), t("isisfish.script.title"));
    }
    
    /**
     * 
     * @param welcomeUI
     */
    public void newSensitivityFrame(WelcomeUI welcomeUI) {
        SensitivityContext sensitivityContext = new SensitivityContext(welcomeUI);
        SensitivityUI sensitivityUI = new SensitivityUI(sensitivityContext);
        openFrame(welcomeUI, sensitivityUI, t("isisfish.sensitivity.title"));
    }
    
    /**
     * 
     * @param welcomeUI
     */
    public void newQueueFrame(WelcomeUI welcomeUI) {
        openFrame(welcomeUI, new QueueUI(), t("isisfish.queue.title"));
    }

    /**
     * 
     * @param welcomeUI parent ui
     */
    public void close(WelcomeUI welcomeUI) {
        WelcomeSaveVerifier verifier = welcomeUI.getContextValue(WelcomeSaveVerifier.class);
        if (verifier.allIsSaved()) {
            welcomeUI.dispose();
        }
    }

    /**
     * 
     * @param welcomeUI parent ui
     */
    public void updateVCS(WelcomeUI welcomeUI) {
        // FIXME a reimplanter
        // thread:Welcome.updateVCS()
    }
    
    /**
     * Show config frame.
     * 
     * @param welcomeUI parent ui
     */
    public void config(WelcomeUI welcomeUI) {
        ConfigUIHelper modelBuilder = new ConfigUIHelper(IsisFish.config);

        // category main
        modelBuilder.addCategory(t("isisfish.config.category.main"), t("isisfish.config.category.main.description"));
        modelBuilder.addOption(IsisConfig.Option.CONFIG_FILE);
        modelBuilder.addOption(IsisConfig.Option.DATABASE_DIRECTORY);
        modelBuilder.addOption(IsisConfig.Option.COMMUNITY_DATABASE_DIRECTORY);
        modelBuilder.addOption(IsisConfig.Option.COMPILATION_DIRECTORY);
        modelBuilder.addOption(IsisConfig.Option.MONITORING_DIRECTORY);
        modelBuilder.addOption(IsisConfig.Option.JAVADOC_DIRECTORY);
        modelBuilder.addOption(IsisConfig.Option.BACKUP_DIRECTORY);
        modelBuilder.addOption(IsisConfig.Option.LOCALE);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_SHOW_ONLY_ERROR);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_SCRIPT_AUTOCONFIG);

        // category simulation
        modelBuilder.addCategory(t("isisfish.config.category.simulation"), t("isisfish.config.category.simulation.description"));
        modelBuilder.addOption(IsisConfig.Option.SIMULATOR_IN_MAXTHREADS);
        modelBuilder.addOption(IsisConfig.Option.SIMULATOR_SUB_MAXPROCESS);
        modelBuilder.addOption(IsisConfig.Option.SIMULATOR_SUB_MAXMEMORY);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_MATRIX_VECTOR_CLASS);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_MATRIX_VECTOR_SPARSE_CLASS);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_MATRIX_THRESHOLD_USE_SPARSE_CLASS);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_MATRIX_USE_LAZY_VECTOR);
        modelBuilder.addOption(IsisConfig.Option.MAPPED_RESULT_MATRIX_VECTOR_CLASS);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_STORE_RESULT_ON_DISK);
        modelBuilder.addOption(IsisConfig.Option.SIMULATION_STORE_RESULT_CACHE_STEP);
        modelBuilder.addOption(IsisConfig.Option.EXPORT_FORCE_COMPRESSION);

        // category database vcs
        modelBuilder.addCategory(t("isisfish.config.category.officialvcs"), t("isisfish.config.category.officialvcs.description"));
        modelBuilder.addOption(IsisConfig.Option.VCS_TYPE);
        modelBuilder.addOption(IsisConfig.Option.VCS_PROTOCOL);
        modelBuilder.addOption(IsisConfig.Option.VCS_HOST_NAME);
        modelBuilder.addOption(IsisConfig.Option.VCS_PATH);
        modelBuilder.addOption(IsisConfig.Option.VCS_USER_NAME);
        modelBuilder.addOption(IsisConfig.Option.VCS_USER_PASSWORD);

        // category community vcs
        modelBuilder.addCategory(t("isisfish.config.category.communityvcs"), t("isisfish.config.category.communityvcs.description"));
        modelBuilder.addOption(IsisConfig.Option.VCS_COMMUNITY_TYPE);
        modelBuilder.addOption(IsisConfig.Option.VCS_COMMUNITY_PROTOCOL);
        modelBuilder.addOption(IsisConfig.Option.VCS_COMMUNITY_HOST_NAME);
        modelBuilder.addOption(IsisConfig.Option.VCS_COMMUNITY_PATH);
        modelBuilder.addOption(IsisConfig.Option.VCS_COMMUNITY_USER_NAME);
        modelBuilder.addOption(IsisConfig.Option.VCS_COMMUNITY_PASSWORD);

        // category misc
        modelBuilder.addCategory(t("isisfish.config.category.misc"), t("isisfish.config.category.misc.description"));
        modelBuilder.addOption(IsisConfig.Option.SSH_KEY_FILE);
        modelBuilder.addOption(IsisConfig.Option.USER_NAME);
        modelBuilder.addOption(IsisConfig.Option.SMTP_SERVER);
        modelBuilder.addOption(IsisConfig.Option.USER_MAIL);

        ConfigUI configUI = modelBuilder.buildUI(welcomeUI, t("isisfish.config.category.main"));
        JDialog dialogUI = new JDialog(welcomeUI, t("isisfish.config.title"));
        dialogUI.add(configUI);
        dialogUI.pack();
        dialogUI.setLocationRelativeTo(welcomeUI);
        dialogUI.setVisible(true);
    }

    /**
     * Display VCS config ui.
     * 
     * @param welcomeUI parent ui
     */
    public void configVCS(WelcomeUI welcomeUI) {
        VCSConfigUI vcsConfig = new VCSConfigUI(welcomeUI, welcomeUI);
        vcsConfig.pack();
        vcsConfig.setLocationRelativeTo(welcomeUI);
        vcsConfig.setVisible(true);
    }

    /**
     * Display ssh launcher config ui.
     * 
     * @param welcomeUI parent ui
     */
    public void configurationSSHLauncher(WelcomeUI welcomeUI) {
        SSHLauncherConfigUI configUI = new SSHLauncherConfigUI(welcomeUI, welcomeUI);
        configUI.pack();
        configUI.setSize(600, configUI.getHeight());
        configUI.setLocationRelativeTo(welcomeUI);
        configUI.setVisible(true);
    }

    /**
     * Display R config tester UI.
     * 
     * @param welcomeUI parent ui
     */
    public void configurationR(WelcomeUI welcomeUI) {
        RConfigUI rconfigUI = new RConfigUI(welcomeUI, welcomeUI);
        rconfigUI.pack();
        rconfigUI.setLocationRelativeTo(welcomeUI);
        rconfigUI.setVisible(true);
    }

    /**
     * Open url using default system browser.
     * 
     * @param welcomeUI parent ui
     * @param urlOrAlias url alias, or full url
     */
    public void help(WelcomeUI welcomeUI, String urlOrAlias) {
        try {
            String url = URLSALIAS.get(urlOrAlias);

            if (url == null) {
                url = urlOrAlias;
            }

            Desktop.getDesktop().browse(new URL(url).toURI());
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't show help", e);
            }
        }
    }

    /**
     * Show about frame from nuiton-widgets.
     * 
     * @param welcomeUI parent ui
     */
    public void about(WelcomeUI welcomeUI) {
       AboutFrame about = new AboutFrame();
       about.setTitle(t("isisfish.about.title"));
       about.setIconPath("images/isislogo.png");
       about.setAboutHtmlText(t("isisfish.about.abouthtmltext", IsisConfig.getVersion()));
       about.setLicenseText(t("isisfish.about.licensetext"));
       about.setBackgroundColor(Color.WHITE);
       about.setSize(640, 520);
       about.setLocationRelativeTo(welcomeUI);
       about.setVisible(true);
    }
}
