/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiStorageServiceImpl.java 1876 2011-01-10 14:39:23Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.2/vradi-services/src/main/java/com/jurismarches/vradi/services/VradiStorageServiceImpl.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.VradiServiceConfigurationHelper;
import com.jurismarches.vradi.beans.SendingHelper;
import com.jurismarches.vradi.beans.XmlStreamImportResult;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.managers.BindingManager;
import com.jurismarches.vradi.services.managers.ClientManager;
import com.jurismarches.vradi.services.managers.FormManager;
import com.jurismarches.vradi.services.managers.FormTypeManager;
import com.jurismarches.vradi.services.managers.MailingManager;
import com.jurismarches.vradi.services.managers.SearchManager;
import com.jurismarches.vradi.services.managers.TemplateManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;
import com.jurismarches.vradi.services.tasks.TasksManager;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.DateUtil;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.addons.WikittyImportExportService;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.jurismarches.vradi.VradiConstants.SendingStatus;
import static com.jurismarches.vradi.VradiConstants.SessionStatus;
import static org.nuiton.i18n.I18n._;

/**
 * Vradi storage implementation.
 * 
 * @author morin
 * @author chatellier
 */
public class VradiStorageServiceImpl implements VradiStorageService {

    static private final Log log = LogFactory.getLog(VradiStorageServiceImpl.class);

    protected ApplicationConfig config;

    /**
     * Proxy to store and retrieve data.
     */
    protected WikittyProxy wikittyProxy;

    protected WikittyImportExportService importExportService;
    
    protected ThesaurusManager thesaurusManager;
    protected BindingManager bindingManager;
    protected FormManager formManager;
    protected FormTypeManager formTypeManager;
    protected ClientManager clientManager;
    protected SearchManager searchManager;
    protected MailingManager mailingManager;
    protected TasksManager tasksManager;
    protected FileService fileService;

    /**
     * Remote instantiation and empty constructor for hessian.
     */
    public VradiStorageServiceImpl() {
        this(VradiServiceConfiguration.getInstance(), ServiceFactory.getWikittyProxy());
    }

    /**
     * This constructor exists for local wikittyProxy instanciation.
     * 
     * @param config vradi config
     * @param wikittyProxy custom wikitty proxy
     */
    protected VradiStorageServiceImpl(ApplicationConfig config, WikittyProxy wikittyProxy) {
        this.wikittyProxy = wikittyProxy;
        this.config = config;

        thesaurusManager = new ThesaurusManager(wikittyProxy);
        clientManager = new ClientManager(config, wikittyProxy);
        formTypeManager = new FormTypeManager(config, wikittyProxy);
        mailingManager = new MailingManager(config, wikittyProxy);

        fileService = new FileServiceImpl(config);
        searchManager = new SearchManager(wikittyProxy, thesaurusManager);
        formManager = new FormManager(config, wikittyProxy, thesaurusManager, searchManager);
        bindingManager = new BindingManager(config, wikittyProxy, formTypeManager, formManager);
        tasksManager = new TasksManager(config, wikittyProxy, mailingManager, bindingManager);

        // init cron tasks
        tasksManager.initTasks();
        wikittyProxy.getWikittyService().addWikittyServiceListener(tasksManager, WikittyService.ServiceListenerType.ALL);
    }

    /**
     * Get {@link WikittyImportExportService} singl instance.
     * 
     * @return WikittyImportExportService singleton
     */
    protected WikittyImportExportService getImportExportService() {
        if (importExportService == null) {
            VradiServiceConfiguration config = VradiServiceConfiguration.getInstance();

            importExportService = new WikittyImportExportService(
                    config,
                    wikittyProxy.getSecurityToken(),
                    wikittyProxy.getWikittyService());
            
        }
        return importExportService;
    }

    /**
     * Create all default status.
     *
     * @return all status created
     */
    @Override
    public List<Status> createDefaultStatuses() throws VradiException {
        return formManager.createDefaultStatuses();
    }

    @Override
    public Session createNewSession() throws VradiException {
        return formManager.createNewSession();
    }

    @Override
    public Session bindForms(Session session) throws VradiException {
        return formManager.bindForms(session);
    }

    @Override
    public List<Sending> createAllSending(String sessionId, Group group, List<String> formsIdsToBind) throws VradiException {
        return formManager.createAllSending(sessionId, group, formsIdsToBind);
    }

    @Override
    public Sending createSending(String sessionId, User user, List<String> formsIdsToBind) throws VradiException {
        return formManager.createUserSending(sessionId, user, formsIdsToBind);
    }

    @Override
    public List<Sending> removeAllSending(String sessionId, List<String> formsIds, Group group)
            throws VradiException{
        return formManager.removeAllSending(sessionId, formsIds, group);
    }

    @Override
    public Sending removeSending(String sessionId, List<String> formsIds, User user)
            throws VradiException{
        return formManager.removeSending(sessionId, formsIds, user);
    }

    /**
     * Creates and store forms from an {@code WebHarvestStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param webHarvestStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    @Override
    public XmlStreamImportResult importFormsFromWebHarvestStream(WebHarvestStream webHarvestStream)
            throws VradiException {
        return bindingManager.importFormsFromWebHarvestStream(webHarvestStream);
    }

    /**
     * Creates and store forms from an {@code XmlStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param xmlStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    @Override
    public XmlStreamImportResult importFormsFromXmlStream(XmlStream xmlStream)
            throws VradiException {
        return bindingManager.importFormsFromXmlStream(xmlStream);
    }

    @Override
    public VradiUser createUser(VradiUser user) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateVradiUser(" + user.getLogin() + ")");
        }
        
        Search search = Search.query().eq(VradiUser.FIELD_WIKITTYUSER_LOGIN, user.getLogin());
        Criteria criteria = search.criteria();
        criteria.setEndIndex(1); // one is enought
        
        PagedResult<VradiUser> users = wikittyProxy.findAllByCriteria(VradiUser.class, criteria);
        
        if (users.getNumFound() > 0) {
            if (log.isDebugEnabled()) {
                log.debug("User " + user.getLogin() + " already exists");
            }
            return null;
            
        } else {
            return wikittyProxy.store(user);
        }
    }

    @Override
    public VradiUser loginUser(String login, String mp5password) {
        if (log.isDebugEnabled()) {
            log.debug("logVradiUser(" + login + ", " + mp5password + ")");
        }
        
        Criteria criteria = Search.query()
                .eq(VradiUser.FQ_FIELD_WIKITTYUSER_LOGIN, login)
                .eq(VradiUser.FQ_FIELD_WIKITTYUSER_PASSWORD, mp5password)
                .criteria();
        criteria.setEndIndex(1); // max 1
        
        VradiUser vradiUser = wikittyProxy.findByCriteria(VradiUser.class, criteria);
        return vradiUser;
    }

    /*@Override
    public void importData(File file) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("importData(file)");
        }

        // TODO: create an archive of vradi.home directory before

        try {
            String toURI = file.toURI().toString();
            log.info("importing file: " + toURI);
            
            WikittyImportExportService importExportService = getImportExportService();
            importExportService.syncImportFromUri(WikittyImportExportService.FORMAT.CSV, toURI);
    
            log.info("import done");
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't import data", e);
            }
            throw new VradiException("Can't import data", e);
        }
    }

    @Override
    public String exportData() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("exportData()");
        }

        try {
            Search search = Search.query().keyword("*");
            Criteria criteria = search.criteria();

            WikittyImportExportService importExportService = getImportExportService();
            String export = importExportService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, criteria);
            log.info("export done");
            return export;

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't export data", e);
            }
            throw new VradiException("Can't export data", e);
        }
    }*/

    @Override
    public void importAsCSV(String uri) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("Importing data as CSV with url = " + uri);
        }

        try {
            WikittyImportExportService ieService = getImportExportService();
            ieService.syncImportFromUri(WikittyImportExportService.FORMAT.CSV, uri);
        }
        catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import data", eee);
            }
            throw new VradiException("Can't import data", eee);
        }
    }

    @Override
    public String exportAsCSV(Criteria criteria) throws VradiException {

        //FIXME JC 01/09/2010 Should create a file and make him downloadable through FileService.

        if (log.isDebugEnabled()) {
            log.debug("Exporting data as CSV with criteria " + criteria);
        }

        String result = null;
        try {
            WikittyImportExportService ieService = getImportExportService();
            result = ieService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, criteria);
        }
        catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export data", eee);
            }
        }
        return result;
    }

    @Override
    public String exportThesaurusAsCSV() throws VradiException {
        // creation du criteria wikitty
        // (export de tous les wikitty avec extension "thesaurus"
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                WikittyTreeNode.EXT_WIKITTYTREENODE);
        Criteria criteria = search.criteria();
        return exportAsCSV(criteria);
    }

    @Override
    public void reindexData() {
        if (log.isDebugEnabled()) {
            log.debug("reindexData()");
        }
        
        wikittyProxy.getWikittyService().syncSearchEngine(null);
    }

    @Override
    public void archiveQueries(QueryMaker queryMaker)
            throws VradiException {
        clientManager.archiveQueries(queryMaker);
    }

    @Override
    public void setAssociatedFields(String extensionName,
                                           String templateName,
                                    Map<String, String> fieldMap)
            throws VradiException {

        formTypeManager.setAssociatedFields(extensionName, templateName, fieldMap);

    }

    @Override
    public String generatePDF(String formId, boolean force) throws VradiException {

        String uri = null;

        String extensionName = null;
        Form form = wikittyProxy.restore(Form.class, formId);
        for(String ext : form.getExtensionNames()) {
            if(!ext.equals(Infogene.EXT_INFOGENE)
                    && !ext.equals(Form.EXT_FORM)) {
                extensionName = ext;
                break;
            }

        }
        WikittyExtension extension = formTypeManager.getFormType(extensionName);
        // VradiHelper.getFormTypeTemplate(extension);
        String template = extension.getTagValue(VradiConstants.FormTypeTemplateEnum.FIELD_TEMPLATE_ATTRIBUTE.getValue());
        File templateFile = formTypeManager.getTemplate(extensionName, template);
        Map<String, String> fieldMap = getAssociatedFields(extension.getName(), template);

        if (log.isDebugEnabled()) {
            log.debug("Generating PDF file for form " + form.getWikittyId() +
                    " (template = " + template + ")");
        }

        try {
            // but only those associated with current templates !
            List<Form> currentForms = Collections.singletonList(form);
            List<File> filesToAdd = generateFilledDocumentInPDF(
                            templateFile, currentForms, fieldMap, force);

            if (filesToAdd != null && !filesToAdd.isEmpty()) {
                uri = filesToAdd.get(0).getName();
            }
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't generate pdf", ex);
            }
            throw new VradiException("Can't generate pdf", ex);
        }

        return uri;
    }


    @Override
    public Collection<String> getTemplateFilenames(WikittyExtension extension) {
        return formTypeManager.getTemplateFilenames(extension);
    }
    
    @Override
    public Map<String, String> getAssociatedFields(String extensionName,
                                                   String templateName)
            throws VradiException {
        return formTypeManager.getAssociatedFields(extensionName, templateName);
    }

    @Override
    public List<String> getDocumentFields(WikittyExtension extension, String templateName) throws VradiException {

        List<String> documentsFields;

        try {
            File templateDir = VradiServiceConfigurationHelper.getTemplatesDir(config);
            File extensionTemplate = new File(templateDir, extension.getName());
            File templateFile = new File(extensionTemplate, templateName);
    
            TemplateManager templateManager = new TemplateManager(config, templateFile);
            documentsFields = templateManager.getDocumentFields();
        }
        catch (VradiException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get documents fields", ex);
            }
            throw ex;
        }
        return documentsFields;
    }

    /**
     * Generate offer PDF form given forms with specified template.
     * 
     * @param template template to use to generate PDF
     * @param forms to to generate
     * @param fieldBindings map between template field name and form fields names
     * @param replace replace already existing generated forms
     * @return generated pdf files
     * @throws VradiException for various possible errors
     */
    protected List<File> generateFilledDocumentInPDF(File template, List<Form> forms,
                                            Map<String, String> fieldBindings,
                                            boolean replace)
            throws VradiException {

        List<File> generatedPDFList = null;
        
        if(template == null) {
            if (log.isWarnEnabled()) {
                log.warn("Try to generate form pdf without template");
            }
        } else if (fieldBindings == null) {
            if (log.isWarnEnabled()) {
                log.warn("No association field found, abording");
            }
        }
        else {
            TemplateManager templateManager = new TemplateManager(config, template);
            
            // split fqn form field name in an array
            // composed of extension name, and extension field
            // TODO EC20100510 set in commun with same function in
            // {@link generateFilledDocumentInPDF(TemplateManager, Form form, Map<String, String[]>, boolean)
            Map<String, String[]> map = new HashMap<String, String[]>();
            for(Map.Entry<String, String> binding : fieldBindings.entrySet()) {
                String fieldFQName = binding.getValue();
                if(fieldFQName != null && fieldFQName.indexOf(".") > 0) {
                    int dot = fieldFQName.indexOf(".");
                    String[] fqField = new String[2];
                    fqField[0] = fieldFQName.substring(0, dot);
                    fqField[1] = fieldFQName.substring(dot + 1);
                    map.put(binding.getKey(), fqField);
                }
            }

            generatedPDFList = new ArrayList<File>();
            for (Form form : forms) {
                File f = generateFilledDocumentInPDF(templateManager, form, map, replace);
                generatedPDFList.add(f);
            }
        }

        return generatedPDFList;
    }
    
    /**
     * Generate a single pdf file for specified form.
     * 
     * @param templateManager pdf template manager
     * @param form form to get pdf
     * @param fieldBindings association between field name and form fields (couple extension name, extension field)
     * @param replace replace already existing pdf
     * @return the pdf file associated with pdf
     * @throws VradiException for various possible errors
     */
    protected File generateFilledDocumentInPDF(TemplateManager templateManager,
            Form form, Map<String, String[]> fieldBindings,
            boolean replace) throws VradiException {

        File result = mailingManager.getGeneratedPdfFile(form);
        if (!replace && result.exists()) {
            return result;
        }

        Map<String, Object> fieldValues = new HashMap<String, Object>();
        for(Map.Entry<String, String[]> binding : fieldBindings.entrySet()) {
            String[] fqField = binding.getValue();
            Object value = form.getField(fqField[0], fqField[1]);
            fieldValues.put(binding.getKey(),
            value != null ? value.toString() : null);
        }

        // Extract files
        List<String> formUrls = new ArrayList<String>();
        if(form.getFiles() != null) {
            for(String file : form.getFiles()) {
                String formUrl =  fileService.getFormEmbeddedUrl(file, form.getWikittyId());
                formUrls.add(formUrl);
            }
        }

        templateManager.generateDoc(result.getPath(), fieldValues,
        formUrls.toArray(new String[formUrls.size()]));
        
        return result;
    }

    @Override
    public String sendMessages(String sessionId) throws VradiException {

        Session session = wikittyProxy.restore(Session.class, sessionId);

        if (SessionStatus.isSent(session)) {
            return StringUtils.EMPTY;
        }

        // Pass to active for session in error or stopped
        if (SessionStatus.isToRestart(session)) {

            log.info("Session was stopped or in error, resume sending");

            session.setStatus(SessionStatus.ACTIVE.getValue());
            session.setSessionLogs(StringUtils.EMPTY);
            session = wikittyProxy.store(session);
        }

        // Get all files to attach
        Set<String> filesNames = session.getFiles();
        List<File> files = new ArrayList<File>();
        if (filesNames != null) {
            for (String fileName : filesNames) {
                File file = fileService.downloadSessionAttachment(fileName, sessionId);
                files.add(file);
                log.info("Files founds for session : " + session.getSessionDate() + " : " + (file == null ? "not found" : file.getName()));
            }
        }

        // Init logger to keep exception
        List<String> sessionLogger = new ArrayList<String>();

        Set<String> sendingIds = session.getSending();
        if (sendingIds != null) {

            for (String sendingId : sendingIds) {

                // Flag to detect if it's canceled by user
                session = wikittyProxy.restore(Session.class, session.getWikittyId());
                if (SessionStatus.isStopped(session)) {
                    log.warn("Sending stopped by user");
                    sessionLogger.add(_("vradi.error.session.stopped", session.getNum(),
                            DateUtil.formatDate(session.getSessionDate(), DateUtil.DEFAULT_PATTERN)));
                }

                Sending sending = wikittyProxy.restore(Sending.class, sendingId);
                if (sending != null) {
                    if (SendingStatus.isSent(sending)) {
                        if (log.isDebugEnabled()) {
                            log.debug("Sending " + sendingId + " already sent, skip");
                        }
                    } else if (SendingStatus.isToSend(sending)) {

                        // Send
                        String log = sendMessage(sendingId, session.getParagraph(), files);
                        if (log != null) {
                            sessionLogger.add(log);
                        }
                    }
                }
            }
        }

        // Get logs
        String formatedLogs = formatSessionLog(sessionLogger); 

        // Set session to sent or error
        if (sessionLogger.isEmpty()) {
            session.setStatus(SessionStatus.SENT.getValue());
            session.setSessionLogs(StringUtils.EMPTY);
        } else {
            session.setStatus(SessionStatus.ERROR.getValue());
            session.setSessionLogs(formatedLogs);
        }
        session.setSessionDate(new Date());
        wikittyProxy.store(session);

        return formatedLogs;
    }

    @Override
    public String sendMessage(String sendingId, String sessionParagraph, Collection<File>  filesToAttach) throws VradiException {
        return sendMessage(sendingId, sessionParagraph, null, false, filesToAttach);
    }

    @Override
    public String sendMessage(String sendingId, String sessionParagraph, String email, Collection<File> filesToAttach) throws VradiException {
        return sendMessage(sendingId, sessionParagraph, email, true, filesToAttach);
    }

    public String sendMessage(String sendingId, String sessionParagraph, String email, boolean reSend, Collection<File> filesToAttach) throws VradiException {

        Sending sending = wikittyProxy.restore(Sending.class, sendingId);
        User user = clientManager.getUser(sending.getUser());

        // Will resend sending already sent
        if (!reSend) {
            // Do nothing
            if (!SendingStatus.isToSend(sending)){
                return null;
            }
        }

        // If its client or group sending : do nothing
        // Never append
//        if (sending.getUser() == null) {
//            sending.setStatus(VradiConstants.SendingStatus.SENT.getValue());
//            return wikittyProxy.store(sending);
//        }

        Set<String> formsIds = SendingHelper.extractForms(wikittyProxy, sending.getGroupForms());

        // If no forms founds
        if (formsIds.isEmpty()) {

            // Set session in error
            sending.setStatus(SendingStatus.DELETED.getValue());
            wikittyProxy.store(sending);

            // Do nothing
            return null;
        }

        // Get email address
        // si aucune n'est specifiée, on utilise celle du client
        if (StringUtils.isBlank(email)) {
            email = user.getEmail();
            
            // il peut arriver qu'un user n'est pas d'email et que l'admin
            // souhaite quand meme faire l'envoie d'une session.
            // dans ce cas on ne traite pas l'envoi de mail courant.
            if (StringUtils.isBlank(email)) {

                if (log.isWarnEnabled()) {
                    log.warn("User " + user.getName() + " has not valid email (skip current mail sending)");
                }
                // Set session in error
                sending.setStatus(SendingStatus.ERROR.getValue());
                wikittyProxy.store(sending);

                return _("vradi.error.sending.emailIsBlank", user.getName());
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Sending message to " + email);
        }

        String subject = VradiServiceConfigurationHelper.getOfferMailSubject(config);

        // Set message
        String message = sessionParagraph + "\n\n" + sending.getParagraph();

        List<Form> forms = wikittyProxy.restore(Form.class, new ArrayList<String>(formsIds));

        boolean receptionProof = sending.getReceptionProof();

        // Post mail and save msg id
        String messageId;
        try {
            messageId = mailingManager.postMail(
                email, subject, message, forms, receptionProof, filesToAttach);

        } catch (VradiException eee) {

            log.error("Failed to send sending for email : " + email, eee);

            // Set sending in error
            sending.setStatus(SendingStatus.ERROR.getValue());
            wikittyProxy.store(sending);

            return _("vradi.error.sending.failedToSend", user.getName(), eee.getMessage());
        }
        sending.setMessageId(messageId);

        // Set sending status
        sending.setStatus(receptionProof ?
                SendingStatus.WAITING_RECEPTION_PROOF.getValue() :
                SendingStatus.SENT.getValue());

        // Update sending
        wikittyProxy.store(sending);

        return null;
    }

    protected String formatSessionLog(List<String> logs) {
        return StringUtils.join(logs, ",\n");
    }

    @Override
    public void receiveMessages() throws VradiException {
        mailingManager.receiveMails();
    }

    @Override
    public Session stopSentMail(Session session) throws VradiException {

        // Flag status to stop traitement
        session.setStatus(SessionStatus.STOPPED.getValue());

        // Update
        session = wikittyProxy.store(session);

        return session;
    }

    /**
     * Generate all the PDFs needed to send a session.
     *
     * @param sessionId the session wikitty id
     */
    @Override
    public void generatePDFForSession(String sessionId) throws VradiException {
        List<String> formsIdsToGenerate = new ArrayList<String>();

        Session session = wikittyProxy.restore(Session.class, sessionId);

        Set<String> sendingIds = session.getSending();

        if (sendingIds == null) {
            return;
        }
        
        for (String id : sendingIds) {
            Sending sending = wikittyProxy.restore(Sending.class, id);

            if (sending != null) {
                Set<String> formsIds = SendingHelper.extractForms(wikittyProxy, sending.getGroupForms());

                if (formsIds != null) {
                    for (String formId : formsIds) {

                        if (!formsIdsToGenerate.contains(formId)) {
                            formsIdsToGenerate.add(formId);
                        }
                    }
                }
            }
        }

        for (String formId : formsIdsToGenerate) {
            try {
                generatePDF(formId, true);
            } catch (VradiException eee) {
                throw new VradiException("An error occurred while generating PDF file for " +
                        "form : " + formId, eee);
            }
        }
    }

    /**
     * Get all the groups a user is in.
     * @param userId the user we want the groups of
     * @return the group for our user
     * @throws VradiException for various possible errors
     */
    public List<Group> getGroupsForUser(String userId) throws VradiException {
        return clientManager.getGroupsForUser(userId);
    }

    @Override
    public String exportClientDB() throws VradiException {

        // creation du criteria wikitty
        // (export de tous les wikitty avec extension "client"
        // "groupement" ou "user"
        Search restriction = Search.query().or(); //definit un "groupement" qui se fera en "or"
        restriction.eq(Element.ELT_EXTENSION, Group.EXT_GROUP);
        restriction.eq(Element.ELT_EXTENSION, Client.EXT_CLIENT);
        restriction.eq(Element.ELT_EXTENSION, User.EXT_USER);
        Criteria criteria = restriction.criteria();

        // export
        String vscContent = exportAsCSV(criteria);

        return vscContent;
    }

    @Override
    public void deleteSession(String sessionId) throws VradiException {
        formManager.deleteSession(sessionId);
    }

    @Override
    public boolean checkTemplateExist(String extensionName) throws VradiException {
        boolean fileExists;

        try {
            WikittyExtension extension = formTypeManager.getFormType(extensionName);

            String templateName = formTypeManager.getFormTypeTemplateName(extension);

            if (templateName == null) {
                return false;
            }

            // just verify file existence
            // File file = storageService.getTemplate(extensionName, template);
            File template = formTypeManager.getTemplate(extensionName, templateName);

            fileExists = (template != null && template.exists()); 
        } catch (Exception eee) {
            throw new VradiException("Can't check if template existe : ", eee);
        }
        return fileExists;
    }

    @Override
    public boolean removeFormsFromSession(String sessionId, List<String> formsIds) throws VradiException {
        return formManager.removeFormsFromSession(sessionId, formsIds);
    }

    @Override
    public Sending revertDeletion(String sendingId, List<String> formsIdsToUpdate) {
        return formManager.revertDeletion(sendingId, formsIdsToUpdate);
    }

    @Override
    public Sending revertUserDeletion(String sessionId, String userId, String formId) {
        return formManager.revertUserDeletion(sessionId, userId, formId);
    }

    @Override
    public List<Sending> revertGroupDeletion(String sessionId, String groupId, String formId) {
        return formManager.revertGroupDeletion(sessionId, groupId, formId);
    }
}
