/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: BindingManager.java 1807 2010-11-24 15:11:33Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.1/vradi-services/src/main/java/com/jurismarches/vradi/services/managers/BindingManager.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services.managers;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import com.jurismarches.vradi.VradiServiceConfigurationHelper;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.Namespace;
import org.jdom.input.SAXBuilder;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;
import org.webharvest.definition.ScraperConfiguration;
import org.webharvest.runtime.Scraper;
import org.webharvest.runtime.variables.Variable;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.beans.XmlStreamImportResult;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.util.SSLUtils;

/**
 * Class containing the methods to manage the binding of the xml streams fields
 * with the form fields :
 * - xml field bindings creation, update and retrieving
 * - xml streams retrieving
 * - form creation with the data from an xml stream
 *
 * @author schorlet
 * @date 2010-01-22 20:18:29
 * @version $Revision: 1807 $ $Date: 2010-11-24 16:11:33 +0100 (mer., 24 nov. 2010) $
 */
public class BindingManager {
    private static final Log log = LogFactory.getLog(BindingManager.class);

    protected ApplicationConfig config;
    protected WikittyProxy wikittyProxy;
    protected FormTypeManager formTypeManager;
    protected FormManager formManager;
    protected Timer timer;
    protected TimerTask xmlStreamTask;

    static {
        // used for xml stream on https/ssl
        SSLUtils.intallCertificateTruster();
    }

    public BindingManager(ApplicationConfig config,
                          WikittyProxy wikittyProxy,
                          FormTypeManager formTypeManager,
                          FormManager formManager) {

        this.config = config;
        this.wikittyProxy = wikittyProxy;
        this.formTypeManager = formTypeManager;
        this.formManager = formManager;
    }

    /**
     * Retrieves the xml field bindings whose xml stream is xmlStream
     *
     * @param xmlStream the xml stream associated with the xml field bindings
     * we want to retrieve
     * @return a list containing the xml field bindings associated with
     * the xml stream xmlStream
     */
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();

        if (xmlStream != null && xmlStream.getXmlFieldBinding() != null) {
            if (log.isDebugEnabled()) {
                log.debug("getXmlFieldBindings(" + xmlStream.getName() + ")");
                log.debug(xmlStream.getXmlFieldBinding());
            }

            List<String> bindings = new ArrayList<String>();
            bindings.addAll(xmlStream.getXmlFieldBinding());
            
            List<XmlFieldBinding> restore = wikittyProxy.restore(XmlFieldBinding.class, bindings);
            list.addAll(restore);
        }

        return list;
    }

    /**
     * Retrieves the xml field binding whose id is xmlFieldBindingId
     *
     * @param xmlFieldBindingId the id of the xml field binding we want to retrieve
     * @return the xml field binding whose id is xmlFieldBindingId
     */
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBinding(" + xmlFieldBindingId + ")");
        }
        
        XmlFieldBinding xmlFieldBinding = wikittyProxy.restore(XmlFieldBinding.class, xmlFieldBindingId);
        return xmlFieldBinding;
    }

    /**
     * Retourne toutes les instance de stream avec leur réelle entité.
     */
    public List<XmlStream> getAllXmlStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllXmlStreams()");
        }
        
        List<XmlStream> result = new ArrayList<XmlStream>();

        // XmlStream
        Criteria criteriaXmlStream = Search.query()
                .eq(Element.ELT_EXTENSION, XmlStream.EXT_XMLSTREAM)
                .neq(Element.ELT_EXTENSION, WebHarvestStream.EXT_WEBHARVESTSTREAM)
                .criteria();
        
        PagedResult<XmlStream> xmlStreams = wikittyProxy.findAllByCriteria(XmlStream.class, criteriaXmlStream);
        result.addAll(xmlStreams.getAll());
        
        // XmlStream
        Criteria criteriaWebHarvestStream = Search.query()
                .eq(Element.ELT_EXTENSION, XmlStream.EXT_XMLSTREAM)
                .eq(Element.ELT_EXTENSION, WebHarvestStream.EXT_WEBHARVESTSTREAM)
                .criteria();
        
        PagedResult<WebHarvestStream> xmlWebHarvestStream = wikittyProxy.findAllByCriteria(WebHarvestStream.class, criteriaWebHarvestStream);
        result.addAll(xmlWebHarvestStream.getAll());

        return result;
    }

    public List<WebHarvestStream> getAllWebHarvestStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllWebHarvestStreams()");
        }

        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, WebHarvestStream.EXT_WEBHARVESTSTREAM)
                .criteria();

        PagedResult<WebHarvestStream> xmlStreams = wikittyProxy.findAllByCriteria(WebHarvestStream.class, criteria);
        List<WebHarvestStream> all = xmlStreams.getAll();

        List<WebHarvestStream> list = new ArrayList<WebHarvestStream>();
        list.addAll(all);

        return list;
    }

    public URI getWebHarvestPreviewUrl(WebHarvestStream stream) throws VradiException {
        String path = getWebHarvestPreviewPath(stream);
        File previewFile = new File(path);

        // If doesnt exist, creating one
        if (!previewFile.exists()) {
            if (log.isDebugEnabled()) {
                log.debug("Preview file doesnt exist, creating one for script path : " + path);
            }
            previewFile = createPreviewFile(stream);
        }

        URI uri = previewFile.toURI();
        return uri;
    }

    public String getWebHarvestPreviewUrlAsString(WebHarvestStream stream) throws VradiException {
        String uri = getWebHarvestPreviewUrl(stream).toString();

        if (log.isDebugEnabled()) {
            log.debug("Previews url is : " + uri);
        }

        return uri;
    }

    public String getWebHarvestPreviewPath(WebHarvestStream stream) {
        File scriptFile = new File(VradiServiceConfigurationHelper.getWebHarvestScriptDir(config) +
                File.separator + stream.getScriptUrl());

        String previewPath = VradiServiceConfigurationHelper.getWebHarvestPreviewDir(config) +
                File.separator +
                "Preview" +
                scriptFile.getName();

        if (log.isDebugEnabled()) {
            log.debug("Previews path is : " + previewPath);
        }

        return previewPath;
    }

    public XmlStream getXmlStream(String xmlStreamId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getXmlStream(" + xmlStreamId + ")");
        }

        XmlStream xmlStream = wikittyProxy.restore(XmlStream.class, xmlStreamId);
        return xmlStream;
    }

    private static class BindingContext {
        int dateParsingError = 0;
        int numberParsingError = 0;
        int nbCreated = 0;
    }

    /**
     * Create from from feed element.
     * 
     * @param formType
     * @param bindings
     * @param feed
     * @param namespace namespace for getting correct field with fieldnames
     * @param bindingContext
     * @return
     */
    protected FormImpl createForm(WikittyExtension formType, List<XmlFieldBinding> bindings,
             org.jdom.Element feed, Namespace namespace, BindingContext bindingContext) throws VradiException {
        FormImpl form = new FormImpl();
        Wikitty wikitty = form.getWikitty();
        wikitty.addExtension(formType);
        
        for (XmlFieldBinding binding : bindings) {
            String fqFormField = binding.getFormField();
            FieldType fieldType;
            try {
                fieldType = wikitty.getFieldType(fqFormField);
            } catch (Exception e) {
                continue;
            }
            
            fillFormField(wikitty, fieldType, binding, feed, namespace, bindingContext);
        }
        
        return form;
    }

    /**
     * Fill form with feed element.
     * 
     * @param wikitty
     * @param fieldType
     * @param binding
     * @param feed
     * @param namespace namespace to get field from feed element
     * @param bindingContext
     */
    protected void fillFormField(Wikitty wikitty, FieldType fieldType, XmlFieldBinding binding,
            org.jdom.Element feed, Namespace namespace, BindingContext bindingContext) throws VradiException {
        
        String fqFormField = binding.getFormField();
        Set<String> xmlFields = binding.getXmlField();

        if (xmlFields == null || xmlFields.isEmpty()) {
            // no mapping
            String defaultValue = binding.getDefaultValue();
            fillFormField2(wikitty, fieldType, fqFormField, defaultValue, bindingContext);
            return;
        }
            
        for (String xmlField : xmlFields) {
            org.jdom.Element child = feed.getChild(xmlField, namespace);
            String feedValue = null;
            
            // get feed field text
            if (child != null) {
                feedValue = child.getTextTrim();
            }
            
            // get default value
            if (feedValue == null || feedValue.isEmpty()) {
                feedValue = binding.getDefaultValue();
            }
            
            fillFormField2(wikitty, fieldType, fqFormField, feedValue, bindingContext);
        }
    }
    
    private void fillFormField2(Wikitty wikitty, FieldType fieldType, String fqFormField,
            String feedValue, BindingContext bindingContext) throws VradiException {
        // if no value then return
        if (feedValue == null || feedValue.isEmpty()) {
            return;
        }

        switch (fieldType.getType()) {
        case DATE:
            Date date = DateParser.parse(feedValue);
            if (date != null) {
                try {
                    wikitty.setFqField(fqFormField, WikittyUtil.formatDate(date));
                } catch (ParseException eee) {
                    throw new VradiException("Cant parse date " + date, eee);
                }
            } else {
                bindingContext.dateParsingError++;
            }
            break;

        case NUMERIC:
            if (NumberUtils.isNumber(feedValue)) {
                Double value = Double.valueOf(feedValue);
                 wikitty.setFqField(fqFormField, value);
            } else {
                bindingContext.numberParsingError++;
            }
            break;

        default:
            Object fieldValue = wikitty.getFqField(fqFormField);
            String newValue = null;

            if (fieldValue != null) {
                newValue = fieldValue + "\n" + feedValue;
            } else {
                newValue = feedValue;
            }

             wikitty.setFqField(fqFormField, newValue);
        }
     }
    
    /**
     * Creates and store forms from an {@code WebHarvestStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param webHarvestStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    public XmlStreamImportResult importFormsFromWebHarvestStream(WebHarvestStream webHarvestStream) throws VradiException {

        log.info("getFormsFromWebHarvestStream for " + webHarvestStream.getName());

        // Create file
        File previewFile = createPreviewFile(webHarvestStream, true);

        Document document = null;
        try {
            SAXBuilder sxb = new SAXBuilder();
            document = sxb.build(previewFile);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't read xml stream", e);
            }
            throw new VradiException("Can't read xml stream : ", e);
        }
        return getFormsFromStream(webHarvestStream, document);
    }

    /**
     * Creates file result of webharvest scrip
     *
     * @param webHarvestStream
     * @return Preview file
     * @throws VradiException for various possible errors
     */
    public File createPreviewFile(WebHarvestStream webHarvestStream) throws VradiException{
        return createPreviewFile(webHarvestStream, false);
    }

    // Create file if necessary
    protected File createPreviewFile(WebHarvestStream webHarvestStream, boolean override) throws VradiException{
        String path = webHarvestStream.getScriptUrl();
        if (path == null) {
            return null;
        }

        // Get script path
        path = VradiServiceConfigurationHelper.getWebHarvestPreviewDir(config) +
                File.separator +
                path;

        File scriptFile = new File(path);

        if (!override && scriptFile.exists()) {
            return scriptFile;
        }

        // Init webharvest
        ScraperConfiguration scrapperConfig;
        try {
            scrapperConfig = new ScraperConfiguration(scriptFile);
        } catch (FileNotFoundException eee) {
            throw new VradiException("Cant open script " + webHarvestStream.getScriptUrl(), eee);
        }
        Scraper scraper = new Scraper(scrapperConfig, VradiServiceConfigurationHelper.getDataDir(config));

        scraper.setDebug(log.isDebugEnabled());

        scraper.addVariableToContext("url", webHarvestStream.getUrl());

        long startTime = System.currentTimeMillis();
        scraper.execute();
        log.info("Script execution time elapsed: " + (System.currentTimeMillis() - startTime));

        // takes variable created during execution
        Variable resultVar = (Variable) scraper.getContext().get("result");
        String result = resultVar.toString();

        result = result.replaceAll("&", "&amp;");

        File previewFile = new File(path);

        try {
            FileUtils.writeStringToFile(previewFile, result);
            log.info("Store preview for script " + scriptFile.getPath());
        } catch (IOException eee) {
            log.error("Failed to write preview for script " + scriptFile.getName(), eee);
        }

        return previewFile;
    }

    /**
     * Creates and store forms from an {@code XmlStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param xmlStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    public XmlStreamImportResult importFormsFromXmlStream(XmlStream xmlStream) throws VradiException {
        Document document;
        try {
            
            // rome : but not done with rome :(
            //SyndFeedInput input = new SyndFeedInput();
            //URL rssUrl = new URL(xmlStream.getUrl());
            //SyndFeed feed = input.build(new XmlReader(rssUrl));

            SAXBuilder sxb = new SAXBuilder(false);
            
            // this prevent xerces to look for external dtd
            sxb.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);

            URL rssUrl = new URL(xmlStream.getUrl());
            log.info("Reading xmlStream url from: " + rssUrl);
            document = sxb.build(rssUrl);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't read xml stream", e);
            }
            throw new VradiException("Can't read xml stream : ", e);
        }
        return getFormsFromStream(xmlStream, document);
    }
    
    /**
     * Creates and store forms from an {@code stream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param <E> xml stream type ({@link XmlStream} or {@link WebHarvestStream}).
     * @param stream stream to import
     * @param document of rss
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    public <E extends XmlStream> XmlStreamImportResult getFormsFromStream(E stream, Document document) throws VradiException {

        XmlStreamImportResult result = new XmlStreamImportResult();

        if (log.isDebugEnabled()) {
            log.debug("getFormsFromStream(" + stream.getName());
        }

        if (stream.getFormTypeName() == null) {
            throw new VradiException("xmlStream.formTypeName is null");
        }
        
        WikittyExtension formType = formTypeManager.getFormType(
                stream.getFormTypeName());
        if (formType == null) {
            throw new VradiException("Extension of name xmlStream.formTypeName does not exists");
        }

        org.jdom.Element racine = document.getRootElement();
        // namespace should be used, otherwize, getChild(String) return
        // nothing in some feeds:
        // ex : http://www.gentoo.org/rdf/en/gentoo-news.rdf
        // https://labs.libre-entreprise.org/export/rss_sfnews.php
        Namespace namespace = racine.getNamespace("");
        List<org.jdom.Element> elements = null;

        // must start with ITEM :
        // some feed have both channel and item
        // http://www.gentoo.org/rdf/en/gentoo-news.rdf
        if (racine.getChild(VradiConstants.ITEM, namespace) != null) {
            elements = racine.getChildren(VradiConstants.ITEM, namespace);
        } else if (racine.getChild(VradiConstants.CHANNEL, namespace) != null) {
            org.jdom.Element channel = racine.getChild(VradiConstants.CHANNEL, namespace);
            elements = channel.getChildren(VradiConstants.ITEM, namespace);
        } else  if (racine.getChild(VradiConstants.ENTRY, namespace) != null) {
            elements = racine.getChildren(VradiConstants.ENTRY, namespace);
        }

        if (elements == null) {
            if (log.isWarnEnabled()) {
                log.warn("Enable to find items or entries in stream");
            }
            return result;
        }

        List<Form> forms = new ArrayList<Form>();
        List<String> xmlFieldBindingIds = new ArrayList<String>(stream.getXmlFieldBinding());
        List<XmlFieldBinding> bindings = wikittyProxy.restore(XmlFieldBinding.class, xmlFieldBindingIds);
        String formDateId = VradiConstants.FORM_ID_DATE_FORMAT.format(new Date());
        String toTreatId = formManager.getNonTraiteStatus().getWikittyId();
        BindingContext bindingContext = new BindingContext();

        for (int index = 0 ; index < elements.size() ; index++) {
            org.jdom.Element feedElement = elements.get(index);

            // calculate element content sha1 sum
            StringBuffer sb = new StringBuffer();
            List<org.jdom.Element> fields = feedElement.getChildren();
            for (org.jdom.Element field : fields) {
                sb.append(field.getText());
            }
            String contentSHA1Hash = StringUtil.encodeSHA1(sb.toString());

            // check if one element with this hash already exists
            Criteria criteria = Search.query().eq(Form.FQ_FIELD_FORM_IMPORTCONTENTHASH, contentSHA1Hash).criteria();
            criteria.setEndIndex(0);
            PagedResult<Form> pagesResult = wikittyProxy.findAllByCriteria(Form.class, criteria);
            if (pagesResult.getNumFound() > 0) {
                // TODO EC20100928 sha1 collision found
                // check form field to detect proper content collision
                if (log.isWarnEnabled()) {
                    log.warn("SHA1 content collision detected");
                }

                // for now, just break
                break;
            }

            // create the form with the info from the xml stream
            FormImpl form = createForm(formType, bindings, feedElement, namespace, bindingContext);
            bindingContext.nbCreated++;

            form.setId(formDateId + form.getWikittyId());
            form.setXmlStream(stream.getWikittyId());
            form.setStatus(toTreatId);
            form.setImportContentHash(contentSHA1Hash);
            forms.add(form);

            if (forms.size() > 100) {
                formManager.updateForms(forms, null);
                forms.clear();
            }
        }

        if (!forms.isEmpty()) {
            formManager.updateForms(forms, null);
        }

        result.setCreatedFormCount(bindingContext.nbCreated);
        result.setDateParsingError(bindingContext.dateParsingError);
        result.setNumberParsingError(bindingContext.numberParsingError);

        // equals to : elements.size() - result.getCreatedFormCount()
        int alreadyExists = elements.size() - result.getCreatedFormCount();
        result.setAlreadyExistsFormCount(alreadyExists);

        // TODO EC-20100428 : redirect log output into file
        if (log.isInfoEnabled()) {
            log.info("Form import from stream, created = " + result.getCreatedFormCount());
            log.info("Form import from stream, already existing = " + result.getAlreadyExistsFormCount());
            log.info("Form import from stream, dateParsingError = " + result.getDateParsingError());
            log.info("Form import from stream, numberParsingError = " + result.getNumberParsingError());
        }

        return result;
    }
}
