/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiConstants.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://tchemit@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3/vradi-services/src/main/java/com/jurismarches/vradi/VradiConstants.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi;

import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;

import java.text.SimpleDateFormat;
import java.util.regex.Pattern;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * Vradi application constants.
 *
 * @author $Author: tchemit $
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 * @since 2 mars 2010 15:37:19
 */
public class VradiConstants {

    /**
     * Pattern de l'heure d'import des flux xml.
     * 
     * Autorisé :
     * 2:13
     */
    public static final Pattern XML_STREAM_IMPORT_TIME = Pattern.compile("^(\\d{1,2})\\:(\\d{1,2})$");

    // names of feed elements
    public static final String CHANNEL = "channel";
    public static final String ENTRY = "entry";
    public static final String ITEM = "item";

    // Name of context to get client user decorator 
    public static final String USER_ONLY_DECORATOR = "userOnly";

    /**
     * Form Id date format.
     */
    public static final SimpleDateFormat FORM_ID_DATE_FORMAT =
            new SimpleDateFormat("yyyy-MM-dd");

    /**
     * default value of the sending paragraph.
     */
    public static final String DEFAULT_SENDING_PARAGRAPH = "";

    /**
     * default value of the sending status.
     */
    public static final int DEFAULT_SENDING_STATUS = 0;

    /**
     * Forms Status.
     */
    public enum FormStatus {
        NON_TRAITE(n_("vradi.status.nontraite.name"),n_("vradi.status.nontraite.description")),
        NON_SELECTIONNE(n_("vradi.status.nonselectionne.name"), n_("vradi.status.nonselectionne.description")),
        PRESELECTIONNE(n_("vradi.status.preselectionne.name"), n_("vradi.status.preselectionne.description")),
        SELECTIONNE(n_("vradi.status.selectionne.name"), n_("vradi.status.selectionne.description"));

        private String name;
        private String description;

        private FormStatus(String name, String description) {
            this.name = name;
            this.description = description;
        }

        public String getName() {
            return name;
        }

        public String getDescription() {
            return description;
        }

        public int getValue() {
            return ordinal();
        }
    }

    /**
     * Sending Status.
     */
    public enum SendingStatus {
        DELETED(_("vradi.email.status.deleted")),
        TO_SEND(_("vradi.email.status.toSend")),
        SENT(_("vradi.email.status.sent")),
        WAITING_RECEPTION_PROOF(_("vradi.email.status.waitingReceptionProof")),
        RECEIVED(_("vradi.email.status.received")),
        ERROR(_("vradi.email.status.error"));

        private final String description;
        
        private SendingStatus(String description) {
            this.description = description;
        }

        public String getDescription() {
            return description;
        }

        public int getValue() {
            // Deleted must return -1
            return ordinal() - 1;
        }

        public static SendingStatus getStatus(int i){
            for (SendingStatus status : values()){
                if (status.getValue() == i){
                    return status;
                }
            }
            return null;
        }

        // Status methods utility for sending
        public static boolean isSent(Sending sending) {
            int status = sending.getStatus();
            return status == SENT.getValue() ||
                   status == WAITING_RECEPTION_PROOF.getValue() ||
                   status == RECEIVED.getValue();
        }

        public static boolean isDeleted(Sending sending) {
            int status = sending.getStatus();
            return status == DELETED.getValue();
        }

        public static boolean isToSend(Sending sending) {
            int status = sending.getStatus();
            return status == TO_SEND.getValue() ||
                   status == ERROR.getValue();
        }

        public static boolean isWaitingReceptionProof(Sending sending) {
            int status = sending.getStatus();
            return status == WAITING_RECEPTION_PROOF.getValue();
        }

        public static boolean isInError(Sending sending) {
            int status = sending.getStatus();
            return status == ERROR.getValue();
        }

        public static boolean isClosed(Sending sending) {
            return isDeleted(sending) || isSent(sending);
        }
    }

    /**
     * Session status.
     */
    public enum SessionStatus {
        ACTIVE(_("vradi.sessionStatus.active")),
        STOPPED(_("vradi.sessionStatus.stopped")),
        SENT(_("vradi.sessionStatus.sent")),
        ERROR(_("vradi.sessionStatus.error"));

        protected final String description;

        private SessionStatus(String description) {
            this.description = description;
        }

        public String getDescription() {
            return description;
        }

        public static SessionStatus getStatus(int i) {
            SessionStatus result = null;
            for (SessionStatus status : values()) {
                if (status.ordinal() == i) {
                    result = status;
                }
            }
            return result;
        }

        public int getValue() {
            return ordinal();
        }

        // Status methods utility for session
        public static boolean isActive(Session session) {
            if (session == null) {
                return false;
            }
            int status = session.getStatus();
            return status == ACTIVE.getValue();
        }

        public static boolean isSent(Session session) {
            int status = session.getStatus();
            return status == SENT.getValue();
        }

        public static boolean isStopped(Session session) {
            int status = session.getStatus();
            return status == STOPPED.getValue();
        }

        public static boolean isToRestart(Session session) {
            int status = session.getStatus();
            return status == STOPPED.getValue() ||
                   status == ERROR.getValue();
        }

        public static boolean isToSend(Session session) {
            int status = session.getStatus();
            return status == ACTIVE.getValue() ||
                   status == STOPPED.getValue() ||
                   status == ERROR.getValue();
        }
    }

    /**
     * Form type enum used to xml file markup list.
     */
    public enum FormTypeTemplateEnum {
        FIELDS_ELEMENT("fields"),
        FIELD_ELEMENT("field"),
        FIELD_TEMPLATE_ATTRIBUTE("template"),
        FIELD_EXTENSION_ATTRIBUTE("extension");

        private final String value;

        private FormTypeTemplateEnum(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }
    
    /**
     * All file type managed by vradi.
     */
    public enum FileType {
        /** Template files (ott, odt). */
        TEMPLATE,
        /** Files embedded into generated PDF. */
        EMBEDDED_FORM_FILES,
        /** Files attached to sended mails. */
        ATTACHMENTS_FORM_FILES,
        /** Generated pdf files. */
        PDF("application/pdf"),
        /** Web harvest script files. */
        WEB_HARVEST,
        /** Request history files. */
        QUERY_HISTORY("application/rss+xml"),
        /** Temp files */
        TEMP_FILE;

        private final String mineType;


        private FileType() {
            this("application/octet-stream");
        }

        private FileType(String mineType) {
            this.mineType = mineType;
        }

        public String getMineType() {
            return mineType;
        }
    }
}
