/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: TasksManager.java 1533 2010-09-29 15:25:42Z chatellier $
 * $HeadURL: svn+ssh://chatellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.2.0/vradi-services/src/main/java/com/jurismarches/vradi/services/tasks/TasksManager.java $
 * %%
 * Copyright (C) 2010 JurisMarches, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi.services.tasks;

import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyServiceEvent;
import org.nuiton.wikitty.WikittyServiceListener;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.managers.BindingManager;
import com.jurismarches.vradi.services.managers.MailingManager;

/**
 * This manager handle tasks management in
 * {@code com.jurismarches.vradi.services.tasks} package.
 * 
 * @author chatellier
 * @version $Revision: 1533 $
 * 
 * Last update : $Date: 2010-09-29 17:25:42 +0200 (mer., 29 sept. 2010) $
 * By : $Author: chatellier $
 */
public class TasksManager implements WikittyServiceListener {

    private static final Log log = LogFactory.getLog(TasksManager.class);

    /** Cron task to check mails every 10 minutes. */
    protected Timer mailCronTimer;

    /** Cron task to import streams. */
    protected ScheduledExecutorService xmlStreamScheduler;

    protected ReceiveMailTasks receiveMailTasks;

    /**
     * Map entre les id des flux, et les taches gérant.
     */
    protected Map<String, ScheduledFuture<?>> xmlStreamImportTasks;

    protected WikittyProxy wikittyProxy;

    protected MailingManager mailingManager;

    protected BindingManager bindingManager;

    public TasksManager(WikittyProxy wikittyProxy, MailingManager mailingManager, BindingManager bindingManager) {
        this.wikittyProxy = wikittyProxy;
        this.mailingManager = mailingManager;
        this.bindingManager = bindingManager;
        xmlStreamImportTasks = new HashMap<String, ScheduledFuture<?>>();
    }

    /**
     * Init all tasks.
     */
    public void initTasks() {
        initMailTask();
        initXmlStreamTasks();
    }

    /**
     * Init and start mail tasks (check every 10 minutes)
     */
    protected void initMailTask() {
        mailCronTimer = new Timer("vradi-mail-task");
        receiveMailTasks = new ReceiveMailTasks(mailingManager);
        mailCronTimer.schedule(receiveMailTasks, 0, 10 * 60 * 1000);
    }

    /**
     * Init and start xml stream tasks.
     */
    protected void initXmlStreamTasks() {

        // les imports sont fait sequentiellement
        xmlStreamScheduler = Executors.newScheduledThreadPool(1);

        List<XmlStream> xmlStreams = bindingManager.getAllXmlStreams();

        for (XmlStream xmlStream : xmlStreams) {
            initXmlStreamTask(xmlStream);
        }
    }

    /**
     * Init single xml stream task.
     * 
     * @param xmlStream xml stream
     */
    protected void initXmlStreamTask(XmlStream xmlStream) {
        String streamImportTime = xmlStream.getImportTime();

        if (StringUtils.isBlank(streamImportTime)) {
            // par defaut minuit, si pas de données
            streamImportTime = "0:0";
        }

        Matcher matchImportTime = VradiConstants.XML_STREAM_IMPORT_TIME.matcher(streamImportTime);
        if (matchImportTime.find()) {
            String hours = matchImportTime.group(1);
            String minutes = matchImportTime.group(2);

            // calcule le nombre de minutes entre maitenant
            // et l'heure d'import du flux
            Calendar nowCalendar = Calendar.getInstance();
            int nowHours = nowCalendar.get(Calendar.HOUR_OF_DAY);
            int nowMinutes = nowCalendar.get(Calendar.MINUTE);

            int toHours = Integer.parseInt(hours);
            int toMinutes = Integer.parseInt(minutes);

            int delayInMinute = toHours * 60 + toMinutes - (nowHours * 60 + nowMinutes);
            if (delayInMinute < 0) {
                delayInMinute += 24 * 60;
            }

            // TODO EC20100921 prevoir le cas où une tâche dure plus d'une journée
            ImportXmlStreamTasks importTask = new ImportXmlStreamTasks(wikittyProxy, bindingManager, xmlStream.getWikittyId());
            ScheduledFuture<?> sheduledImportTask = xmlStreamScheduler.scheduleAtFixedRate(importTask, delayInMinute, 24 * 60, TimeUnit.MINUTES);
            xmlStreamImportTasks.put(xmlStream.getWikittyId(), sheduledImportTask);

            if (log.isDebugEnabled()) {
                log.debug("Starting import of xml stream " + xmlStream.getName() + " at " + streamImportTime);
                log.debug(" - in scheduler, delai : " + delayInMinute + "min, period " + (24 * 60) + "min");
            }
        }
        else {
            if (log.isWarnEnabled()) {
                log.warn("Can't parse " + streamImportTime + " for stream " + xmlStream.getWikittyId());
            }
        }
    }

    /**
     * Cancel import task for specified id and restart a new task if requiered.
     * 
     * @param xmlStreamId xml stream id
     * @param restart restart a new import task for stream id
     */
    protected void cancelXmlImportTask(String xmlStreamId, boolean restart) {

        // just cancel task
        if (log.isDebugEnabled()) {
            log.debug("Canceling import stream task for " + xmlStreamId);
        }
        ScheduledFuture<?> sheduledImportTask = xmlStreamImportTasks.get(xmlStreamId);
        if (sheduledImportTask != null) {
            xmlStreamImportTasks.remove(xmlStreamId);
            sheduledImportTask.cancel(false);
        }
        else {
            // ca peut arriver lors de la creation de stream (pas de précédente)
            if (log.isDebugEnabled()) {
                log.debug("Nothing to cancel for wikitty id " + xmlStreamId);
            }
        }

        // restart when stream is updated
        if (restart) {
            XmlStream xmlStream = wikittyProxy.restore(XmlStream.class, xmlStreamId);
            if (xmlStream == null) {
                if (log.isErrorEnabled()) {
                    log.error("Can't restart import task for non existing xml stream " + xmlStreamId);
                }
            }
            else {
                initXmlStreamTask(xmlStream);
            }
        }
    }

    @Override
    public void putWikitty(WikittyServiceEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("Receiving wikitty event : " + event);
        }

        // ecoute les modification faites sur les formulaires
        Map<String, Set<String>> wikittyExtensions = event.getIdExtensions();
        for (Map.Entry<String, Set<String>> wikittyExtension : wikittyExtensions.entrySet()) {
            String wikittyId = wikittyExtension.getKey();
            Set<String> extensions = wikittyExtension.getValue();
            if (extensions.contains(XmlStream.EXT_XMLSTREAM)) {
                cancelXmlImportTask(wikittyId, true);
            }
        }
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("Receiving wikitty event : " + event);
        }

        // ecoute les modification faites sur les formulaires
        // on ne peut pas savoir si c'est un xmlstream
        // mais on c'est si l'id est lancé
        Set<String> wikittyIds = event.getIds();
        for (String wikittyId : wikittyIds) {
            // if pas indispensable, mais ca evite des log inutiles
            if (xmlStreamImportTasks.containsKey(wikittyId)) {
                cancelXmlImportTask(wikittyId, false);
            }
        }
    }

    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        
    }

    @Override
    public void putExtension(WikittyServiceEvent event) {
        
    }

    @Override
    public void removeExtension(WikittyServiceEvent event) {
        
    }

    @Override
    public void clearExtension(WikittyServiceEvent event) {
        
    }
}
