/*
 * #%L
 * Vradi :: Services
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.VradiException;

/**
 * This class reference all action run by Vradi Service :
 *
 * -- reIndexSolr : reindex solr index
 *
 * @author sletellier <letellier@codelutin.com>
 */
public class VradiServiceAction {

    /** log. */
    private static final Log log = LogFactory.getLog(VradiServiceAction.class);

    /**
     * Display help and exit action.
     */
    public void help() {
        System.out.println("Usage: vradi-services [-h]");
        System.out.println("Options (set with --option <key> <value>) :");
        for (VradiServiceConfiguration.Option o : VradiServiceConfiguration.Option.values()) {
            System.out.println("\t" + o.key + "(" + o.defaultValue + ") : " + o.description);
        }
        
        System.out.println("Actions:");
        for (VradiServiceConfiguration.ActionDefinition a : VradiServiceConfiguration.ActionDefinition.values()) {
            System.out.println("\t" + Arrays.toString(a.getAliases()) + "(" + a.getAction() + ") : " + a.getDescription());
        }
        System.exit(0);
    }

    /**
     * Reload solr index.
     * 
     * TODO EC20100920 : peut etre desactiver la notification pendant la reindexation
     * TODO EC20100920 : peut etre desactiver la reindexation automatique du changement de version
     */
    public static void reIndexSolr() {

        if (log.isInfoEnabled()) {
            log.info("Starting reIndex solr");
        }

        long startingTime = System.nanoTime();

        ServiceFactory.getWikittyProxy().syncEngin();

        if (log.isInfoEnabled()) {
            log.info("ReIndex solr is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));
        }
        System.exit(0);
    }

    /**
     * Delete obselete thesaurus.
     */
    public static void deleteObseleteThesaurus() {
        log.info("Starting deleting obselete thesaurus");

        final long startingTime = System.nanoTime();

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS).criteria();

        WikittyProxy proxy = ServiceFactory.getWikittyProxy();
        List<Thesaurus> thesauruses = proxy.findAllByCriteria(Thesaurus.class, criteria).getAll();

        for (Thesaurus thesaurus : thesauruses) {
            String parentId = thesaurus.getParent();
            if (parentId == null) {
                proxy.delete(thesaurus.getWikittyId());

                log.info("Found thesaurus '" + thesaurus.getName() + "' has no parentId, deleting this");
            } else {
                Thesaurus parent = proxy.restore(Thesaurus.class, parentId);
                if (parent == null) {
                    proxy.delete(thesaurus.getWikittyId());
                    log.info("Found thesaurus '" + thesaurus.getName() + "' has no parent with id '" + parentId + ", deleting this");
                }
            }
        }

        log.info("Deleting obselete thesaurus is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));
        System.exit(0);
    }

    /**
     * Delete obselete status.
     */
    public static void reinitStatus() {
        log.info("Starting reinit status");

        final long startingTime = System.nanoTime();

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Status.EXT_STATUS).criteria();

        WikittyProxy proxy = ServiceFactory.getWikittyProxy();
        List<Status> statuses = proxy.findAllByCriteria(Status.class, criteria).getAll();
        List<String> statusesIds = new ArrayList<String>();

        for (Status status : statuses) {
            statusesIds.add(status.getWikittyId());
            log.info("Deleting status " + status.getName());
        }

        proxy.delete(statusesIds);

        try {
            ServiceFactory.getVradiStorageService().createDefaultStatuses();
        } catch (VradiException eee) {
            log.error(eee);
        }

        log.info("reinit status is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));
        System.exit(0);
    }

    /**
     * Force all old user to enable.
     */
    public static void forceAllOldUserEnable() {
        log.info("Starting forcing all user enabled");

        final long startingTime = System.nanoTime();

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER).criteria();

        WikittyProxy proxy = ServiceFactory.getWikittyProxy();
        List<User> users = proxy.findAllByCriteria(User.class, criteria).getAll();

        for (User user : users) {
            if (user == null) {
                break;
            }
            Object field = user.getField(User.EXT_USER, User.FIELD_USER_ENABLE);

            if (field == null) {
                log.info("Found user '" + user.getName() + "' with no enable");
                
                // FIXME EC20100910 remove direct access to wikitty
                user.setField(User.EXT_USER, User.FIELD_USER_ENABLE, true);
                proxy.store(user);
            }
        }

        log.info("Forcing all user enabled is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));
        System.exit(0);
    }

    /**
     * Delete all user with no client attached.
     */
    public static void deleteObseleteUsers() {
        log.info("Starting obselete user deletion");

        final long startingTime = System.nanoTime();

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER).criteria();

        WikittyProxy proxy = ServiceFactory.getWikittyProxy();
        List<User> users = proxy.findAllByCriteria(User.class, criteria).getAll();

        for (User user : users) {
            if (user == null) {
                break;
            }
            String clientId = user.getClient();

            if (clientId == null) {
                log.info("Deleting user " + user.getName() + " with no client attached");
                proxy.delete(user.getWikittyId());
            } else {
                Client client = proxy.restore(Client.class, clientId);
                if (client == null) {
                    log.info("Deleting user " + user.getName() + " with no client attached");
                    proxy.delete(user.getWikittyId());
                }
            }
        }

        log.info("Forcing all user enabled is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));
        System.exit(0);
    }
}
