/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlStream;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.WikittyExtension;

import com.jurismarches.vradi.beans.QueryBean;

/**
 * Vradi storage service.
 * 
 * This service is intended to be used in remote mode.
 * 
 * Typical usage is code that can be run only on server :
 * <ul>
 *  <li>Openoffice template generation
 *  <li>Mail sending
 * </ul>
 *
 * @author morin
 * @version $Revision: 1317 $ $Date: 2010-09-10 10:29:46 +0200 (ven., 10 sept. 2010) $
 */
public interface VradiStorageService {

    /**
     * Creation of default status
     *
     * Warn, dont use it otherwise for vradi initialisation !!!
     *
     * @return list of defautl status created
     * @throws VradiException
     */
    List<Status> createDefaultStatuses() throws VradiException;
    
    /**
     * Finds fields of a form type.
     *
     * @param name the name of the form type
     * @return a map containing the names of the fields and their type
     */
    Map<String, FieldType> getFormTypeFields(String name) throws VradiException;

    /**
     * Returns a form type with the specified <code>name</code>.
     *
     * @param name the name of the form type
     * @return a <code>WikittyExtension</code>
     */
    WikittyExtension getFormType(String name) throws VradiException;

    /**
     * Binds all queryMaker with the forms found by their queries
     * and not already sent.
     */
    Session bindForms(Session session) throws VradiException;

    /**
     * Create new Sending
     *
     * @param session
     * @param queryMaker
     * @param formsToBind
     * @return
     * @throws VradiException
     */
    List<Sending> createSending(Session session, QueryMaker queryMaker, List<Form> formsToBind) throws VradiException;

    Session createNewSession() throws VradiException;

    /**
     * Get all session by date
     *
     * @param sessionDate           Date concerned
     * @return                      a list of VradiSessionDTO
     * @throws VradiException
     */
    List<Session> getSessions(Date sessionDate) throws VradiException;

    /**
     * Get last closed session
     *
     * @return last opened session
     * @throws VradiException
     */
    Session getLastCloseSession() throws VradiException;

    /**
     * Get last opened session
     *
     * @return last opened session
     * @throws VradiException
     */
    Session getLastOpenSession() throws VradiException;

    /**
     * Creates and store forms from an <code>WebHarvestXmlStream</code> by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * @param webHarvestStream the web harvest xml stream to execute
     * @return an array containing :
     * - the number of created forms
     * - the number of already existing forms (non created)
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * TODO EC-20100428 return a serializable structure (easier to use)
     *
     * @throws VradiException for various possible errors
     */
    public int[] getFormsFromWebHarvestStream(WebHarvestStream webHarvestStream) throws VradiException;

    /**
     * Creates and store forms from an <code>XmlStream</code> by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * @param xmlStream        the xml stream to read
     * @return an array containing :
     * - the number of created forms
     * - the number of already existing forms (non created)
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     * 
     * TODO EC-20100428 return a serializable structure (easier to use)
     * 
     * @throws VradiException for various possible errors
     */
    int[] getFormsFromXmlStream(XmlStream xmlStream) throws VradiException;


    /**
     * Computes a list of queries returning the specified <code>Form</code>.
     *
     * @param form the form which match queries
     * @return a list of queries
     */
    Map<QueryMaker, List<QueryBean>> findQueriesReturningForm(Form form) throws VradiException;

    /**
     * Create a {@code VradiUser}.
     *
     * @param user the user to create
     * @return the vradiUser updated or {@code null} if user can't be created (already exists)
     * @throws VradiException
     */
    VradiUser createUser(VradiUser user) throws VradiException;

    /**
     * Checks if {@code userName} and {@code userPassword} matches a valid {@code VradiUser}.
     *
     * @param login the login of a WikittyUser
     * @param md5Password the password of a WikittyUser (md5 encoded)
     * @return a {@code WikittyUser} or null if either userName or userPassword don't match
     * @throws VradiException
     */
    VradiUser loginUser(String login, String md5Password) throws VradiException;

    /*
     * Import data from an XML file.
     *
     * @param file the XML file containing datas
     * @throws VradiException
     *
    void importData(File file) throws VradiException;

    /*
     * Export the datas into an XML format.
     *
     * @return the datas as an XML String
     * @throws VradiException
     *
    String exportData() throws VradiException;*/

    /*
     * Create or Update a list of <code>Forms</code>.
     *
     * @param forms the list of forms to insert or update
     * @return an updated list of <code>Form</code>
     *
    List<Form> updateForms(List<Form> forms) throws VradiException;*/

    /**
     * Import all data as CSV.
     * 
     * FIXME EC-20100415 will not work in remote mode (or uri maybe be
     * valid for remote server too)
     * 
     * @param uri uri used to read input stream
     * @throws VradiException if any error happen during import
     */
    void importAsCSV(String uri) throws VradiException;

    /**
     * Get csv export.
     * 
     * @param criteria export criteria
     * @return csv export
     * @throws VradiException if any error happen during export
     */
    String exportAsCSV(Criteria criteria) throws VradiException;

    /**
     * Reindex the datas.
     */
    void reindexData() throws VradiException;

    

    /**
     * Propose thesaurus nodes that might be in relation with a specified form
     *
     * @param form the <code>Form</code> containing the information needed
     * to search the thesaurus nodes
     * @return a list of <code>Thesaurus</code>
     * @throws VradiException
     */
    List<Thesaurus> proposeThesaurus(Form form)
            throws VradiException;

    /**
     * Archives the queries of a user, client or group in a RSS file.
     * @param queryMaker the query maker whose queries are archived
     * @throws VradiException
     */
    void archiveQueries(QueryMaker queryMaker) throws VradiException;

    /**
     * Regularly retrieves the information from all the xml streams
     * and create new forms
     *
     * @param intervalUnit unit of te interval between 2 retrievings (minute, hour or day)
     * @param intervalValue interval value between two retrievings
     * @param hour hour of the retrieving if the unit is day
     * @param minute of the retrieving if the unit is day or hour
     * @throws VradiException
     */
    void autoLoadFormsFromXmlStreams(String intervalUnit,
            int intervalValue, Integer hour, Integer minute)
            throws VradiException;

    /**
     * Lists the template filename (simple name) associated with the specified
     * extension.
     * 
     * This need to be called by a WS because an {@link WikittyExtension}
     * doesn't have a files fields.
     * 
     * @param extension
     * @return
     */
    Collection<String> getTemplateFilenames(WikittyExtension extension);

    /**
     * Save association between template fields and extensions fields.
     * 
     * @param extensionName
     * @param templateName
     * @param fieldMap
     * @throws VradiException
     */
    void setAssociatedFields(String extensionName, String templateName,
            Map<String, String> fieldMap) throws VradiException;

    /**
     * Get association between template fields and extension fields.
     * 
     * @param extensionName
     * @param templateName
     * @return association map or {@code null} is there is no association yet
     * @throws VradiException
     */
    Map<String, String> getAssociatedFields(String extensionName,
            String templateName) throws VradiException;

    /**
     * Generate pdf file for specified form.
     * 
     * @param form form to generate pdf
     * @param force force regeneration
     * @return uri to download file
     */
    String generatePDF(Form form, boolean force) throws VradiException;

    /**
     * Gets the query makers whose queries are potentially to modify
     * after a thesaurus node modification
     * @param rootThesaurusName the modified root thesaurus name node
     * @param thesaurusName the modified thesaurus name node
     * @return a map containing the query makers and their queries which contains
     * the thesaurus node name
     */
    Map<Group, List<QueryBean>> getQueriesToModifyAfterThesaurusModification(
            String rootThesaurusName, String thesaurusName);

    /**
     * Remove all sending corresponding with form and query maker from session
     * @param session concerned
     * @param form concerned
     * @param queryMaker concerned
     * @return List of sending after deletion
     */
    List<Sending> removeAllSending(Session session, Form form, QueryMaker queryMaker)
            throws VradiException;

    /**
     * Get web harvest preview url from script
     * @param stream
     * @return url of preview file
     */
    String getWebHarvestPreviewURL(WebHarvestStream stream) throws VradiException;
   
    /**
     * Get template document fields names.
     * 
     * @param extension extension
     * @param templateName template name
     * @return field names
     * @throws VradiException
     */
    List<String> getDocumentFields(WikittyExtension extension, String templateName) throws VradiException;

    Session sendMessages(String sessionId) throws VradiException;

    Sending sendMessage(String sendingId, String sessionParagraph) throws VradiException;

    Sending sendMessage(String sendingId, String sessionParagraph, String email) throws VradiException;

    void receiveMessages() throws VradiException;

    Session stopSentMail(Session session) throws VradiException;
    
    String[] getXmlStreamConfig();

    /**
     * Get all WebHarvest script name in server's WebHarvest directory.
     * 
     * All filename doesn't contains "/".
     * 
     * @return script file names
     */
    List<String> getAllWebHarvestScripts();

    /**
     * Generate all the PDFs needed to send a session.
     *
     * @param sessionId the session wikitty id
     */
    void generatePDFForSession(String sessionId);

    /**
     * Get all the groups a user is in.
     * @param userId the user we want the groups of
     * @return the group for our user
     * @throws VradiException
     */
    public List<Group> getGroupsForUser(String userId) throws VradiException;

    /**
     * Export all the clients, users and groups in database
     *
     * @return the csv export as a String
     * @throws VradiException if any error happen during export
     */
    public String exportClientDB() throws VradiException;

    /**
     * Get the status passed in parameter. Init status if it have not been done
     * already
     *
     * @param formStatus the status from enum
     * @return the Status object
     * @throws VradiException if an error occur during statuses initialisation
     */
    public Status getStatus(VradiConstants.FormStatus formStatus) throws VradiException;

}
