/* *##%
 * Vradi :: Services
 * Copyright (C) 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services.managers;

import java.io.File;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FacetTopic;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyService;
import org.sharengo.wikitty.WikittyTransaction;
import org.sharengo.wikitty.WikittyUtil;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import au.com.bytecode.opencsv.CSVReader;
import au.com.bytecode.opencsv.CSVWriter;

import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.VradiStorageServiceImpl;

/**
 * Import/export manager.
 * 
 * (called by {@link VradiStorageServiceImpl}).
 * 
 * @author chatellier
 * @version $Revision: 876 $
 * 
 * Last update : $Date: 2010-05-07 18:08:39 +0200 (ven., 07 mai 2010) $
 * By : $Author: chatellier $
 */
public class ImportExportManager {

    /** log. */
    private static final Log log = LogFactory.getLog(ImportExportManager.class);

    /** Wikitty id header name. */
    protected final static String FIELD_WIKITTY_ID = "Wikitty.Id";

    protected final WikittyProxy proxy;

    /** Executor that do import export task */
    protected ExecutorService importExportExecutor = Executors.newFixedThreadPool(1);

    /** contains all import or export task, key is job id send to client */
    protected Map<String, Future<String>> importExportTask =
            new HashMap<String, Future<String>>();

    /** directory path where export asynchronous file are stored */
    protected String EXPORT_DIRECTORY = "/tmp/";

    /** url used by client to retrieve export file when job is ended */
    protected String EXPORT_URL = "file:///tmp/";

    public ImportExportManager() {
        this(ServiceFactory.getWikittyProxy());
    }

    public ImportExportManager(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    /**
     * Synchronous import of CSV string.
     * 
     * @param csv
     */
    public void syncImportFromCSV(String csv) {
        Reader reader = new StringReader(csv);
        ImportCSVTask task = new ImportCSVTask(proxy.getWikittyService(), reader);
        task.run();
    }

    /**
     * Synchronous import of CSV uri.
     * 
     * @param uri uri of file to read (valid url)
     */
    public void syncImportFromCSVUri(String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportCSVTask task = new ImportCSVTask(proxy.getWikittyService(), reader);
            task.run();
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    /**
     * 
     * @param uri
     * @return
     */
    public String asyncImportFromCSVUri(String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportCSVTask task = new ImportCSVTask(proxy.getWikittyService(), reader);
            FutureTask<String> future = new FutureTask<String>(task, null);
            importExportExecutor.submit(future);

            String jobId = UUID.randomUUID().toString();
            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    public String asyncExportCSVAllByCriteria(Criteria criteria) {
        try {
            String jobId = UUID.randomUUID().toString();

            File file = new File(EXPORT_DIRECTORY, jobId);
            String url = EXPORT_URL + jobId;
            Writer result = new FileWriter(file);
            ExportCSVTask task = new ExportCSVTask(proxy.getWikittyService(), criteria, result);
            FutureTask<String> future = new FutureTask<String>(task, url);
            importExportExecutor.submit(future);

            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    public String syncExportCSVAllByCriteria(Criteria criteria) {
        StringWriter result = new StringWriter();
        ExportCSVTask task = new ExportCSVTask(proxy.getWikittyService(), criteria, result);
        task.run();
        return result.toString();
    }

    static public class ImportCSVTask implements Runnable {
        protected WikittyService ws;
        protected WikittyTransaction transaction;
        protected Reader reader;

        protected Pattern queryPattern = Pattern.compile("^((\\w+)\\.(\\w+))=(\"(.+)\"|([^\"]+))$");

        public ImportCSVTask(WikittyService ws, Reader reader) {
            this.ws = ws;
            this.reader = reader;
            //this.transaction = new WikittyTransaction();
        }
        
        @Override
        public void run() {
            try {

                // get index of wikitty.id field
                int wikittyIdIndex = -1;

                // read header to get extension names
                // and build two array for ext and fieldName
                CSVReader csvReader = new CSVReader(reader);
                String[] header = csvReader.readNext();
                String[] ext = new String[header.length];
                String[] fieldsName = new String[header.length];
                for (int i = 0; i < header.length ; ++i) {
                    // wikitty id is technical (special management)
                    if (FIELD_WIKITTY_ID.equals(header[i])) {
                        wikittyIdIndex = i;
                    }
                    else {
                        ext[i] = header[i].substring(0, header[i].indexOf("."));
                        fieldsName[i] = header[i].substring(header[i].indexOf(".") + 1);
                    }
                }

                // create a wikitty for each next line
                String[] currentLine = null;
                while ((currentLine = csvReader.readNext()) != null) {

                    String wikittyId = currentLine[wikittyIdIndex];
                    Wikitty currentWikitty = null;
                    if (StringUtils.isNotEmpty(wikittyId)) {
                        currentWikitty = new Wikitty(wikittyId);
                    }
                    else {
                        currentWikitty = new Wikitty();
                    }

                    for (int i = 0; i < header.length ; ++i) {

                        // wikitty id column, already managed
                        if (i == wikittyIdIndex) {
                            continue;
                        }

                        String extName = ext[i];
                        String fieldName = fieldsName[i];
                        String value =  currentLine[i];

                        // case null or empty
                        if (StringUtils.isNotEmpty(value)) {

                            // extension must exists on wikitty to set a field value
                            addMissingExtension(ws, currentWikitty, extName);

                            // convert link values (if necessary)
                            value = convertLinkValues(value);

                            // add value to correct field
                            FieldType fieldType = currentWikitty.getFieldType(extName + "." + fieldName);
                            if(fieldType.isCollection()) {
                                String[] multiplesValues = StringUtil.split(value);
                                for (String multiplesValue : multiplesValues) {
                                    // begin and ends with () only if fields
                                    // has multiples values during import
                                    if (multiplesValue.startsWith("(") && multiplesValue.endsWith(")")) {
                                        multiplesValue = multiplesValue.substring(1, multiplesValue.length() - 1);
                                    }
                                    currentWikitty.addToField(extName, fieldName, multiplesValue);
                                }
                            } else {
                                currentWikitty.setField(extName, fieldName, value);
                            }
                        }
                    }

                    // add it into datas
                    // FIXME EC-20100414 use store methods with transaction
                    // TODO EC-20100414 use store (List) with 1000/1000
                    ws.store(currentWikitty);
                }
            } catch (Exception eee) {
                //transaction.rollback();
                throw new WikittyException(eee);
            }
        }

        /**
         * Recusively add missing extension of not exist and required extension too.
         */
        protected void addMissingExtension(WikittyService ws, Wikitty currentWikitty, String extName) {
            // extension must exists on wikitty to set a field value
            if (!currentWikitty.hasExtension(extName)) {
                WikittyExtension extension = ws.restoreExtensionLastVersion(extName);

                String requires = extension.getRequires();
                if (StringUtils.isNotEmpty(requires)) {
                    // add required extensions BEFORE current
                    for (String require : requires.split(",")) {
                        String localRequire = require.trim();
                        addMissingExtension(ws, currentWikitty, localRequire);
                    }
                }

                currentWikitty.addExtension(extension);
            }
        }

        /**
         * Convert internal queries to search for other real wikitty ids.
         * 
         * @param value
         * @return
         */
        protected String convertLinkValues(String value) {

            String originalValue = value;
            String resultValue = "";
            String separator = "";
            boolean correctQueries = true;

            // manage multiples query comma separated
            try {
                String[] queries = StringUtil.split(value, ",");
                
                for (String query : queries) {
                    Matcher m = queryPattern.matcher(query.trim());
                    if (m.find()) {
                        String fqField = m.group(1);
                        String fValue = m.group(5);
                        if (fValue == null) {
                            // quoted value
                            fValue = m.group(6);
                        }
                        Criteria criteria = Search.query().eq(fqField, fValue).criteria();
                        Wikitty wikitty = ws.findByCriteria(criteria);
                        if (wikitty == null) {
                            correctQueries = false;
                        }
                        else {
                            resultValue += separator + wikitty.getId();
                            separator = ",";
                        }
                    }
                    else {
                        // global parsing fail
                        // return original value
                        correctQueries = false;
                    }
                }
            }
            catch (StringIndexOutOfBoundsException eee) {
                if (log.isTraceEnabled()) {
                    log.trace("Can't split field on , skipping");
                }
            }
            
            // if convertion has not been done, return original value
            if (!correctQueries || StringUtils.isEmpty(resultValue)) {
                resultValue = originalValue;
            }
            return resultValue;
        }

    } // end ImportCSVTask

    static public class ExportCSVTask implements Runnable {
        protected WikittyService ws;
        protected WikittyTransaction transaction;
        protected Writer writer;
        protected Criteria criteria;

        public ExportCSVTask(WikittyService ws, Criteria criteria, Writer writer) {
            this.ws = ws;
            this.writer = writer;
            this.transaction = new WikittyTransaction();
            this.criteria = criteria;
        }
        
        @Override
        public void run() {
            try {

                //transaction.begin();
                CSVWriter csvWriter = new CSVWriter(writer);

                // write extensions header
                // use a facet to get only extension used in export
                criteria.addFacetField(Element.ELT_EXTENSION);
                
                // write all data into writer
                // Criteria criteria = Search.query().eq(Element.ELT_EXTENSION,
                //    AbstractTestConformance.EXTNAME).criteria();
                PagedResult<Wikitty> pageResult = ws.findAllByCriteria(criteria);
                List<String> extensionHeader = new LinkedList<String>();
                extensionHeader.add(FIELD_WIKITTY_ID);
                for (FacetTopic topic : pageResult.getTopic(Element.ELT_EXTENSION)) {
                    String extName = topic.getTopicName();
                    log.debug("Extention = " + extName);
                    
                    WikittyExtension extension = ws.restoreExtensionLastVersion(extName);
                    String ext = WikittyExtension.computeName(extName);
                    for (String fieldName : extension.getFieldNames()) {
                        extensionHeader.add(ext + "." + fieldName);
                    }
                }
                csvWriter.writeNext(extensionHeader.toArray(new String[extensionHeader.size()]));

                if (log.isDebugEnabled()) {
                    log.debug("Exporting wikitty : " + pageResult.getAll().size() + " results");
                }

                // Export wikitty data
                for (Wikitty w : pageResult.getAll()) {

                    String[] wikittyField = new String[extensionHeader.size()];

                    // first, add technical id
                    wikittyField[extensionHeader.indexOf(FIELD_WIKITTY_ID)] = w.getId();

                    // wikitty export must be composed of all field
                    // corresponding to header extensions names
                    for (String fieldName : w.fieldNames()) {
                        String currentField = "";

                        FieldType type = w.getFieldType(fieldName);
                        if (type.isCollection()) {
                            Object fqField = w.getFqField(fieldName);
                            if (fqField != null) {
                                String separator = "";
                                for (Object o : (Collection) fqField) {
                                    String fqFieldValue = WikittyUtil.toString(type, o);
                                    currentField += separator + "(" + fqFieldValue + ")";
                                    separator = ",";
                                }
                            }
                        } else {
                            String fqFieldValue = WikittyUtil.toString(type, w.getFqField(fieldName));
                            currentField = fqFieldValue;
                        }
                        
                        wikittyField[extensionHeader.indexOf(fieldName)] = currentField;
                    }
                    
                    csvWriter.writeNext(wikittyField);
                }
                
                csvWriter.close();
            } catch (Exception eee) {
                //transaction.rollback();
                throw new WikittyException(eee);
            }
        }
    } // end ExportCSVTask
}
