/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import static java.util.Collections.EMPTY_LIST;

import java.io.File;
import java.io.IOException;
import java.util.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.DateUtils;
import org.nuiton.util.FileUtil;
import org.sharengo.wikitty.BusinessEntityWikitty;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiConstants.FormStatus;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.QueryMakerImpl;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.SessionImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.Configuration;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.ServiceUtil;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.dto.DTOHelper;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiQueryBean;

/**
 * Class containing the methods to manage the forms :
 * - creation, update, deletion, retrieving
 * - binding forms with clients whose queries are returning the forms
 *
 * @author schorlet
 * @date 2010-01-27 11:03:15
 * @version $Revision: 898 $ $Date: 2010-05-10 22:41:47 +0200 (lun., 10 mai 2010) $
 */
public class FormManager {
    
    private static final Log log = LogFactory.getLog(FormManager.class);

    private final WikittyProxy proxy;
    private final ThesaurusManager thesaurusManager;
    private final SearchManager searchManager;

    public FormManager(WikittyProxy proxy,
            ThesaurusManager thesaurusManager, SearchManager searchManager) {
        this.proxy = proxy;
        this.thesaurusManager = thesaurusManager;
        this.searchManager = searchManager;
    }

    public FormManager(ThesaurusManager thesaurusManager,
            SearchManager searchManager) {
        proxy = ServiceFactory.getWikittyProxy();
        this.thesaurusManager = thesaurusManager;
        this.searchManager = searchManager;
    }

    public List<Form> getAllForms() {
        if (log.isDebugEnabled()) {
            log.debug("getAllForms()");
        }
        
        Search search = Search.query().eq(Element.ELT_EXTENSION, Form.EXT_FORM);
        Criteria criteria = search.criteria();
        
        PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
        List<Form> all = forms.getAll();
        
        return all;
    }
    
    public List<Form> getForms(List<String> formIds) {
        if (log.isDebugEnabled()) {
            log.debug("getForms(" + formIds + ")");
        }

        List<Form> forms = proxy.restore(Form.class, formIds);
        return forms;
    }
    
    public Form getForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("getForm(" + formId + ")");
        }
        
        Form form = proxy.restore(Form.class, formId);
        return form;
    }
    
    public void deleteForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("deleteForm(" + formId + ")");
        }
        
        proxy.delete(formId);
    }
    
    public Form updateForm(Form form) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateForm(form)");
        }
        
        List<Form> forms = Arrays.asList(form);
        List<Form> updateForms = updateForms(forms);
        
        Form updatedForm = updateForms.get(0);
        return updatedForm;
    }

    public List<Form> updateForms(List<Form> forms) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateForms(forms)");
        }

        if (forms == null || forms.isEmpty()) {
            return forms;
        }

        List<TreeNodeImpl> thesaurusList = new ArrayList<TreeNodeImpl>();
        for (Form form : forms) {
            if (log.isDebugEnabled()) {
                log.debug("updating form: " + form.getId());
            }

            if (form.getCreationDate() == null) {
                form.setCreationDate(new Date());

            } else {
                if(!form.getExtensionNames().contains(ModificationTag.EXT_MODIFICATION_TAG)) {
                    BusinessEntityWikitty businessEntityWikitty = (BusinessEntityWikitty)form;
                    Wikitty wikitty = businessEntityWikitty.getWikitty();
                    wikitty.addExtension(ModificationTag.EXTENSION_MODIFICATION_TAG);
                }
                form.setField(ModificationTag.EXT_MODIFICATION_TAG,
                        ModificationTag.FIELD_LAST_MODIFIED, new Date());
            }

            Set<String> thesaurus = form.getThesaurus();
            if (thesaurus != null) {
                for (String thesaurusId : thesaurus) {
                    TreeNodeImpl node = thesaurusManager.getThesaurus(thesaurusId);

                    if (node.getChildren() == null
                            || !node.getChildren().contains(form.getWikittyId())) {

                        node.addChildren(form.getWikittyId());

                        if (!thesaurusList.contains(node)) {
                            thesaurusList.add(node);
                        }
                    }
                }
            }
        }

        List<Form> result = proxy.store(forms);

        if (!thesaurusList.isEmpty()) {
            proxy.store(thesaurusList);
        }

        return result;
    }

    public Session getLastOpenSession(){
        
        Criteria criteria = Search.query().or()
                .eq(Session.FQ_FIELD_STATUS,
                String.valueOf(VradiConstants.SessionStatus.ACTIVE.getValue()))
                .eq(Session.FQ_FIELD_STATUS,
                String.valueOf(VradiConstants.SessionStatus.STOPPED.getValue()))
                .eq(Session.FQ_FIELD_STATUS,
                String.valueOf(VradiConstants.SessionStatus.ERROR.getValue()))
                .criteria();

        criteria.setEndIndex(0);
        criteria.addSortDescending(Session.FQ_FIELD_SESSIONDATE);

        PagedResult<Session> sessions = proxy.findAllByCriteria(Session.class, criteria);
        if (sessions.size() == 0){
            return null;
        }

        Session result = sessions.getFirst();

        if (log.isDebugEnabled()){
            log.debug("Last sessions found : " + result.getSessionDate() + " status : " + result.getStatus());
        }
        return result;
    }

    public List<Session> getSessions(Date sessionDate){
        if (log.isDebugEnabled()) {
            log.debug("getSessions for date " + sessionDate.toString());
        }
        Search search = Search.query();

        search.bw(Session.FQ_FIELD_SESSIONDATE,
                ServiceUtil.getBeginOfDaySolrDate(sessionDate),
                ServiceUtil.getEndOfDaySolrDate(sessionDate));

        // Get result
        Criteria criteria = search.criteria();
        List<Session> listSessions = new ArrayList<Session>(proxy.findAllByCriteria(Session.class, criteria).getAll());

        if (listSessions.isEmpty()) {
            Session lastOpenSession = getLastOpenSession();
            if (lastOpenSession != null){
                listSessions.add(lastOpenSession);
            }
        }

        if (log.isDebugEnabled()){
            log.debug(listSessions.size() + " sessions found");
        }

        return listSessions;
    }
    
    public void bindForms() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("bindForms");
        }

        try {
            // Searching current session
            Session currentSession = getLastOpenSession();

            // If all are close
            if (currentSession == null) {

                // Get all session for this date
                List<Session> sessionList = getSessions(new Date());

                int num = 0;
                if (sessionList != null && !sessionList.isEmpty()){

                    // Get last number
                    for (Session session : sessionList){
                        if (num < session.getNum()){
                           num = session.getNum();
                        }
                    }

                }
                // Increment
                num = num + 1;

                // create new session
                if (log.isDebugEnabled()){
                    log.debug("No session found, creating a new one");
                }
                currentSession = new SessionImpl();
                currentSession.setSessionDate(new Date());
                currentSession.setNum(num);
                currentSession.setStatus(VradiConstants.SessionStatus.ACTIVE.getValue());
            } else {

                // Mark all sending as deleted to not return them when search
                Set<String> sendingsIds = currentSession.getSending();

                if (log.isDebugEnabled()) {
                    log.debug("Mark to not search in " + (sendingsIds == null ? 0 : sendingsIds.size()) + " sendings");
                    if (sendingsIds != null) {
                        for (String sending : sendingsIds){
                            log.debug("Deleting sending : " + sending);
                        }
                    }
                }

                if (sendingsIds != null){
                    proxy.delete(sendingsIds);
                }
                currentSession.clearSending();
                currentSession = proxy.store(currentSession);
            }
            
            List<Sending> toSend = executeQueries(currentSession);

            // store the data
//            proxy.store(toSend);
//
//            for (Sending s : toSend){
//                currentSession.addSending(s.getWikittyId());
//            }
//            proxy.store(currentSession);

            // Store the session
            if (log.isDebugEnabled()) {
                
                int size = (currentSession.getSending() == null ||
                        currentSession.getSending().isEmpty()) ?
                        0 : currentSession.getSending().size();
                
                log.debug("Saving session " + currentSession.getSessionDate()
                        + " number " + currentSession.getNum()
                        + " with " + size
                        + " sending");
            }

        } catch (Exception e) {
            log.error("Cant bind form : ", e);
            throw new VradiException("Cant bind form : ", e);
        }
    }

    protected List<Sending> executeQueries(Session currentSession) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("executeQueries");
        }
        
        // List of the forms to bind ( = not yet bound)
        List<Sending> toSend = new ArrayList<Sending>();

        List<QueryMaker> queryMakers = new ArrayList<QueryMaker>(searchManager.findQueryMakersWithQueries());

        // iterate on the client and its users
        for (QueryMaker queryMaker : queryMakers) {

            // List of the forms to bind ( = not yet bound)
            List<Form> formsToBind = null;

            Set<String> queries = queryMaker.getQueries();
            if (queries == null) {
                continue;
            }

            for (String queryEntry : queries) {
                try {
                    VradiQueryBean queryBean = new VradiQueryBean(queryEntry);
                    String query = queryBean.getQuery();
                    
                    // create a new VradiFormPageDTO to find all the forms corresponding to the query
                    VradiFormPageDTO vradiFormPage = new VradiFormPageDTO();

                    // Only selected forms
                    Status status = getSelectionneStatus();

                    // Limited for older inscription date
                    Date olderInscriptionDate = getOlderInscriptionDate(queryMaker);
                    
                    // Execute request
                    vradiFormPage = searchManager.findForms(query, vradiFormPage, olderInscriptionDate, status.getWikittyId());

                    // Extract result
                    formsToBind = vradiFormPage.getFormsToShow();

                    if (log.isDebugEnabled()) {
                        log.debug("Query " + query + " return " + formsToBind.size() + " forms");
                    }
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            }

            // if some forms are to be bound, bind them
            if (formsToBind != null && !formsToBind.isEmpty()) {
                toSend.addAll(createAllSending(currentSession, queryMaker, formsToBind, true));
            }
        }
        toSend = addEmptyClientSending(currentSession, toSend);
        return toSend;
    }

    // Add empty client sending to display it in email ui
    protected List<Sending> addEmptyClientSending(Session currentSession, List<Sending> toSend) {
        log.info("addEmptyClientSending " + toSend.size());
        List<Sending> result = new ArrayList<Sending>(toSend);
        for (Sending s : toSend) {
            String userId = s.getUser();
            if (userId != null) {

                User user = proxy.restore(User.class, userId);
                log.info("Search sending for client of " + user.getName());

                Client client = proxy.restore(Client.class, user.getClient());
                Sending clientSending = findExistingOrCreateSending(currentSession, client);

                Set<String> forms = clientSending.getForm();
                log.info("Client sending " + client.getName() + " has " + (forms == null ? 0 : forms.size()) + " forms");

                clientSending.setClient(client.getWikittyId());

                currentSession.addSending(clientSending.getWikittyId());

                result.add(proxy.store(clientSending));
            }
        }
        currentSession = proxy.store(currentSession);

        return result;
    }

    protected Date getOlderInscriptionDate(QueryMaker queryMaker) {
        Date result = new Date();
        if (queryMaker instanceof User) {
            User user = (User)queryMaker;
            Date inscriptionDate = user.getInscriptionDate();
            if (inscriptionDate != null && result.before(inscriptionDate)){
                result = inscriptionDate;
            }
            if (log.isDebugEnabled()){
                log.debug("The older inscription date for user " + user.getName() + " is " + result) ;
            }
        }
        if (queryMaker instanceof Client) {
            Client client = (Client)queryMaker;
            //Set<String> userIds = client.getUser();
            //if (userIds != null){
            //    for (String userId : userIds){
            //        User user = proxy.restore(User.class, userId);
            List<User> users = DTOHelper.getClientUser(client.getWikittyId());
            for (User user : users) {
                    Date inscriptionDate = getOlderInscriptionDate(user);
                    if (inscriptionDate != null && result.before(inscriptionDate)){
                        result = inscriptionDate;
                    }
                //}
            }
            if (log.isDebugEnabled()){
                log.debug("The older inscription date for client " + client.getName() + " is " + result) ;
            }
        }
        if (queryMaker instanceof Group) {
            Group group = (Group)queryMaker;
            Set<String> userIds = group.getUser();
            if (userIds != null){
                for (String userId : userIds){
                    User user = proxy.restore(User.class, userId);
                    Date inscriptionDate = getOlderInscriptionDate(user);
                    if (inscriptionDate != null && result.before(inscriptionDate)){
                        result = inscriptionDate;
                    }
                }
            }
            Set<String> clientIds = group.getClient();
            if (clientIds != null){
                for (String clientId : clientIds){
                    Client client = proxy.restore(Client.class, clientId);
                    Date inscriptionDate = getOlderInscriptionDate(client);
                    if (inscriptionDate != null && result.before(inscriptionDate)){
                        result = inscriptionDate;
                    }
                }
            }
            if (log.isDebugEnabled()){
                log.debug("The older inscription date for group " + group.getName() + " is " + result) ;
            }
        }

        if (log.isDebugEnabled()){
            log.debug("The older inscription date " + result) ;
        }
        return result;
    }

    public List<Sending> createAllSending(Session currentSession, QueryMaker queryMaker, List<? extends Form> formsToBind, boolean removeAlreadyBind){
        List<Sending> result = new ArrayList<Sending>();

        // Get casted entity
        queryMaker = ClientManager.castAsRealQueryMaker((QueryMakerImpl)queryMaker);

        // Create or find current sending for queryMaker
        Sending sending = findExistingOrCreateSending(currentSession, queryMaker);

        if (queryMaker instanceof Client) {
            sending.setClient(queryMaker.getWikittyId());
            if (log.isDebugEnabled()) {
                log.debug("Add client " + queryMaker.getWikittyId());
            }

            // Create sending for user of clients
            //Set<String> ids = ((Client) queryMaker).getUser();
            //if (ids != null){
            //    for (String userId : ids){
            //        User user = proxy.restore(User.class, userId);
            List<User> users = DTOHelper.getClientUser(queryMaker.getWikittyId());
            for (User user : users) {
                    //if (user != null) {
                        result.addAll(createAllSending(currentSession, user, formsToBind, removeAlreadyBind));
                    //}
                //}
            }

        } else if (queryMaker instanceof User) {
            sending.setUser(queryMaker.getWikittyId());
            if (log.isDebugEnabled()) {
                log.debug("Add user " + queryMaker.getWikittyId());
            }

        } else if (queryMaker instanceof Group) {
            sending.setGroup(queryMaker.getWikittyId());
            if (log.isDebugEnabled()) {
                log.debug("Add group " + queryMaker.getWikittyId());
            }

            // Create sending for client of the group
            Set<String> clientIds = ((Group) queryMaker).getClient();
            if (clientIds != null){
                for (String clientId : clientIds){
                    Client client = proxy.restore(Client.class, clientId);
                    if (client != null) {
                        result.addAll(createAllSending(currentSession, client, formsToBind, removeAlreadyBind));
                    }
                }
            }

            // Create sending for user of the group
            Set<String> usersIds = ((Group) queryMaker).getUser();
            if (usersIds != null){
                for (String userId : usersIds){
                    User user = proxy.restore(User.class, userId);
                    if (user != null) {
                        result.addAll(createAllSending(currentSession, user, formsToBind, removeAlreadyBind));
                    }
                }
            }
        }

        // Attach forms
        if (formsToBind != null){

            if (removeAlreadyBind) {

                // Remove already bound forms with queryMaker
                formsToBind = removeAlreadyBound(formsToBind, queryMaker);

                if (log.isDebugEnabled()) {
                    log.debug("Adding " + formsToBind.size() + " forms to sending");
                }

                if (queryMaker instanceof User){

                    User user = (User) queryMaker;

                    // Check date
                    for (Form f : new ArrayList<Form>(formsToBind)) {

                        // Using last modification date
                        Date formCreationDate = (Date)f.getField(ModificationTag.EXT_MODIFICATION_TAG,
                                ModificationTag.FIELD_LAST_MODIFIED);

                        // Get user inscritpion date
                        Date userInscriptionDate = user.getInscriptionDate();

                        // This test never append, new date is set by default
                        if (userInscriptionDate != null) {
                            if (formCreationDate.before(userInscriptionDate)) {
                                formsToBind.remove(f);
                            }
                        }
                    }
                }

            }
            // Attach forms to sending
            for (Form f : formsToBind){
                String fid = f.getWikittyId();
                sending.removeForm(fid);
                sending.addForm(fid);
            }
        }

        Set<String> formsAttached = sending.getForm();
        // If not empty
        if (formsAttached != null && !formsAttached.isEmpty()){
            result.add(sending);
        }

        // If not empty
        if (!result.isEmpty()){

            // Attach sending to session
            for (Sending s : result){
                currentSession.addSending(s.getWikittyId());
            }

            // store datas
            result = proxy.store(result);
            currentSession = proxy.store(currentSession);
        }

        return result;
    }

    public List<Sending> removeAllSending(Session session, Form form, QueryMaker queryMaker) {

        Set<String> sendingList = session.getSending();
        List<String> sendingArray = EMPTY_LIST;
        if (sendingList != null){
            sendingArray = new ArrayList<String>(sendingList);
        }

        List<Sending> sendings = proxy.restore(Sending.class, sendingArray);

        // To have the real instance
        queryMaker = ClientManager.castAsRealQueryMaker((QueryMakerImpl)queryMaker);

        // List of impacted queryMakers
        List<String> impacted = new ArrayList<String>();
        impacted.add(queryMaker.getWikittyId());

        if (queryMaker instanceof Client) {
            // Create sending for user of clients
            //Set<String> ids = ((Client) queryMaker).getUser();
            //if (ids != null){
                //for (String userId : ids){
            List<User> users = DTOHelper.getClientUser(queryMaker.getWikittyId());
            for (User user : users) {
                    impacted.add(user.getWikittyId());
                //}
            }

        } else if (queryMaker instanceof Group) {
            // Create sending for client of the group
            Set<String> ids = ((Group) queryMaker).getClient();
            if (ids != null){
                for (String clientId : ids){
                    Client client = proxy.restore(Client.class, clientId);
                    impacted.add(clientId);

                    // Create sending for user of clients
                    //Set<String> userIds = client.getUser();
                    //if (userIds != null){
                        //for (String userId : userIds){
                    List<User> users = DTOHelper.getClientUser(queryMaker.getWikittyId());
                    for (User user : users) {
                            impacted.add(user.getWikittyId());
                        //}
                    }
                }
            }

            // Find sending for user of the group
            Set<String> userIds = ((Group) queryMaker).getUser();
            if (userIds != null){
                for (String userId : userIds){
                    impacted.add(userId);
                }
            }
        }

        // Remove
        List<Sending> sendingImpacted = new ArrayList<Sending>();
        for (Sending sending : sendings){
            if (sending.getForm().contains(form.getWikittyId())){
                if (impacted.contains(sending.getClient()) ||
                    impacted.contains(sending.getUser()) ||
                    impacted.contains(sending.getGroup())) {

                    // Remove
                    sending.removeForm(form.getWikittyId());

                    // Add deleted
                    sending.addDeletedForms(form.getWikittyId());
                    sendingImpacted.add(sending);
                }
            }

            // If is empty, set deleted
            if (sending.getForm().isEmpty()) {
                sending.setStatus(VradiConstants.SendingStatus.DELETED.getValue());
            }
        }
        proxy.store(sendingImpacted);
        
        return proxy.restore(Sending.class, sendingArray);
    }

    public Sending findExistingOrCreateSending(Session currentSession, QueryMaker queryMaker){

        // Restore existing sending in session
        Set<String> sendingsId = currentSession.getSending();
        List<Sending> sendings = new ArrayList<Sending>();
        if (sendingsId != null) {
            sendings = proxy.restore(Sending.class, new ArrayList<String>(sendingsId));
        }

        // chargement, si il existe, du sending concernant
        // ce client / user / group
        for (Sending sending : sendings) {
            String qmId = queryMaker.getWikittyId();
            if (qmId.equals(sending.getUser()) ||
                    qmId.equals(sending.getClient()) ||
                    qmId.equals(sending.getGroup()) &&
                    !(sending.getStatus() == VradiConstants.SendingStatus.DELETED.getValue())) {

                return sending;
            }
        }

        // Set properties in default
        SendingImpl newSending = new SendingImpl();
        newSending.setSentDate(null);
        newSending.setReceptionDate(null);
        newSending.setParagraph(VradiConstants.DEFAULT_SENDING_PARAGRAPH);
        newSending.setReceptionProof(false);
        newSending.setStatus(VradiConstants.SendingStatus.TO_SEND.getValue());
        return newSending;
    }

    protected List<? extends Form> removeAlreadyBound(List<? extends Form> forms, QueryMaker queryMaker) {
        List<String> formIds = new ArrayList<String>();

        // Extract formIds
        for (Form form : forms){
            formIds.add(form.getWikittyId());
        }

        // Recherche d'un sending existant
        String qmId = queryMaker.getWikittyId();

        Search andSearch = Search.query().and();

        // If its attached or deleted
        andSearch.or().contains(Sending.FQ_FIELD_FORM, formIds)
                      .contains(Sending.FQ_FIELD_DELETEDFORMS, formIds);

        // And attached to client / user / group
        andSearch.or().eq(Sending.FQ_FIELD_CLIENT, qmId)
                      .eq(Sending.FQ_FIELD_USER, qmId)
                      .eq(Sending.FQ_FIELD_GROUP, qmId);

        // And not selected
        andSearch.not()
                 .eq(Sending.FQ_FIELD_STATUS, String.valueOf(VradiConstants.SendingStatus.DELETED.getValue()));

        // Find
        PagedResult<Sending> pagedResult = proxy.findAllByCriteria(Sending.class, andSearch.criteria());

        if (log.isDebugEnabled()){
            List<Sending> sendings = pagedResult.getAll();
            log.debug("Remove " + sendings.size() + " already bounds forms");
        }
        for (Sending send : pagedResult.getAll()){
            formIds.removeAll(send.getForm());
        }
        if (log.isDebugEnabled()){
            log.debug("Rest " + formIds.size() + " forms");
        }

        List<Form> result = proxy.restore(Form.class, formIds);
        return result;
    }

    public Status getStatus(String statusId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatus(" + statusId + ")");
        }
        Status status = proxy.restore(Status.class, statusId);
        return status;
    }

    public List<Status> getStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatuses(statusIds)");
        }
        List<Status> statuses = proxy.restore(Status.class, statusIds);
        return new ArrayList<Status>(statuses);
    }

    public List<Status> getAllStatuses() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getAllStatus()");
        }

        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        Criteria criteria = search.criteria();

        PagedResult<Status> statuses = proxy.findAllByCriteria(Status.class,
                criteria);
        List<Status> all = statuses.getAll();

        if(all == null || all.isEmpty()) {
            log.warn("no status found, creating ...");
            all = new ArrayList<Status>();
            all.add(getNonTraiteStatus());
            all.add(getNonSelectionneStatus());
            all.add(getPreselectionneStatus());
            all.add(getSelectionneStatus());
        }
        return new ArrayList<Status>(all);
    }

    public Status getNonTraiteStatus() {
        FormStatus nonTraite = VradiConstants.FormStatus.NON_TRAITE;
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        search.eq(Status.FQ_FIELD_NAME, nonTraite.getName());
        Criteria criteria = search.criteria();

        Status toTreatStatus = proxy.findByCriteria(Status.class,
                criteria);
        if(toTreatStatus == null) {
            toTreatStatus = new StatusImpl();
            toTreatStatus.setName(nonTraite.getName());
            toTreatStatus.setValue(nonTraite.getValue());
            toTreatStatus.setDescription(nonTraite.getDescription());
            toTreatStatus.setModifiable(false);

            toTreatStatus = proxy.store(toTreatStatus);
        }
        return toTreatStatus;
    }

    public Status getNonSelectionneStatus() {
        VradiConstants.FormStatus nonSelectionne = VradiConstants.FormStatus.NON_SELECTIONNE;
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        search.eq(Status.FQ_FIELD_NAME, nonSelectionne.getName());
        Criteria criteria = search.criteria();

        Status toTreatStatus = proxy.findByCriteria(Status.class,
                criteria);
        if(toTreatStatus == null) {
            toTreatStatus = new StatusImpl();
            toTreatStatus.setName(nonSelectionne.getName());
            toTreatStatus.setValue(nonSelectionne.getValue());
            toTreatStatus.setDescription(nonSelectionne.getDescription());
            toTreatStatus.setModifiable(false);

            toTreatStatus = proxy.store(toTreatStatus);
        }
        return toTreatStatus;
    }

    public Status getPreselectionneStatus() {
        VradiConstants.FormStatus preselectionne = FormStatus.PRESELECTIONNE;
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        search.eq(Status.FQ_FIELD_NAME, preselectionne.getName());
        Criteria criteria = search.criteria();

        Status validatedStatus = proxy.findByCriteria(Status.class,
                criteria);
        if(validatedStatus == null) {
            validatedStatus = new StatusImpl();
            validatedStatus.setName(preselectionne.getName());
            validatedStatus.setValue(preselectionne.getValue());
            validatedStatus.setDescription(preselectionne.getDescription());
            validatedStatus.setModifiable(false);

            validatedStatus = proxy.store(validatedStatus);
        }
        return validatedStatus;
    }

    public Status getSelectionneStatus() {
        FormStatus selectionne = VradiConstants.FormStatus.SELECTIONNE;
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        search.eq(Status.FQ_FIELD_NAME, selectionne.getName());
        Criteria criteria = search.criteria();

        Status validatedStatus = proxy.findByCriteria(Status.class,
                criteria);
        if(validatedStatus == null) {
            validatedStatus = new StatusImpl();
            validatedStatus.setName(selectionne.getName());
            validatedStatus.setValue(selectionne.getValue());
            validatedStatus.setDescription(selectionne.getDescription());
            validatedStatus.setModifiable(false);

            validatedStatus = proxy.store(validatedStatus);
        }
        return validatedStatus;
    }

    public List<Status> updateStatuses(List<Status> statuses)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatuses(statuses)");
        }
        
        try {
            List<Status> result = proxy.store(statuses);
            return result;
            
        } catch (Exception e) {
            log.error("Cant update statuses : ", e);
            throw new VradiException("Cant update statuses : ", e);
        }
    }

    public Status updateStatus(Status status) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatus(status)");
        }

        try {
            status = proxy.store(status);
            return status;
            
        } catch (Exception e) {
            log.error("Cant update status : ", e);
            throw new VradiException("Cant update status : ", e);
        }
    }

    public void deleteStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("deleteStatus(" + statusIds + ")");
        }

        try {
            proxy.delete(statusIds);
            
        } catch (Exception e) {
            log.error("Cant delete status : ", e);
            throw new VradiException("Cant delete status : ", e);
        }
    }

    /**
     * 
     * @param form
     * @param files
     * @throws IOException
     * 
     * @deprecated can't use non serializable parameters with hessian
     */
    @Deprecated
    public void uploadFiles(Form form, List<File> files) throws IOException {
        File embededFilesDir = Configuration.getInstance().getEmbededFilesDir();
        File copyDir = new File(embededFilesDir, form.getWikittyId());
        for(File file : files) {
            File copyFile = new File(copyDir, file.getName());
            FileUtil.copy(file, copyFile);
        }
    }

    /**
     * 
     * @param form
     * @param attachments
     * @throws IOException
     * 
     * @deprecated can't use non serializable parameters with hessian
     */
    @Deprecated
    public void uploadAttachments(Form form, List<File> attachments) throws IOException {
        File attachmentsDir = Configuration.getInstance().getAttachmentsDir();
        File copyDir = new File(attachmentsDir, form.getWikittyId());
        for(File attachment : attachments) {
            File copyFile = new File(copyDir, attachment.getName());
            FileUtil.copy(attachment, copyFile);
        }
    }
}
