/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.util.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FacetTopic;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;

/**
 * Class containing the methods to manage the thesaurus :
 * - node creation, update, retrieving, deletion
 * - get child node number
 * - get children
 *
 * @author schorlet
 * @date 2010-01-22 20:18:29
 * @version $Revision: 498 $ $Date: 2010-02-09 12:10:31 +0100 (mar., 09 févr. 2010) $
 */
public class ThesaurusManager {
    private static final Log log = LogFactory.getLog(ThesaurusManager.class);
    
    public static final String ROOT_THESAURUS_NAME = "Thesaurus";
    private final WikittyProxy proxy;
    
    public ThesaurusManager(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    public ThesaurusManager() {
        proxy = ServiceHelper.getWikittyProxy();
    }
    
    public TreeNode getRootThesaurus() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getRootThesaurus()");
        }
        
        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, TreeNode.EXT_TREENODE)
                .eq(TreeNode.FQ_FIELD_NAME, ROOT_THESAURUS_NAME).criteria();
        
        TreeNode rootThesaurus = proxy.findByCriteria(TreeNode.class, criteria);
        
        if (rootThesaurus == null) {
            log.warn("root thesaurus found, creating ...");
            
            rootThesaurus = new TreeNodeImpl();
            rootThesaurus.setName(ROOT_THESAURUS_NAME);
            
//            String incrementMajorRevision = WikittyUtil.incrementMajorRevision(
//                    rootThesaurus.getWikittyVersion());
//            rootThesaurus.setWikittyVersion(incrementMajorRevision);
            
            rootThesaurus = proxy.store(rootThesaurus);
        }
        
        log.info("root thesaurus: " + rootThesaurus);
        return rootThesaurus;
    }

    
    public List<TreeNode> getAllThesaurus() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllThesaurus()");
        }

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION,
                TreeNode.EXT_TREENODE).criteria();

        List<TreeNode> nodes = proxy.findAllByCriteria(
                TreeNode.class, criteria).getAll();
        log.info(String.format("found %s thesaurus entries", nodes.size()));
        
        if (log.isDebugEnabled()) {
            for (TreeNode node : nodes) {
                log.debug(String.format("thesaurus name:%s, children:%s",
                        node.getName(), node.getChildren()));
            }
        }
        
        return nodes;
    }

    public TreeNode getThesaurus(String thesaurusId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusId + ")");
        }
        TreeNode node = proxy.restore(TreeNode.class, thesaurusId);
        
        return node;
    }

    public List<TreeNode> getThesaurus(List<String> thesaurusIds) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusIds + ")");
        }
        List<TreeNode> nodes = proxy.restore(TreeNode.class, thesaurusIds);

        return nodes;
    }

    public List<TreeNode> getChildrenThesaurus(String thesaurusId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getChildrenThesaurus(" + thesaurusId + ")");
        }

        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, TreeNode.EXT_TREENODE);
        query.eq(TreeNode.FQ_FIELD_PARENT, thesaurusId);
        
        Criteria criteria = query.criteria();
        PagedResult<TreeNode> nodes = proxy.findAllByCriteria(TreeNode.class, criteria);
        List<TreeNode> all = nodes.getAll();
        
        return all;
    }

    public Map<TreeNode, Integer> getNbFormsByThesaurus()
            throws TechnicalException {
        Map<TreeNode, Integer> result = new HashMap<TreeNode, Integer>();

        if (log.isDebugEnabled()) {
            log.debug("getNbFormsByThesaurus()");
        }

        List<TreeNode> allThesaurus = getAllThesaurus();
        for (TreeNode node : allThesaurus) {
            Map.Entry<TreeNode, Integer> entry = proxy.restoreNode(
                    TreeNode.class, node.getWikittyId());
            
            if (entry != null) {
                result.put(node, entry.getValue());
                
            } else {
                result.put(node, 0);
            }
        }
        
        return result;
    }

    public int getNbFormsForThesaurus(String thesaurusId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getNbFormsForThesaurus(" + thesaurusId + ")");
        }

        Map.Entry<TreeNode, Integer> entry = proxy.restoreNode(TreeNode.class, thesaurusId);
        
        if (entry == null) {
            return 0;
        }
        
        return entry.getValue();
    }

    /**
     * Propose thesaurus nodes that might be in relation with a specified form
     *
     * @param form the <code>Form</code> containing the information needed
     * to search the thesaurus nodes
     * @param thesaurus the list of all the tree nodes of the thesaurus
     * @return a list of <code>TreeNode</code>
     * @throws TechnicalException
     */
    public List<TreeNode> proposeThesaurus(Form form, List<TreeNode> thesaurus)
            throws TechnicalException {
        if(log.isDebugEnabled()) {
            log.debug("proposeThesaurus(form)");
        }
        if(form == null) {
            return null;
        }
        List<TreeNode> result = new ArrayList<TreeNode>();

        //search the form
        Search search = Search.query();
        search.eq(Element.ELT_EXTENSION, Form.EXT_FORM);
        search.eq(Form.FQ_FIELD_ID, form.getId());
        Criteria criteria = search.criteria();

        //for each thesaurus node, add a facet criteria
        // with the search of the forms containing the tags of the node
        if(thesaurus == null) {
            thesaurus = getAllThesaurus();
        }
        if(log.isDebugEnabled()) {
            log.debug(form.getThesaurus());
        }
        for (TreeNode treeNode : thesaurus) {
            if((form.getThesaurus() == null ||
                    !form.getThesaurus().contains(treeNode.getWikittyId())) &&
                    treeNode.getExtensionNames().contains(
                            VradiThesaurusDTO.EXT_THESAURUS)) {
                List<String> tags = (List<String>) treeNode.getField(
                        VradiThesaurusDTO.EXT_THESAURUS,
                        VradiThesaurusDTO.TAGS_FIELD);

                Search tagSearch = Search.query(Search.KIND.OR);
                tagSearch.keyword(treeNode.getName());
                if(tags != null) {
                    for (String tag : tags) {
                        tagSearch.keyword(tag);
                    }
                    Criteria facetCriteria = tagSearch.criteria(treeNode.getWikittyId());
                    criteria.addFacetCriteria(facetCriteria);
                }
            }
        }
        PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
        Collection<String> facets = forms.getFacetNames();
        List<String> thIds = new ArrayList<String>();
        for (String facet : facets) {
            for (FacetTopic ft : forms.getTopic(facet)) {
                if(ft.getCount() > 0) {
                    thIds.add(ft.getTopicName());
                    break;
                }
            }
        }
        result = getThesaurus(thIds);
        return result;
    }
}
