/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.input.SAXBuilder;
import org.nuiton.util.MD5;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyUtil;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.InfogeneImpl;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.ServiceHelper;

/**
 * Class containing the methods to manage the binding of the xml streams fields
 * with the form fields :
 * - xml field bindings creation, update and retrieving
 * - xml streams retrieving
 * - form creation with the data from an xml stream
 *
 * @author schorlet
 * @date 2010-01-22 20:18:29
 * @version $Revision: 485 $ $Date: 2010-02-08 17:04:52 +0100 (lun., 08 févr. 2010) $
 */
public class BindingManager {
    private static final Log log = LogFactory.getLog(BindingManager.class);

    private final WikittyProxy proxy;
    private final FormTypeManager formTypeManager;
    private final FormManager formManager;
    
    //names of the rss tags
    public static final String CHANNEL = "channel";
    public static final String ENTRY = "entry";
    public static final String FEED = "feed";
    public static final String ITEM = "item";
    public static final String RDF = "rdf";
        
    /**
     * RSS date format
     */
    public static final SimpleDateFormat RSS_DATE_FORMAT =
            new SimpleDateFormat("E, d MMM yyyy HH:mm:ss Z",
                    java.util.Locale.US);

    public BindingManager(WikittyProxy proxy, FormTypeManager formTypeManager,
            FormManager formManager) {
        this.proxy = proxy;
        this.formTypeManager = formTypeManager;
        this.formManager = formManager;
    }

    public BindingManager(FormTypeManager formTypeManager, FormManager formManager) {
        this.proxy = ServiceHelper.getWikittyProxy();
        this.formTypeManager = formTypeManager;
        this.formManager = formManager;
    }

    /**
     * Retrieves the xml field bindings whose xml stream is xmlStream
     *
     * @param xmlStream the xml stream associated with the xml field bindings
     * we want to retrieve
     * @return a list containing the xml field bindings associated with
     * the xml stream xmlStream
     */
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();

        if (xmlStream != null && xmlStream.getXmlFieldBinding() != null) {
            if (log.isDebugEnabled()) {
                log.debug("getXmlFieldBindings(" + xmlStream.getName() + ")");
                log.debug(xmlStream.getXmlFieldBinding());
            }

            List<String> bindings = new ArrayList<String>();
            bindings.addAll(xmlStream.getXmlFieldBinding());
            
            List<XmlFieldBinding> restore = proxy.restore(XmlFieldBinding.class, bindings);
            list.addAll(restore);
        }

        return list;
    }

    /**
     * Retrieves the xml field binding whose id is xmlFieldBindingId
     *
     * @param xmlFieldBindingId the id of the xml field binding we want to retrieve
     * @return the xml field binding whose id is xmlFieldBindingId
     */
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBinding(" + xmlFieldBindingId + ")");
        }
        
        XmlFieldBinding xmlFieldBinding = proxy.restore(XmlFieldBinding.class, xmlFieldBindingId);
        return xmlFieldBinding;
    }

    /**
     * Updates the xml field bindings given in parameters
     *
     * @param bindings the list of the xml field bindings to update
     * @return the list of the xml field bindings up to date
     */
    public List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings) {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlFieldBindings(bindings)");
        }
        
        List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();
        
        if (bindings != null) {
            bindings = proxy.store(bindings);
            list.addAll(bindings);
        }
        
        return list;
    }
    
    public List<XmlStream> getAllXmlStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllXmlStreams()");
        }
        
        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, XmlStream.EXT_XMLSTREAM)
                .criteria();
        
        PagedResult<XmlStream> xmlStreams = proxy.findAllByCriteria(XmlStream.class, criteria);
        List<XmlStream> all = xmlStreams.getAll();
        
        List<XmlStream> list = new ArrayList<XmlStream>();
        list.addAll(all);
        
        return list;
    }
    
    public XmlStream getXmlStream(String xmlStreamId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getXmlStream(" + xmlStreamId + ")");
        }

        XmlStream xmlStream = proxy.restore(XmlStream.class, xmlStreamId);
        return xmlStream;
    }

    /**
     * Creates forms with the data of an xml stream
     *
     * @param xmlStream
     * @param lastItemRecorded
     * @param vradiUser
     * @return An array containing :
     * - the encrypted value of the last read item of the xml stream
     * - the number of created forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     */
    public Object[] getFormsFromXmlStream(XmlStream xmlStream,
            String lastItemRecorded, VradiUser vradiUser) throws TechnicalException {
        Object[] results = new Object[4];
        if (log.isDebugEnabled()) {
            log.debug("getFormsFromXmlStream(" + xmlStream.getName() + ", "
                    + lastItemRecorded + ", " + vradiUser + ")");
        }

        // map linking the form fields with the xmlstream fields
        Map<String, Set<String>> streamBinding = new HashMap<String, Set<String>>();
        List<Form> forms = new ArrayList<Form>();

        WikittyExtension formType = formTypeManager.getFormType(
                xmlStream.getFormTypeName());

        List<XmlFieldBinding> bindings = proxy.restore(XmlFieldBinding.class,
                new ArrayList<String>(xmlStream.getXmlFieldBinding()));

        //look for the WikittyExtension associated with the xmlstream
        for (XmlFieldBinding binding : bindings) {
            //get the first part of the fq field
            String formField = binding.getFormField();
            Set<String> xmlField = binding.getXmlField();
            streamBinding.put(formField, xmlField);

            if (formType == null) {
                String formTypeName = formField.substring(0, formField
                        .indexOf('.'));
                if (!Form.EXT_FORM.equals(formTypeName) &&
                        !Infogene.EXT_INFOGENE.equals(formTypeName)) {
                    formType = formTypeManager.getFormType(formTypeName);
                }
            }
        }
        
        if (formType == null) {
            return null;
        }

        String result = null;
        Document document = null;
        try {
            SAXBuilder sxb = new SAXBuilder();
            URL rssUrl = new URL(xmlStream.getUrl());
            document = sxb.build(rssUrl);
        } catch (Exception e) {
            throw new TechnicalException(e);
        }
        
        org.jdom.Element racine = document.getRootElement();
        List<org.jdom.Element> itemElt = null;
        
        if (racine.getChild(CHANNEL) != null) {
            itemElt = racine.getChild(CHANNEL).getChildren(ITEM);
        } else if (racine.getChild(ITEM) != null) {
            itemElt = racine.getChildren(ITEM);
        } else if (racine.getChild(ENTRY) != null) {
            itemElt = racine.getChildren(ENTRY);
        }

        if (itemElt == null) {
            return null;
        }

        int nbCreated = 0;
        int dateParsingError = 0;
        int numberParsingError = 0;

        // for each element of the xml stream
        for (org.jdom.Element element : itemElt) {
            StringBuffer sb = new StringBuffer();

            List<org.jdom.Element> fields = element.getChildren();
            for (org.jdom.Element field : fields) {
                sb.append(field.getText());
            }
            MD5 lastItem = new MD5(sb.toString());

            //get the encrypted value of the newest element
            if (result == null) {
                result = MD5.asHex(lastItem.Final());
            }

            //check if the element has not yet been read
            if (lastItemRecorded != null
                    && lastItemRecorded.equals(MD5.asHex(lastItem.Final()))) {
                break;
            }

            //create the form with the info from the xml stream
            Form form = new FormImpl();
            form.addExtension(formType);

            for (String field : streamBinding.keySet()) {
                int dot = field.indexOf('.');
                String extName = field.substring(0, dot);
                String fieldName = field.substring(dot + 1);

                FieldType fieldType = null;
                
                //get the field type
                if (extName.equals(Form.EXT_FORM)) {
                    fieldType = FormImpl.extensionForm.getFieldType(fieldName);

                } else if (extName.equals(Infogene.EXT_INFOGENE)) {
                    fieldType = InfogeneImpl.extensionInfogene.getFieldType(fieldName);
                    
                } else if (formType != null) {
                    fieldType = formType.getFieldType(fieldName);
                }

                // format the data
                if (fieldType != null && streamBinding.get(field) != null) {
                    
                    switch (fieldType.getType()) {
                    case DATE:
                        for (String xmlField : streamBinding.get(field)) {
                            org.jdom.Element child = element.getChild(xmlField);
                            
                            if (child != null) {
                                try {
                                    Date value = RSS_DATE_FORMAT.parse(child.getText());
                                    form.setField(extName, fieldName,
                                            WikittyUtil.solrDateFormat.format(value));
                                } catch(ParseException eee) {
                                    if(log.isDebugEnabled()) {
                                        log.debug("Date parsing error : "
                                            + child.getText());
                                    }
                                    dateParsingError++;
                                }
                            }
                        }
                        break;

                    case NUMERIC:
                        for (String xmlField : streamBinding.get(field)) {
                            if (element.getChild(xmlField) != null) {
                                org.jdom.Element child = element.getChild(xmlField);
                                
                                if (child != null) {
                                    try {
                                        Double value = Double.valueOf(child.getText());
                                        form.setField(extName, fieldName, value);
                                    } catch(NumberFormatException eee) {
                                        if(log.isDebugEnabled()) {
                                            log.debug("Number parsing error : "
                                                + child.getText());
                                        }
                                        numberParsingError++;
                                    }
                                }
                            }
                        }
                        break;

                    default:
                        for (String xmlField : streamBinding.get(field)) {
                            if (element.getChild(xmlField) != null) {
                                org.jdom.Element child = element.getChild(xmlField);
                                
                                if (child != null) {
                                    String value = child.getText();
                                    Object fieldValue = form.getField(extName, fieldName);
                                    String newValue = null;

                                    // TODO: check if field is a collection
                                    if (fieldValue != null) {
                                        newValue = fieldValue + "\n" + value;
                                    } else {
                                        newValue = value;
                                    }

                                    form.setField(extName, fieldName, newValue);
                                }
                            }
                        }
                        break;
                    }
                }
            }

//            if (form.getId() == null) {
                form.setId(FormManager.FORM_ID_DATE_FORMAT.format(new Date())
                        + form.getWikittyId());
//            }

            form.setStatus(formManager.getToTreatStatus().getWikittyId());
            forms.add(form);
            nbCreated++;
        }

        formManager.updateForms(forms);
        results[0] = result;
        results[1] = nbCreated;
        results[2] = dateParsingError;
        results[3] = numberParsingError;
        return results;

    }

}
