/*
 * #%L
 * JAXX :: Maven plugin
 * 
 * $Id: AbstractJaxxMojo.java 2118 2010-10-26 17:44:57Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2.5/maven-jaxx-plugin/src/main/java/org/nuiton/jaxx/plugin/AbstractJaxxMojo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jaxx.plugin;

import org.apache.maven.project.MavenProject;
import org.nuiton.io.MirroredFileUpdater;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.plugin.PluginWithEncoding;
import org.nuiton.util.FileUtil;

import java.io.File;

/**
 * Abstract Jaxx Mojo.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public abstract class AbstractJaxxMojo extends AbstractPlugin implements PluginWithEncoding {

    public abstract File getTargetDirectory();

    public abstract void setTargetDirectory(File targetDirectory);

    /**
     * Dépendance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * Encoding pour la generation des fichiers.
     * <p/>
     * <b>Note:</b> If nothing is filled here, we will use the system
     * property {@code file.encoding}.
     *
     * @parameter expression="${jaxx.encoding}" default-value="${project.build.sourceEncoding}"
     * @since 2.0.0
     */
    private String encoding;

    /**
     * verbose flag
     *
     * @parameter expression="${jaxx.verbose}" default-value="false"
     * @since 1.3
     */
    private boolean verbose;

    /**
     * to force generation of java source for any jaxx files with no timestamp checking.
     * <p/>
     * By default, never force generation.
     *
     * @parameter expression="${jaxx.force}" default-value="false"
     */
    private boolean force;

    /**
     * Flag to activate help id detection while parsing jaxx files.
     * <p/>
     * By default, not active.
     *
     * @parameter expression="${jaxx.generateHelp}" default-value="false"
     * @since 1.3
     */
    protected boolean generateHelp;

    @Override
    protected boolean checkPackaging() {
        // not accept pom modules
        return hasClassPath();
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public String getEncoding() {
        return encoding;
    }

    @Override
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    public boolean isForce() {
        return force;
    }

    public void setForce(boolean force) {
        this.force = force;
    }

    class GetLastModifiedFileAction implements FileUtil.FileAction {

        protected File lastFile;

        public GetLastModifiedFileAction(File lastFile) {
            this.lastFile = lastFile;
        }

        @Override
        public boolean doAction(File f) {

            if (f.lastModified() > lastFile.lastModified()) {
                lastFile = f;
            }
            return true;
        }

        public File getLastFile() {
            return lastFile;
        }
    }

    protected Long getLastModified(File dir) {
        if (!dir.exists()) {
            return null;
        }
        GetLastModifiedFileAction fileAction =
                new GetLastModifiedFileAction(dir);
        FileUtil.walkAfter(dir, fileAction);
        return fileAction.getLastFile().lastModified();
    }

    /**
     * To test if a jaxx source (or his css ) file is newser than his generated
     * java source file.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.0.2
     */
    public static class JaxxFileUpdater extends MirroredFileUpdater {

        protected JaxxFileUpdater(File sourceDirectory, File destinationDirectory) {
            super(".jaxx|*.css", ".java", sourceDirectory, destinationDirectory);
        }

        @Override
        public File getMirrorFile(File f) {
            String file = f.getAbsolutePath().substring(prefixSourceDirecotory);
            String extension = FileUtil.extension(f);
            String mirrorRelativePath = file.substring(0, file.length() - extension.length()) + "java";
            return new File(destinationDirectory + File.separator + mirrorRelativePath);
        }
    }
}
