/**
 * ##% Copyright (C) 2008 Code Lutin, Tony Chemit
 * This program is free software; you
 * can redistribute it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details. You
 * should have received a copy of the GNU General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 59 Temple Place
 * - Suite 330, Boston, MA 02111-1307, USA. 
 * ##%
 */
package org.nuiton.jaxx.util.config;

import org.nuiton.jaxx.util.DialogUIHandler;
import org.nuiton.util.ConverterUtil;
import org.nuiton.util.config.Config;
import org.nuiton.util.config.Property;

import javax.swing.JComponent;
import java.awt.Color;
import java.beans.PropertyChangeEvent;
import java.util.EnumMap;
import java.util.EnumSet;

/**
 * DialogUI handler
 *
 * @author chemit
 */
public abstract class DialogConfigUIHandler<E extends Enum<E>, M extends DialogConfigUIModel<E, ?>, U extends DialogConfigUI<E, ?>> extends DialogUIHandler<M, U> {

    protected DialogConfigUIHandler(U ui, M model) {
        super(ui, model);
    }

    public void propertyChange(PropertyChangeEvent evt) {
        if (log.isDebugEnabled()) {
            log.debug(evt.getPropertyName() + " old:" + evt.getOldValue() + ", new:" + evt.getNewValue());
        }
        String action = evt.getPropertyName();

        if (DialogConfigUIModel.CONFIG_PROPERTY_CHANGED.equals(action)) {
            // update ui with model values,
            populateUI();
            // revalidate form
            doCheckAll();
            return;
        }

        if (DialogConfigUIModel.MODIFIED_PROPERTY_CHANGED.equals(action)) {
            Boolean newValue = (Boolean) evt.getNewValue();
            boolean modified = newValue != null && newValue;
            getUi().getReset().setEnabled(modified);
            getUi().getOk().setEnabled(modified && getModel().isConfigValid());
            return;
        }

        if (DialogConfigUIModel.UNVALID_PROPERTY_CHANGED.equals(action)) {
            updateUI();
            return;
        }

        throw new IllegalStateException("unimplemented property changed : " + evt + " for " + this);
    }

    public void doCheck(E key) {
        Object uiValue = getUi().getElementValue(key);
        DialogConfigUIModel<E, ?> model = getModel();
        Object currentValue = model.getCurrent().getProperty(key);
        if (currentValue == null) {
            currentValue = "";
        } else {
            currentValue = String.valueOf(currentValue);
        }

        model.validateProperty(key, uiValue);
        model.changeModifiedState(key, uiValue, currentValue);
    }

    public void doCheckAll() {
        DialogConfigUIModel<E, ?> model = getModel();
        EnumSet<E> unvalids = EnumSet.noneOf(model.klass);
        for (E e : model.getCheckedKeysSet()) {
            Object uiValue = getUi().getElementValue(e);
            if (!model.isValid(e, uiValue)) {
                unvalids.add(e);
            }
        }
        model.setUnvalids(unvalids);
        unvalids.clear();
    }

    protected boolean  prepareSave() {
        DialogConfigUI<E, ?> ui = getUi();
        DialogConfigUIModel<E, ?> model = getModel();

        Config<E> current = model.getCurrent();

        if (!model.isConfigValid()) {
            log.warn("do not save a unvalid config : " + model.getUnvalids());
            return false;
        }

        if (!model.isModified()) {
            log.warn("nothing to save");
            return false;
        }
        EnumSet<E> toTreate = model.getCheckedKeysSet();
        // transfert checkable values from ui to model
        for (E key : model.getModifieds()) {
            if (!toTreate.contains(key)) {
                continue;
            }
            Object value = ui.getElementValue(key);
            Class<?> type = ((Property) key).getType();
            Object newValue = ConverterUtil.convert(type, value);
            current.setProperty(key, newValue);
        }

        return true;
    }

    protected void populateUI() {
        U ui = getUi();
        EnumMap<E, Object> map = getModel().getCurrent().getProperties();
        for (E e : getModel().getCheckedKeysSet()) {
            Object value = map.get(e);
            populateUI(ui, e, value);
        }
    }

    protected void populateUI(U ui, E key, Object value) {
        ui.setElementValue(key, value);
    }

    protected void updateUI() {
        EnumSet<E> unvalids = getModel().getUnvalids();
        for (E key : unvalids) {
            setLabelColor(key, false);
        }
        for (E key : EnumSet.complementOf(unvalids)) {
            setLabelColor(key, true);
        }
    }

    protected void setLabelColor(E key, boolean valid) {
        JComponent component = getUi().getElementLabel(key);
        if (component != null && component.isVisible()) {
            component.setForeground(valid ? Color.black : Color.red);
        }
    }

}