/*
 * #%L
 * I18n :: Api
 * 
 * $Id: I18nStore.java 1789 2010-11-08 10:28:45Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.0.1/nuiton-i18n/src/main/java/org/nuiton/i18n/I18nStore.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.bundle.I18nBundle;
import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.i18n.bundle.I18nBundleUtil;
import org.nuiton.i18n.init.I18nInitializer;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

/**
 * Represents the store of languages of the i18n system.
 * <p/>
 * Replace the {@code org.nuiton.i18n.I18nLoader} class from project
 * http://maven-site.nuiton.org/nuiton-utils
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.1
 */
public class I18nStore {

    /** Logger */
    private static final Log log = LogFactory.getLog(I18nStore.class);

    /** le language actuellement utilise */
    protected I18nLanguage language;

    /** le cache de languages deja charges */
    protected List<I18nLanguage> languages;

    /** le cache de bundles deja charges */
    protected I18nBundle[] bundles;

    /** la locale par defaut a utiliser */
    protected Locale defaultLocale;

    /** le resolver de bundles */
    protected I18nInitializer resolver;

    public I18nStore(Locale defaultLocale, I18nInitializer resolver) {
        this.defaultLocale = defaultLocale;
        this.resolver = resolver;
    }

    /** @return current language loaded or null, if no language was load */
    public I18nLanguage getLanguage() {
        return language;
    }

    /** @return le cache de language avec instanciation paresseuse */
    public List<I18nLanguage> getLanguages() {
        if (languages == null) {
            languages = new ArrayList<I18nLanguage>();
        }
        return languages;
    }

    public Locale getDefaultLocale() {
        return defaultLocale;
    }

    public boolean isEmpty() {
        boolean isEmpty = I18nBundleUtil.isEmpty(getBundles());
        return isEmpty;
    }

    /** @return array of all locales loaded */
    public Locale[] getLocales() {
        Locale[] result = I18nBundleUtil.getLocales(getBundles());
        return result;
    }

    public I18nBundle[] getBundles() {
        checkInit();
        return bundles;
    }

    public I18nBundle[] getBundles(Locale l) {
        I18nBundle[] result = I18nBundleUtil.getBundles(l, getBundles());
        return result;
    }

    public I18nBundleEntry[] getBundleEntries() {
        I18nBundleEntry[] result = I18nBundleUtil.getBundleEntries(getBundles());
        return result;
    }

    public I18nBundleEntry[] getBundleEntries(Locale l) {
        I18nBundleEntry[] result =
                I18nBundleUtil.getBundleEntries(l, getDefaultLocale(),
                                                getBundles());
        return result;
    }

    protected void init() {

        if (isInit()) {
            // already init
            return;
        }

        if (resolver == null) {
            throw new NullPointerException(
                    "resolver can not be null in " +
                    "org.nuiton.i18n.I18nStore.init method");
        }

        try {
            bundles = resolver.resolvBundles();
        } catch (Exception e) {
            throw new RuntimeException(
                    "Could not init store for reason " + e.getMessage(), e);
        }

        if (log.isInfoEnabled()) {
            log.info(bundles.length + " bundle(s) found, [" +
                     getBundleEntries().length + " file(s)].");
        }
    }

    /**
     * Set a new language in store, given a locale.
     *
     * @param locale la locale du language requis
     */
    protected void setLanguage(Locale locale) {
        init();
        if (log.isDebugEnabled()) {
            log.debug("locale: " + locale);
        }
        I18nLanguage result = getLanguage(locale);
        if (result == null) {
            result = addLanguage(locale);
        } else {
            if (log.isDebugEnabled()) {
                log.debug("using cached language : " + result);
            }
        }
        language = result;
        //TC-20090702 the selected langue is the default locale, usefull for
        // objects dealing with the default locale (swing widgets,...)
        Locale.setDefault(locale);
    }

    /**
     * Close store and release cache ofg language.
     * <p/>
     * Current language will be also clean.
     */
    protected void close() {
        if (languages != null) {
            if (log.isInfoEnabled()) {
                log.info("will close " + languages.size() + " language(s).");
            }
            for (I18nLanguage l : languages) {
                l.close();
            }
            languages.clear();
            languages = null;
        }
        if (bundles != null) {
            bundles = null;
        }
        language = null;
    }

    /**
     * @param locale la locale du language recherche
     * @return le language trouve dans le cache, ou null.
     */
    protected I18nLanguage getLanguage(Locale locale) {

        if (!(languages == null || languages.isEmpty())) {
            for (I18nLanguage l : languages) {
                if (locale.equals(l.getLocale())) {
                    return l;
                }
            }
        }
        return null;
    }

    protected I18nLanguage addLanguage(Locale locale) {
        I18nLanguage result;
        result = new I18nLanguage(locale);
        I18nBundleEntry[] entries = getBundleEntries(locale);
        result.load(entries);

        if (log.isInfoEnabled()) {
            log.info(result + ", nbEntries: " + entries.length +
                     ", nbSentences: " + result.size() + ".");
        }
        getLanguages().add(result);
        return result;
    }

    protected boolean isInit() {
        return bundles != null;
    }

    protected void checkInit() {
        if (!isInit()) {
            throw new IllegalStateException(
                    "should call init method on " + I18nStore.class);
        }
    }
}
