package org.nuiton.eugene.plugin;

/*-
 * #%L
 * EUGene :: Maven plugin
 * %%
 * Copyright (C) 2006 - 2016 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.io.Files;
import org.apache.commons.lang3.SystemUtils;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.nuiton.eugene.plugin.modelextension.AttributeBean;
import org.nuiton.eugene.plugin.modelextension.ClassBean;
import org.nuiton.eugene.plugin.modelextension.ModelBean;
import org.nuiton.eugene.plugin.modelextension.ModelBeanBuilder;
import org.nuiton.eugene.plugin.modelextension.PackageBean;
import org.nuiton.plugin.AbstractPlugin;

import java.io.File;
import java.io.FilenameFilter;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.Set;

/**
 * To a transform some object model extension files in compact format to a single flat properties format.
 *
 * <h3>Example of compact format</h3>
 * <pre>
 *
 * [model]
 * modeTagValue value
 * modelStereotype
 *
 * [package]
 * fr.ird.observe.entities
 * packageTagValue value
 * packageStereotype
 * [class]
 * fr.ird.observe.entities.CommentableEntity
 * classTagValue value
 * classStereotype
 * attribute.attributeTagValue value
 * attribute.attributeStereotype
 * </pre>
 *
 * <h3>Example of flat properties format</h3>
 * <pre>
 * model.tagValue.modeTagValue=value
 * model.stereotype.modelStereotype
 * package.fr.ird.observe.entities.tagValue.packageTagValue=value
 * package.fr.ird.observe.entities.stereotype=packageStereotype
 * fr.ird.observe.entities.CommentableEntity.class.tagValue.classTagValue=value
 * fr.ird.observe.entities.CommentableEntity.class.stereotype=classStereotype
 * fr.ird.observe.entities.CommentableEntity.attribute.attribute.tagValue.attributeTagValue=value
 * fr.ird.observe.entities.CommentableEntity.attribute.attribute.stereotype=attributeStereotype*
 * </pre>
 * Created on 09/09/16.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0
 */
@Mojo(name = "transform-compact-to-flat-properties")
public class TransformCompactToFlatPropertiesMojo extends AbstractPlugin {

    /**
     * Name of model (the generated file name is {@code modelName.properties}).
     */
    @Parameter(property = "eugene.modelName", required = true)
    protected String modelName;

    /**
     * Where to find and generate files.
     */
    @Parameter(property = "eugene.directory", defaultValue = "${project.basedir}/src/main/xmi", required = true)
    protected File directory;

    /**
     * Maven project.
     */
    @Parameter(defaultValue = "${project}", readonly = true)
    protected MavenProject project;

    /**
     * Display transformation result, but do not generate file.
     */
    @Parameter(property = "eugene.dryRun")
    protected boolean dryRun;

    /**
     * Verbose mode.
     */
    @Parameter(property = "eugene.verbose", defaultValue = "${maven.verbose}")
    protected boolean verbose;

    /**
     * Encoding to be used for generation of files.
     *
     * <b>Note:</b> If nothing is filled here, we will use the system property {@code file.encoding}.
     */
    @Parameter(property = "eugene.encoding", defaultValue = "${project.build.sourceEncoding}")
    protected String encoding;

    protected File[] inputFiles;
    protected File outputFile;
    protected String eol;

    @Override
    protected void init() throws Exception {

        eol = SystemUtils.LINE_SEPARATOR;

        inputFiles = directory.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".objectmodel-ext");
            }
        });

        getLog().info("Found " + inputFiles.length + " model extension file(s).");

        outputFile = new File(directory, modelName + ".properties");

    }

    @Override
    protected void doAction() throws Exception {

        ModelBeanBuilder modelBeanBuilder = new ModelBeanBuilder(false, modelName);

        for (File inputFile : inputFiles) {

            modelBeanBuilder.addFile(inputFile);
        }

        ModelBean modelBean = modelBeanBuilder.build();

        getLog().info(modelBeanBuilder.getStereotypeHits() + " stereotype(s) detected.");
        getLog().info(modelBeanBuilder.getTagValueHits() + " tag value(s) detected.");

        StringBuilder result = new StringBuilder();

        appendMap(modelBean.getTagValues(), "model", result);
        appendSet(modelBean.getStereotypes(), "model", result);

        if (modelBean.withPackages()) {

            for (PackageBean packageBean : modelBean.getPackages()) {

                appendMap(packageBean.getTagValues(), "package." + packageBean.getName(), result);
                appendSet(packageBean.getStereotypes(), "package." + packageBean.getName(), result);

            }

        }

        if (modelBean.withClasses()) {

            for (ClassBean classBean : modelBean.getClasses()) {

                appendMap(classBean.getTagValues(), classBean.getName() + ".class", result);
                appendSet(classBean.getStereotypes(), classBean.getName() + ".class", result);

                String prefix = classBean.getName() + ".attribute.";
                for (AttributeBean attributeBean : classBean.getAttributes()) {

                    appendMap(attributeBean.getTagValues(), prefix + attributeBean.getName(), result);
                    appendSet(attributeBean.getStereotypes(), prefix + attributeBean.getName(), result);
                }

            }

        }

        if (dryRun) {

            getLog().info("\n\nDryRun mode\ncontent:\n\n" + result.toString() + "\n\n");

        } else {

            getLog().info("Generate to: " + outputFile);
            Files.write(result.toString(), outputFile, Charset.forName(encoding));

        }

    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    protected void appendMap(Map<String, String> map, String prefix, StringBuilder result) {
        if (!map.isEmpty()) {
            for (Map.Entry<String, String> entry : map.entrySet()) {
                result.append(prefix).append(".tagValue.").append(entry.getKey()).append("=").append(entry.getValue()).append(eol);
            }
        }
    }

    protected void appendSet(Set<String> set, String prefix, StringBuilder result) {
        if (!set.isEmpty()) {
            result.append(prefix).append(".stereotype=").append(Joiner.on(',').join(set)).append(eol);
        }
    }

}
