package com.formos.tapestry.testify.internal;

import static java.lang.String.format;

import org.apache.tapestry5.ioc.ObjectCreator;
import org.apache.tapestry5.ioc.ServiceLifecycle;
import org.apache.tapestry5.ioc.ServiceResources;
import org.apache.tapestry5.ioc.services.Builtin;
import org.apache.tapestry5.ioc.services.PlasticProxyFactory;

/**
 * Allows a service to exist "per test" (on the assumption that a test runs in only one thread).
 * This involves an inner proxy, which caches an object derived from a
 * {@link org.apache.tapestry5.ioc.ObjectCreator} as a key in the {@link PerTestDataStore}.
 * Method invocations are delegated to the per-test service instance.
 * <p/>
 * This scheme ensures that, although the service builder method will be invoked many times
 * over the life of the application, the service decoration process occurs only once.
 * The final calling chain is: Service Proxy --&gt; Interceptor(s) (from Decorators) --&gt; Advise Proxy (from Advisiors) --&gt; PerTest Proxy --&gt; (per test) instance.
 */
public class PerTestServiceLifecycle implements ServiceLifecycle {

    private final PerTestDataStore perTestManager;
    private final PlasticProxyFactory proxyFactory;


    public PerTestServiceLifecycle(@Builtin PerTestDataStore perTestManager, @Builtin PlasticProxyFactory proxyFactory) {
        this.perTestManager = perTestManager;
        this.proxyFactory = proxyFactory;
    }


    /**
     * Returns false; this lifecycle represents a service that will be created many times (by each thread).
     */
    public boolean isSingleton() {
        return false;
    }

    
    @SuppressWarnings("unchecked")
    public Object createService(ServiceResources resources, ObjectCreator creator) {
        ObjectCreator perTestCreator = new PerTestServiceCreator(perTestManager, creator);
        Class<?> serviceInterface = resources.getServiceInterface();
        

        String toString = format("<PerTest Proxy for %s(%s)>", resources.getServiceId(), serviceInterface.getName());
        return proxyFactory.createProxy(serviceInterface, perTestCreator, toString);
    }
}
