/*
 * Decompiled with CFR 0.152.
 */
package com.azure.storage.common;

import com.azure.core.credential.AzureNamedKeyCredential;
import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipeline;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.util.Context;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import com.azure.storage.common.implementation.StorageImplUtils;
import com.azure.storage.common.policy.StorageSharedKeyCredentialPolicy;
import java.net.URL;
import java.text.Collator;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;

public final class StorageSharedKeyCredential {
    private static final ClientLogger LOGGER = new ClientLogger(StorageSharedKeyCredential.class);
    private static final Context LOG_STRING_TO_SIGN_CONTEXT = new Context((Object)"Azure-Storage-Log-String-To-Sign", (Object)true);
    private static final String ACCOUNT_NAME = "accountname";
    private static final String ACCOUNT_KEY = "accountkey";
    private final AzureNamedKeyCredential azureNamedKeyCredential;

    public StorageSharedKeyCredential(String accountName, String accountKey) {
        Objects.requireNonNull(accountName, "'accountName' cannot be null.");
        Objects.requireNonNull(accountKey, "'accountKey' cannot be null.");
        this.azureNamedKeyCredential = new AzureNamedKeyCredential(accountName, accountKey);
    }

    private StorageSharedKeyCredential(AzureNamedKeyCredential azureNamedKeyCredential) {
        Objects.requireNonNull(azureNamedKeyCredential, "'azureNamedKeyCredential' cannot be null.");
        this.azureNamedKeyCredential = azureNamedKeyCredential;
    }

    public static StorageSharedKeyCredential fromConnectionString(String connectionString) {
        HashMap<String, String> connectionStringPieces = new HashMap<String, String>();
        for (String connectionStringPiece : connectionString.split(";")) {
            String[] kvp = connectionStringPiece.split("=", 2);
            connectionStringPieces.put(kvp[0].toLowerCase(Locale.ROOT), kvp[1]);
        }
        String accountName = (String)connectionStringPieces.get(ACCOUNT_NAME);
        String accountKey = (String)connectionStringPieces.get(ACCOUNT_KEY);
        if (CoreUtils.isNullOrEmpty((CharSequence)accountName) || CoreUtils.isNullOrEmpty((CharSequence)accountKey)) {
            throw new IllegalArgumentException("Connection string must contain 'AccountName' and 'AccountKey'.");
        }
        return new StorageSharedKeyCredential(accountName, accountKey);
    }

    public static StorageSharedKeyCredential fromAzureNamedKeyCredential(AzureNamedKeyCredential azureNamedKeyCredential) {
        return new StorageSharedKeyCredential(azureNamedKeyCredential);
    }

    public String getAccountName() {
        return this.azureNamedKeyCredential.getAzureNamedKey().getName();
    }

    public String generateAuthorizationHeader(URL requestURL, String httpMethod, Map<String, String> headers) {
        return this.generateAuthorizationHeader(requestURL, httpMethod, headers, false);
    }

    public String generateAuthorizationHeader(URL requestURL, String httpMethod, Map<String, String> headers, boolean logStringToSign) {
        return this.generateAuthorizationHeader(requestURL, httpMethod, new HttpHeaders(headers), logStringToSign);
    }

    public String generateAuthorizationHeader(URL requestURL, String httpMethod, HttpHeaders headers, boolean logStringToSign) {
        String signature = StorageImplUtils.computeHMac256(this.azureNamedKeyCredential.getAzureNamedKey().getKey(), this.buildStringToSign(requestURL, httpMethod, headers, logStringToSign));
        return "SharedKey " + this.azureNamedKeyCredential.getAzureNamedKey().getName() + ":" + signature;
    }

    public String computeHmac256(String stringToSign) {
        return StorageImplUtils.computeHMac256(this.azureNamedKeyCredential.getAzureNamedKey().getKey(), stringToSign);
    }

    private String buildStringToSign(URL requestURL, String httpMethod, HttpHeaders headers, boolean logStringToSign) {
        String contentLength = headers.getValue("Content-Length");
        contentLength = "0".equals(contentLength) ? "" : contentLength;
        String dateHeader = headers.getValue("x-ms-date") != null ? "" : this.getStandardHeaderValue(headers, "Date");
        String stringToSign = String.join((CharSequence)"\n", httpMethod, this.getStandardHeaderValue(headers, "Content-Encoding"), this.getStandardHeaderValue(headers, "Content-Language"), contentLength, this.getStandardHeaderValue(headers, "Content-MD5"), this.getStandardHeaderValue(headers, "Content-Type"), dateHeader, this.getStandardHeaderValue(headers, "If-Modified-Since"), this.getStandardHeaderValue(headers, "If-Match"), this.getStandardHeaderValue(headers, "If-None-Match"), this.getStandardHeaderValue(headers, "If-Unmodified-Since"), this.getStandardHeaderValue(headers, "Range"), this.getAdditionalXmsHeaders(headers), this.getCanonicalizedResource(requestURL));
        if (logStringToSign) {
            StorageImplUtils.logStringToSign(LOGGER, stringToSign, LOG_STRING_TO_SIGN_CONTEXT);
        }
        return stringToSign;
    }

    private String getStandardHeaderValue(HttpHeaders headers, String headerName) {
        HttpHeader header = headers.get(headerName);
        return header == null ? "" : header.getValue();
    }

    private String getAdditionalXmsHeaders(HttpHeaders headers) {
        TreeMap<Object, String> sortedXmsHeaders = new TreeMap<Object, String>(Collator.getInstance(Locale.ROOT));
        int stringBuilderSize = 0;
        for (HttpHeader header : headers) {
            String headerName = header.getName();
            if (!"x-ms-".regionMatches(true, 0, headerName, 0, 5)) continue;
            String headerValue = header.getValue();
            stringBuilderSize += headerName.length() + headerValue.length();
            sortedXmsHeaders.put(headerName.toLowerCase(Locale.ROOT), headerValue);
        }
        if (sortedXmsHeaders.isEmpty()) {
            return "";
        }
        StringBuilder canonicalizedHeaders = new StringBuilder(stringBuilderSize + 2 * sortedXmsHeaders.size() - 1);
        sortedXmsHeaders.forEach((name, value) -> {
            if (canonicalizedHeaders.length() > 0) {
                canonicalizedHeaders.append('\n');
            }
            canonicalizedHeaders.append((String)name).append(':').append((String)value);
        });
        return canonicalizedHeaders.toString();
    }

    private String getCanonicalizedResource(URL requestURL) {
        StringBuilder canonicalizedResource = new StringBuilder("/");
        canonicalizedResource.append(this.azureNamedKeyCredential.getAzureNamedKey().getName());
        if (requestURL.getPath().length() > 0) {
            canonicalizedResource.append(requestURL.getPath());
        } else {
            canonicalizedResource.append('/');
        }
        if (requestURL.getQuery() == null) {
            return canonicalizedResource.toString();
        }
        Map<String, String[]> queryParams = StorageImplUtils.parseQueryStringSplitValues(requestURL.getQuery());
        for (Map.Entry<String, String[]> queryParam : queryParams.entrySet()) {
            Object[] queryParamValues = queryParam.getValue();
            Arrays.sort(queryParamValues);
            String queryParamValuesStr = String.join((CharSequence)",", (CharSequence[])queryParamValues);
            canonicalizedResource.append('\n').append(queryParam.getKey()).append(':').append(queryParamValuesStr);
        }
        return canonicalizedResource.toString();
    }

    public static StorageSharedKeyCredential getSharedKeyCredentialFromPipeline(HttpPipeline httpPipeline) {
        for (int i = 0; i < httpPipeline.getPolicyCount(); ++i) {
            HttpPipelinePolicy httpPipelinePolicy = httpPipeline.getPolicy(i);
            if (!(httpPipelinePolicy instanceof StorageSharedKeyCredentialPolicy)) continue;
            StorageSharedKeyCredentialPolicy storageSharedKeyCredentialPolicy = (StorageSharedKeyCredentialPolicy)httpPipelinePolicy;
            return storageSharedKeyCredentialPolicy.sharedKeyCredential();
        }
        return null;
    }
}

