package org.nuiton.util.updater;

/*
 * #%L
 * Nuiton Utils :: Nuiton Updater
 * $Id: ApplicationUpdater.java 2528 2013-03-12 23:03:34Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.6.11/nuiton-updater/src/main/java/org/nuiton/util/updater/ApplicationUpdater.java $
 * %%
 * Copyright (C) 2013 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs2.AllFileSelector;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.VFS;
import org.apache.commons.vfs2.provider.http.HttpFileSystemConfigBuilder;
import org.nuiton.util.VersionUtil;
import org.nuiton.util.config.ApplicationConfig;
import org.nuiton.util.config.ArgumentsParserException;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Permet de telecharger des mises a jour d'application.
 * <p/>
 * Le principe est qu'un fichier properties pointe par une URL indique les
 * information necessaire pour la recuperation de l'application.
 * <p/>
 * Si une nouvelle version de l'application existe, elle est alors telechargee
 * et decompressee dans un repertoire specifique (elle ne remplace pas l'application
 * courante).
 * <p/>
 * Il est alors a la charge d'un script de mettre en place cette nouvelle application
 * a la place de l'ancienne.
 * <p/>
 * Il est possible d'interagir avec ApplicationUpdater via l'implantation d'un
 * {@link ApplicationUpdaterCallback} passer en parametre de la methode {@link #update}
 * <p/>
 * <h3>Configuration possible</h3>
 * Vous pouvez passer un ApplicationConfig dans le constructeur ou utiliser
 * la recherche du fichier de configuration par defaut (ApplicationUpdater.properties)
 * <p/>
 * Cette configuration permet de récupérer les informations suivantes:
 * <li>http_proxy: le proxy a utiliser pour l'acces au reseau (ex: squid.chezmoi.fr:8080)
 * <li>os.name: le nom du systeme d'exploitation sur lequel l'application fonctionne (ex: Linux)
 * <li>os.arch: l'architecture du systeme d'exploitation sur lequel l'application fonctionne (ex: amd64)
 * <p/>
 * <h3>format du fichier de properties</h3>
 * <p/>
 * <li>[osName.][osArch.]appName.version=version de l'application</li>
 * <li>[osName.][osArch.]appName.auth=true ou false selon que l'acces a l'url
 * demande une authentification a fournir par le callback
 * (voir {@link ApplicationUpdaterCallback#updateToDo})</li>
 * <li>[osName.][osArch.]appName.url=url du fichier compresse de la nouvelle version
 * (format <a href="http://commons.apache.org/vfs/filesystems.html">commons-vfs2</a>)</li>
 * <p/>
 * appName est a remplacer par le nom de l'application. Il est possible
 * d'avoir plusieurs application dans le meme fichier ou plusieurs version
 * en fonction de l'os et de l'architecture.
 * <p/>
 * osName et osArch sont toujours en minuscule
 * <p/>
 * <h3>format des fichiers compresses</h3>
 * <p/>
 * Le fichier compresse doit avoir un repertoire racine qui contient l'ensemble de l'application
 * c-a-d que les fichiers ne doivent pas etre directement a la racine lorsqu'on
 * decompresse le fichier.
 * <p/>
 * exemple de contenu de fichier compresse convenable
 * <pre>
 * MonApp-0.3/Readme.txt
 * MonApp-0.3/License.txt
 * </pre>
 * <p/>
 * Ceci est du au fait qu'on renomme le repertoire racine avec le nom de l'application,
 * donc si le repertoire racine n'existe pas ou qu'il y a plusieurs repertoires
 * a la racine le resultat de l'operation n'est pas celui souhaite
 * <p/>
 * <h3>os.name and os.arch</h3>
 * <table>
 * <th><td>os.name</td><td>os.arch</td></th>
 * <tr><td>linux</td><td>amd64</td></tr>
 * <tr><td>linux</td><td>i386</td></tr>
 * <tr><td>mac</td><td>ppc</td></tr>
 * <tr><td>windows</td><td>x86</td></tr>
 * <tr><td>solaris</td><td>sparc</td></tr>
 * </table>
 * <p/>
 * os.name est tronque apres le 1er mot donc "windows 2000" et "windows 2003"
 * deviennet tous les deux "windows". Si vous souhaitez gérer plus finement vos
 * url de telechargement vous pouvez modifier les donnees via
 * {@link ApplicationUpdaterCallback#updateToDo(java.util.Map) } en modifiant
 * l'url avant de retourner la map
 *
 * @author bpoussin <poussing@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @since 2.7
 */
public class ApplicationUpdater {

    /** Logger. */
    private static final Log log = LogFactory.getLog(ApplicationUpdater.class);

    final static private String SEPARATOR_KEY = ".";

    public static final String HTTP_PROXY = "http_proxy";

    public static final String URL_KEY = "url";

    public static final String AUTHENTICATION_KEY = "auth";

    public static final String VERSION_KEY = "version";

    public static final String VERSION_FILE = "version.appup";

    protected ApplicationConfig config;

    protected DownloadMonitor downloadMonitor;

    /** Utilise le fichier de configuration par defaut: ApplicationUpdater.properties */
    public ApplicationUpdater() {
        this(null);
    }

    /**
     * @param config La configuration a utiliser pour rechercher le proxy (http_proxy)
     *               et os.name, os.arch
     */
    public ApplicationUpdater(ApplicationConfig config) {
        if (config == null) {
            try {
                config = new ApplicationConfig(
                        ApplicationUpdater.class.getSimpleName() + ".properties");
                config.parse();
                config = config.getSubConfig(
                        ApplicationUpdater.class.getSimpleName() + SEPARATOR_KEY);
            } catch (ArgumentsParserException eee) {
                throw new RuntimeException(eee);
            }
        }
        this.config = config;
    }

    public void setDownloadMonitor(DownloadMonitor downloadMonitor) {
        this.downloadMonitor = downloadMonitor;
    }

    /**
     * @param url        url where properties file is downloadable. This properties
     *                   must contains information on application release
     * @param currentDir directory where application is currently
     * @param destDir    default directory to put new application version, can be null if you used callback
     * @param async      if true, check is done in background mode
     * @param callback   callback used to interact with updater, can be null
     */
    public void update(String url,
                       File currentDir,
                       File destDir,
                       boolean async,
                       ApplicationUpdaterCallback callback) {
        Updater up = new Updater(config, url, currentDir, destDir, downloadMonitor, callback);
        if (async) {
            Thread thread = new Thread(up, ApplicationUpdater.class.getSimpleName());
            thread.start();
        } else {
            up.run();
        }
    }

    public static void createVersionFile(File dir, String version) throws IOException {
        File versionFile = new File(dir, VERSION_FILE);
        FileUtils.writeStringToFile(versionFile, version);
    }

    /**
     * La classe ou le travail est reellement fait, peut-etre appeler dans
     * un thread si necessaire.
     */
    public static class Updater implements Runnable {

        protected ApplicationConfig config;

        protected String vfsPropertiesUrl;

        protected File currentDir;

        protected File destDir;

        protected ApplicationUpdaterCallback callback;

        protected DownloadMonitor downloadMonitor;

        public Updater(ApplicationConfig config,
                       String vfsPropertiesUrl,
                       File currentDir,
                       File destDir,
                       DownloadMonitor downloadMonitor,
                       ApplicationUpdaterCallback callback) {
            this.config = config;
            this.vfsPropertiesUrl = vfsPropertiesUrl;
            this.currentDir = currentDir;
            this.destDir = destDir;
            this.downloadMonitor = downloadMonitor;
            this.callback = callback;
        }

        /**
         * <li>Recupere le fichier properties contenant les informations de mise a jour
         * <li>liste les applications et leur version actuelle
         * <li>pour chaque application a mettre a jour recupere le zip et le decompresse
         * <p/>
         * Si callback existe envoi les messages necessaire
         */
        @Override
        public void run() {
            try {
                FileSystemOptions vfsConfig = getVFSConfig(config);
                ApplicationConfig releaseConfig = getUpdaterConfig(vfsConfig, vfsPropertiesUrl);

                List<String> appNames = getApplicationName(releaseConfig);
                Map<String, String> appVersions = getCurrentVersion(appNames, currentDir);

                log.debug("application current version: " + appVersions);

                // recherche des applications a mettre a jour
                Map<String, ApplicationInfo> appToUpdate = new HashMap<String, ApplicationInfo>();
                for (String app : appNames) {
                    String currentVersion = appVersions.get(app);
                    String newVersion = releaseConfig.getOption(app + SEPARATOR_KEY + VERSION_KEY);
                    boolean greater = VersionUtil.greaterThan(newVersion, currentVersion);
                    log.debug(String.format("for %s Current(%s) < newVersion(%s) ? %s",
                                            app, currentVersion, newVersion, greater));
                    if (greater) {
                        String urlString = releaseConfig.getOption(
                                app + SEPARATOR_KEY + URL_KEY);
                        boolean needAuthentication = releaseConfig.getOptionAsBoolean(
                                app + SEPARATOR_KEY + AUTHENTICATION_KEY);

                        appToUpdate.put(app, new ApplicationInfo(
                                app, currentVersion, newVersion, urlString, destDir, needAuthentication));
                    }
                }

                // offre la possibilite a l'appelant de modifier les valeurs par defaut
                if (callback != null) {
                    appToUpdate = callback.updateToDo(appToUpdate);
                }

                // mise a jour
                Map<String, Exception> appUpdateError = new HashMap<String, Exception>();
                for (Map.Entry<String, ApplicationInfo> appInfo : appToUpdate.entrySet()) {
                    String app = appInfo.getKey();
                    ApplicationInfo info = appInfo.getValue();
                    try {
                        doUpdate(vfsConfig, appInfo.getValue());
                    } catch (Exception eee) {
                        appUpdateError.put(app, eee);
                        try {
                            // clear data if error occur during uncompress operation
                            File dest = new File(info.destDir, info.name);
                            if (dest.exists()) {
                                log.debug(String.format("Cleaning destination directory due to error '%s'", dest));
                                FileUtils.deleteDirectory(dest);
                            }
                        } catch (Exception doNothing) {
                            log.debug("Can't clean directory", doNothing);
                        }

                        log.warn(String.format(
                                "Can't update application '%s' with url '%s'",
                                app, info.url));
                        log.debug("Application update aborted because: ", eee);
                    }
                }

                // envoi le resultat a l'appelant s'il le souhaite
                if (callback != null) {
                    callback.updateDone(appToUpdate, appUpdateError);
                }
            } catch (Exception eee) {
                log.warn("Can't update");
                log.info("Application update aborted because: ", eee);
                if (callback != null) {
                    callback.aborted(vfsPropertiesUrl, eee);
                }
            }
        }

        /**
         * Decompresse le zip qui est pointer par l'url dans le repertoire
         * specifie, et ajoute le fichier contenant la version de l'application.
         * Le repertoire root du zip est renomme par le nom de l'application.
         * Par exemple si un fichier se nomme "monApp-1.2/Readme.txt" il se
         * nommera au final "monApp/Readme.txt"
         *
         * @param vfsConfig le proxy a utiliser pour la connexion a l'url
         * @param info      information sur l'application a mettre a jour
         * @throws Exception
         */
        protected void doUpdate(FileSystemOptions vfsConfig,
                                ApplicationInfo info) throws Exception {
            if (info.destDir != null) {
                File dest = new File(info.destDir, info.name);
                String url = toVfsURL(info.url);
                if (info.needAuthentication) {
                    url = StringUtils.replaceOnce(url, "://",
                                                  String.format("://%s:%s@", info.login, new String(info.password)));
                }
                if (callback != null) {
                    callback.startUpdate(info);
                }

                // le type de l'archive contenant la mise à jour
                String archiveType = url.substring(0, url.indexOf(':'));

                // recuperation de l'archive en locale (dans /tmp)
                File archive = downloadUpdate(vfsConfig, info, url.substring(archiveType.length() + 1));

                // extraction depuis l'archive téléchargée de l'unique répertoire vers la destination
                explodeUpdate(vfsConfig,
                              info,
                              archiveType,
                              archive,
                              dest);

                // ajout du fichier de version
                createVersionFile(dest, info.newVersion);
                log.info(String.format(
                        "Application '%s' is uptodate with version '%s' in '%s'",
                        info.name, info.newVersion, info.destDir));
            } else {
                log.info(String.format("Update for '%s' aborted because destination dir is set to null", info.name));
            }
        }

        /**
         * Télécharge une archive dans un fichier temporaraire.
         * <p/>
         * Si l'archive a plus d'un repertoire root, une exception est levee
         *
         * @param info
         * @param srcPath source path de la forme vfs2 ex:"zip:http://www.nuiton.org/attachments/download/830/nuiton-utils-2.6.5-deps.zip"
         * @throws FileSystemException
         */
        protected File downloadUpdate(FileSystemOptions vfsConfig,
                                      ApplicationInfo info,
                                      String srcPath) throws IOException {
            File result = new File(FileUtils.getTempDirectory(), srcPath);

            FileSystemManager fsManager = VFS.getManager();
            FileObject source = fsManager.resolveFile(srcPath, vfsConfig);

            if (!source.exists()) {
                throw new UpdateNotFoundException(info);
            }
            FileObject target = fsManager.toFileObject(result);
            InputStream input = source.getContent().getInputStream();
            try {
                OutputStream output = target.getContent().getOutputStream();
                try {
                    long inputSize = source.getContent().getSize();
                    if (downloadMonitor != null) {
                        downloadMonitor.setSize(inputSize);
                    }
                    long count = 0;
                    int n;
                    byte[] buffer = new byte[1024];
                    while (-1 != (n = input.read(buffer))) {
                        output.write(buffer, 0, n);
                        count += n;
                        if (downloadMonitor != null) {
                            downloadMonitor.setCurrent(count);
                        }
                    }
                    output.close();
                } finally {

                    IOUtils.closeQuietly(output);
                }
                input.close();
            } finally {
                IOUtils.closeQuietly(input);
            }
            return result;
        }

        /**
         * Recopie le contenu du répertoire de l'archive dans le
         * répertoire {@code target}.
         * <p/>
         * Si le répertoire cible existe déjà, il sera alors vidé.
         * <p/>
         * Si l'archive ne contient pas exactement un répertoire alors une exception est levée
         *
         * @param archiveType le type de l'archive
         * @param source      l'archive à décompresser
         * @param target      le répertoire cible
         * @throws FileSystemException
         * @throws UpdateInvalidArchiveLayoutException
         *                             si l'archive n'a pas le bon format
         */
        protected void explodeUpdate(FileSystemOptions vfsConfig,
                                     ApplicationInfo info,
                                     String archiveType,
                                     File source,
                                     File target) throws FileSystemException, UpdateInvalidArchiveLayoutException {

            FileSystemManager fsManager = VFS.getManager();
            FileObject sourceObject = fsManager.resolveFile(archiveType + ":" + source.getAbsolutePath(), vfsConfig);

            FileObject[] children = sourceObject.getChildren();
            if (children.length != 1) {
                throw new UpdateInvalidArchiveLayoutException(info, source);
            }
            // clean target
            FileObject targetObject = fsManager.toFileObject(target);
            targetObject.delete(new AllFileSelector());

            //copy to it the archive only directory
            FileObject child = children[0];
            targetObject.copyFrom(child, new AllFileSelector());
        }

        /**
         * Recupere le contenu du repertoire de l'archive pour le mettre dans targetPath
         * si targetPath existait deja, il est supprime au prealable.
         * <p/>
         * Si l'archive a plus d'un repertoire root, une exception est levee
         *
         * @param srcPath    source path de la forme vfs2 ex:"zip:http://www.nuiton.org/attachments/download/830/nuiton-utils-2.6.5-deps.zip"
         * @param targetPath le path destination
         * @throws FileSystemException
         * @deprecated since 2.6.11, no more used
         */
        @Deprecated
        protected void deepCopy(FileSystemOptions vfsConfig,
                                String srcPath,
                                String targetPath) throws FileSystemException {
            FileSystemManager fsManager = VFS.getManager();
            FileObject archive = fsManager.resolveFile(srcPath, vfsConfig);

            FileObject[] children = archive.getChildren();
            if (children.length == 1) {
                FileObject child = children[0];

                FileObject target = fsManager.resolveFile(toVfsURL(targetPath), vfsConfig);
                target.delete(new AllFileSelector());
                target.copyFrom(child, new AllFileSelector());
            } else {
                throw new RuntimeException("must have only one root directory");
            }
        }

        /**
         * Converti le path en URL vfs2. Path doit etre une URL, mais pour les fichiers
         * au lieu d'etre absolue ils peuvent etre relatif, un traitement special
         * est donc fait pour ce cas. Cela est necessaire pour facilement faire
         * des tests unitaires independant de la machine ou il sont fait
         *
         * @param path
         * @return
         */
        protected String toVfsURL(String path) {
            String result = path;
            Pattern p = Pattern.compile("(.*?file:)([^/][^!]*)(.*)");
            Matcher m = p.matcher(path);
            if (m.matches()) {
                String filepath = m.group(2);
                File f = new File(filepath);
                result = path.replaceAll(
                        "(.*?file:)([^/][^!]*)(.*)",
                        "$1" + f.getAbsolutePath() + "$3");
            }
            return result;
        }

        /**
         * Return config prepared for os and arch
         *
         * @return
         * @throws Exception
         */
        protected ApplicationConfig getUpdaterConfig(FileSystemOptions vfsConfig, String vfsPropertiesUrl) throws Exception {
            String osName = StringUtils.lowerCase(config.getOsName());
            String osArch = StringUtils.lowerCase(config.getOsArch());
            // take only first part for osName (windows 2000 or windows 2003 -> windows)
            osName = StringUtils.substringBefore(osName, " ");

            if (log.isDebugEnabled()) {
                log.debug(String.format("Try to load properties from '%s'", vfsPropertiesUrl));
            }

            Properties prop = new Properties();

            FileSystemManager fsManager = VFS.getManager();
            FileObject properties = fsManager.resolveFile(toVfsURL(vfsPropertiesUrl), vfsConfig);
            try {
                InputStream in = new BufferedInputStream(properties.getContent().getInputStream());
                prop.load(in);
            } finally {
                try {
                    properties.close();
                } catch (Exception doNothing) {
                    log.debug("Can't close vfs file", doNothing);
                }
            }

            if (log.isDebugEnabled()) {
                log.debug(String.format(
                        "Properties loaded from '%s'\n%s",
                        vfsPropertiesUrl, prop));
            }

            // load config with new properties as default
            ApplicationConfig result = new ApplicationConfig(prop);
            // don't parse. We want only prop in applicationConfig
            result = result.getSubConfig(
                    ApplicationUpdater.class.getSimpleName() + SEPARATOR_KEY);

            result = result.getSubConfig(osName + SEPARATOR_KEY);
            result = result.getSubConfig(osArch + SEPARATOR_KEY);
            return result;
        }

        /**
         * Recupere le proxy http a utiliser pour les connexions reseaux
         *
         * @param config
         * @return
         */
        protected FileSystemOptions getVFSConfig(ApplicationConfig config) {
            FileSystemOptions result = new FileSystemOptions();
            String proxyHost = config.getOption(HTTP_PROXY);
            try {
                proxyHost = StringUtils.substringAfter(proxyHost, "://");
                if (StringUtils.isNotBlank(proxyHost)) {
                    String hostname = StringUtils.substringBefore(proxyHost, ":");
                    String port = StringUtils.substringAfter(proxyHost, ":");
                    if (StringUtils.isNumeric(port)) {

                        int portNumber = Integer.parseInt(port);

                        HttpFileSystemConfigBuilder.getInstance().setProxyHost(result, hostname);
                        HttpFileSystemConfigBuilder.getInstance().setProxyPort(result, portNumber);
                    } else {
                        log.warn(String.format("Invalide proxy port number '%s', not used proxy", port));
                    }
                }
            } catch (Exception eee) {
                log.warn(String.format("Can't use proxy '%s'", proxyHost), eee);
            }
            return result;
        }

        /**
         * Recherche pour chaque application la version courante
         *
         * @param apps la liste des applications a rechercher
         * @return
         */
        protected Map<String, String> getCurrentVersion(List<String> apps, File dir) {
            Map<String, String> result = new HashMap<String, String>();
            for (String app : apps) {
                File f = new File(dir, app + File.separator + VERSION_FILE);
                String version = "0";
                try {
                    version = FileUtils.readFileToString(f);
                } catch (IOException ex) {
                    log.warn(String.format(
                            "Can't find file version '%s' for application '%s', this file should be '%s'",
                            VERSION_FILE, app, f));
                }
                version = StringUtils.trim(version);
                result.put(app, version);
            }
            return result;
        }

        /**
         * Retourne la liste des noms d'application se trouvant dans la
         * configuration
         *
         * @param config
         * @return
         */
        protected List<String> getApplicationName(ApplicationConfig config) {
            Pattern p = Pattern.compile("([^.]+)\\.version");
            List<String> result = new LinkedList<String>();
            for (String v : config.getFlatOptions().stringPropertyNames()) {
                Matcher match = p.matcher(v);
                if (match.matches()) {
                    result.add(match.group(1));
                } else if (StringUtils.endsWith(v, ".version")) {
                    log.debug(String.format("value is not valid application version '%s'", v));
                }
            }
            return result;
        }

    }
}
