/*
 * *##% Plugin maven pour i18n
 * Copyright (C) 2007 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */
package org.nuiton.i18n.plugin;

import org.apache.maven.plugin.AbstractMojo;
import org.nuiton.i18n.plugin.parser.ParserEvent;
import org.nuiton.i18n.I18nUtil;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.SortedSet;
import org.nuiton.util.PluginHelper;

/**
 * Classe permettant d'obenir les parametres pendant les différentes phases
 * du plugin.
 *
 * @author julien
 */
public abstract class AbstractI18nPlugin extends AbstractMojo {

    /**
     * Le nombre de getters détectés pendant le cycle de vie du build.
     */
    private static int NB_GETTER_FILES = 0;
    /**
     * Répertoire de stockage des fichiers i18n pour la recuperation des fichiers
     * de traduction entre librairie
     */
    protected static final String DIRECTORY_INSTALL = "i18n" + File.separatorChar;
    /**
     * Nom du projet.
     *
     * @parameter expression="${i18n.artifactId}" default-value="${project.artifactId}"
     * @readonly
     */
    protected String artifactId;
    /**
     * Langues des bundles generes.
     * <p/>
     *
     * @parameter expression="${i18n.bundles}" default-value="fr_FR,en_GB"
     * @required
     */
    protected String bundles;
    /**
     * Repertoire sources des fichiers i18n.
     *
     * @parameter expression="${i18n.src}" default-value="${basedir}/src/main/resources/i18n"
     * @required
     */
    protected File src;
    /**
     * Repertoire des fichiers generes i18n.
     *
     * @parameter expression="${i18n.out}" default-value="${basedir}/target/generated-sources/i18n"
     * @required
     */
    protected File out;
    /**
     * encoding a utiliser pour charger et sauver les bundles
     *
     * @parameter expression="${i18n.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     */
    protected String encoding;
    /**
     * Met les fichiers generes dans le repertoire des sources i18n.
     * <p/>
     * Note: Par défaut active, pour pouvoir paquager avec les bundles mis a jour.
     *
     * @parameter expression="${i18n.genSrc}" default-value="true"
     */
    protected boolean genSrc;
    /**
     * Active la modification de cle.
     * <p/>
     * Note: par defaut, on ne l'active pas (build sur serveur non ui).
     *
     * @parameter expression="${i18n.keysModifier}" default-value="false"
     */
    protected boolean keysModifier;
    /**
     * verbose flag
     * <p/>
     * Note: si non renseigne, on utilise la propiété <code>maven.verbose</code>.
     *
     * @parameter expression="${i18n.verbose}" default-value="${maven.verbose}"
     */
    protected boolean verbose;
    /**
     * silent flag
     *
     * @parameter expression="${i18n.silent}" default-value="false"
     * @since 1.0.0-rc-5
     */
    protected boolean silent;
    /**
     * conserve les anciens fichiers de traduction avec un suffix ~
     * <p/>
     * Note: par defaut, on ne l'active pas.
     *
     * @parameter expression="${i18n.keepBackup}" default-value="false"
     */
    protected boolean keepBackup;
    /**
     * ne conserve que les clef scannees (et donc traite tous les fichiers)
     *
     * <p/>
     * Note : par default, on ne l'active car rescanne tous les fichiers.
     *s
     * @parameter expression="${i18n.strictMode}" default-value="false"
     */
    protected boolean strictMode;
    /** Liste des évènements */
    protected List<ParserEvent> events = new ArrayList<ParserEvent>();
    protected Locale[] locales;
    /**logger verbeux */
    protected I18nLogger verboseLog;

    /**
     * Ajoute un évènement
     *
     * @param parserEvent l'évènement d'ajout
     */
    protected void addParserEvent(ParserEvent parserEvent) {
        this.events.add(parserEvent);
    }

    /**
     * Supprime un évènement
     *
     * @param parserEvent l'évènement de suppression
     */
    protected void removeParserEvent(ParserEvent parserEvent) {
        this.events.remove(parserEvent);
    }

    public void init() {

        verboseLog = new I18nLogger(this);

        if (verbose) {
            getLog().info("config - verbose mode is on");
        }
        locales = I18nUtil.parseLocales(bundles);
    }

    public String getArtifactId() {
        return artifactId;
    }

    /**
     *
     * @return <code>true</code> si des getters ont etes enregistres pendant
     * le cycle de vie, <code>false</code> sinon.
     */
    protected boolean needGeneration() {
        boolean needGeneration = NB_GETTER_FILES > 0;
        return needGeneration;
    }

    /**
     * Prend en compte qu'un getter a été détecté.
     *
     * Cela veut dire qu'un goal de parser a détecté des clefs. Il
     * faudra donc activer les goal get et gen.
     */
    protected void addGetter() {
        NB_GETTER_FILES++;
    }

    protected I18nLogger getVerboseLog() {
        return verboseLog;
    }

    /**
     * @param root       le repertoire ou sont stockes les fichiers i18n
     * @param artifactId le nom de l'artifact
     * @param locale     le nom du bundle
     * @param create     <code>true</code> pour creer le fichier si non present
     * @return le fichier i18n
     * @throws java.io.IOException si probleme lors de la creation du fichier
     */
    public File getI18nFile(File root, String artifactId, Locale locale, boolean create) throws IOException {
        File file = new File(root.getAbsolutePath() + File.separatorChar + artifactId + "-" + locale.toString() + ".properties");
        if (create && !file.exists()) {
            if (!file.exists()) {
                file.getParentFile().mkdirs();
            }
            file.createNewFile();
        }
        return file;
    }

    /**
     * @param root   le repertoire ou sont stockes les fichiers getter
     * @param getter le nom du getter
     * @param create <code>true</code> pour creer le fichier si non present
     * @return le fichier i18n
     * @throws java.io.IOException si probleme lors de la creation du fichier
     */
    public File getGetterFile(File root, String getter, boolean create) throws IOException {
        File file = new File(root.getAbsolutePath() + File.separatorChar + getter);
        if (create && !file.exists()) {
            if (!file.exists()) {
                file.getParentFile().mkdirs();
            }
            file.createNewFile();
        }
        return file;
    }

    /**
     * @param root   le repertoire ou sont stockes les fichiers getter
     * @param getter le nom du getter
     * @return le fichier i18n
     */
    public File getGetterFileBackup(File root, String getter) {
        return new File(root.getAbsolutePath() + File.separatorChar + getter + '~');
    }

    /**
     * @param root       le reertoire ou sont stockes les fichiers i18n
     * @param artifactId le nom de l'artifact
     * @param bundle     le nom du bundle
     * @return le fichier i18n de backup
     */
    public File getI18nFileBackup(File root, String artifactId, Locale bundle) {
        return new File(root.getAbsolutePath() + File.separatorChar + artifactId + "-" + bundle.toString() + ".properties~");
    }

    protected void checkBundle(Locale locale, Properties propertiesOut, boolean showEmpty) {
        // on verifie qu'il n'y a pas de traduction vide
        SortedSet<String> emptyEntries = PluginHelper.getEmptyKeys(propertiesOut);
        if (!emptyEntries.isEmpty()) {
            StringBuilder buffer = new StringBuilder();
            int size = emptyEntries.size();
            buffer.append("bundle " + locale + " contains " + size + "/" + propertiesOut.size() + " empty entries!");
            if (showEmpty) {
                int index = 0;
                for (String key : emptyEntries) {
                    buffer.append("\n  - " + (index++) + "/" + size + " : " + key);
                }
            } else {
                buffer.append(" (use -Di18n.showEmpty to see these entries)");
            }
            getLog().warn(buffer.toString());
        } else {
            if (!silent && verbose) {
                getLog().info("bundle " + locale + " is valid (no empty entries).");
            }
        }
    }

    protected void backupFile(File f) throws IOException {
        PluginHelper.copy(f, new File(f.getAbsolutePath() + "~"));
    }

    protected void copyFile(File src, File dst) throws IOException {
        PluginHelper.copy(src, dst);
    }
}
