/* *##% 
 * ToPIA :: Service History
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/

/* *
 * HistoryServiceImpl.java
 *
 * Created: 14 oct. 06 00:54:35
 *
 * @author poussin
 * @version $Revision: 1459 $
 *
 * Last update: $Date: 2009-05-16 09:56:47 +0200 (Sat, 16 May 2009) $
 * by : $Author: tchemit $
 */

package org.nuiton.topia.history;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.event.TopiaTransactionEvent;
import org.nuiton.topia.event.TopiaTransactionVetoable;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.history.entities.HistoryImpl;
import org.nuiton.topia.security.util.TopiaSecurityUtil;
import org.hibernate.Criteria;
import org.hibernate.FlushMode;
import org.hibernate.ScrollableResults;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;


/**
 * Pour active l'historisation des actions utilisateur il faut mettre dans le
 * fichier de propriete la cle suivante:
 * <li> topia.history=org.nuiton.topia.history.TopiaHistoryServiceImpl
 * <p> 
 * puis par defaut toutes les actions sont historis�es, mais on peut choisir
 * dans desactiver certaines:
 * <li> topia.history.create=[true|false]
 * <li> topia.history.delete=[true|false]
 * <li> topia.history.update=[true|false]
 * <li> topia.history.load=[true|false]
 * <p>
 * On peut demander a ce que l'historique soit nettoyer automatiquement avec:
 * <li> topia.history.clean.frequency=1
 * <li> topia.history.clean.date=3
 * <li> topia.history.clean.number=1000
 * <li> topia.history.store.file=/path/to/file/history
 *
 * Si seul frequency est specifier alors l'history est nettoyer a cette
 * frequence et pour le nombre de jour indiqu�. Par exemple si on indique 1d
 * alors tous les jours l'history est nettoy� pour ne garder qu'un jour.
 * <p>
 * Si on souhaite garder plus d'un jour, mais nettoyer tout de meme tous les
 * jours, il faut utiliser l'option topia.history.clean.date qui donne le 
 * nombre de jour a conserver.
 * <p>
 * Si on ne veut pas conserver un nombre de jour, mais un nombre
 * d'enregistrement il faut utiliser a la place topia.history.clean.number
 * <p>
 * Si l'on souhaite sauver l'history avant de le nettoyer il faut utiliser
 * l'option topia.history.store.file. Le fichier indiqu� est utilis� pour
 * ajouter l'history. Si le fichier existe, les nouveaux history lui sont
 * ajouter, sinon le fichier est cr��. Sous Linux on peut utiliser logrotate
 * sur ce fichier.
 *
 * @author poussin
 */
public class TopiaHistoryServiceImpl implements TopiaHistoryService, Runnable, TopiaTransactionVetoable {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(TopiaHistoryServiceImpl.class);

    static final private int FIELD_DATE = 0;  
    static final private int FIELD_USER = 1;  
    static final private int FIELD_ACTION = 2;  
    static final private int FIELD_TYPE = 3;  
    static final private int FIELD_TARGET = 4;  
    
    static private final String HISTORY_CREATE_KEY = "topia.service.history.create";
    static private final String HISTORY_DELETE_KEY = "topia.service.history.delete";
    static private final String HISTORY_UPDATE_KEY = "topia.service.history.update";
    static private final String HISTORY_LOAD_KEY = "topia.service.history.load";
    
    static private final String HISTORY_CLEAN_FREQUENCY_KEY = "topia.service.history.clean.frequency";
    static private final String HISTORY_CLEAN_DATE_KEY = "topia.service.history.clean.date";
    static private final String HISTORY_CLEAN_NUMBER_KEY = "topia.service.history.clean.number";
    static private final String HISTORY_STORE_FILE_KEY = "topia.service.history.store.file";
    
    
    protected TopiaHistoryListener historyListener = null;
    protected TopiaContextImplementor context = null;
    
    /** en jour */
    protected float cleanFrequency = -1;
    /** nombre de jour a garder */
    protected int cleanDate = -1;
    /** nombre d'history a garder */
    protected int cleanNumber = -1;
    /** fichier on conserver les histories */
    protected File storeFile = null;
    
    /** Scheduler de nettoyage */
    protected ScheduledThreadPoolExecutor task = null;
    
    /* (non-Javadoc)
     * @see org.nuiton.topia.framework.TopiaService#getServiceName()
     */
    @Override
    public String getServiceName() {
        return TopiaHistoryService.SERVICE_NAME;
    }
    
    /* (non-Javadoc)
     * @see org.nuiton.topia.framework.TopiaService#getPersistenceClasses()
     */
    @Override
    public Class<?>[] getPersistenceClasses() {
        return new Class<?>[]{HistoryImpl.class};
    }
    
    /* (non-Javadoc)
     * @see org.nuiton.topia.framework.TopiaService#init(org.nuiton.topia.framework.TopiaContextImplementor)
     */
    @Override
    public boolean preInit(TopiaContextImplementor context) {
        return true;
    }
    	
    /* (non-Javadoc)
     * @see org.nuiton.topia.framework.TopiaService#init(org.nuiton.topia.framework.TopiaContextImplementor)
     */
    @Override
    public boolean postInit(TopiaContextImplementor context) {
        this.context = context;

        Properties config = context.getConfig();
        
        boolean historyCreate = "true".equalsIgnoreCase(config.getProperty(HISTORY_CREATE_KEY, "true"));
        boolean historyDelete = "true".equalsIgnoreCase(config.getProperty(HISTORY_DELETE_KEY, "true"));
        boolean historyUpdate = "true".equalsIgnoreCase(config.getProperty(HISTORY_UPDATE_KEY, "true"));
        boolean historyLoad = "true".equalsIgnoreCase(config.getProperty(HISTORY_LOAD_KEY, "true"));
        
        historyListener = new TopiaHistoryListener(context,
                historyCreate, historyDelete, historyUpdate, historyLoad);
        context.addTopiaEntityListener(historyListener);
        context.addTopiaTransactionVetoable(this);
        
        cleanFrequency = Float.parseFloat(config.getProperty(HISTORY_CLEAN_FREQUENCY_KEY, ""+cleanFrequency));
        cleanDate = Integer.parseInt(config.getProperty(HISTORY_CLEAN_DATE_KEY, ""+cleanDate));
        cleanNumber = Integer.parseInt(config.getProperty(HISTORY_CLEAN_NUMBER_KEY, ""+cleanNumber));
        
        String storeFilename = config.getProperty(HISTORY_STORE_FILE_KEY);
        if (storeFilename != null) {
            storeFile = new File(storeFilename);
        }
        
        if (cleanFrequency > 0) {
            task = new ScheduledThreadPoolExecutor(1);
            // on fait le nettoyage a minuit
            Calendar cal = Calendar.getInstance();
            cal.setLenient(true);
            cal.set(Calendar.HOUR_OF_DAY, 0);
            cal.set(Calendar.DAY_OF_YEAR, cal.get(Calendar.DAY_OF_YEAR) + 1);
            long initialDelay = cal.getTimeInMillis() - System.currentTimeMillis();
            // ensuite on fait ca a interval regulier
            long period = (long)cleanFrequency * 24 * 3600 * 1000;
            task.scheduleAtFixedRate(this, initialDelay, period, TimeUnit.MILLISECONDS);
        }
        
        return true;
    }

    /**
     * Permet de propager le l'historisation sur l'ensemble des contextes
     * @param event
     */
    @Override
    public void beginTransaction(TopiaTransactionEvent event) {
        TopiaContext tx = event.getTopiaContext();
        tx.addTopiaEntityListener(historyListener);
        tx.addTopiaTransactionVetoable(this);
    }
    
    /* (non-Javadoc)
     * @see org.nuiton.topia.security.entities.authorization.History#clear(java.util.Date)
     */
    @Override
    public void clear(Date toDate) throws Exception {
        TopiaContext tx = context.beginTransaction();
        tx.find("delete from " + HistoryImpl.class.getName() + " where actionDate <= :date", "date", toDate);
        tx.commitTransaction();
        tx.closeContext();
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.security.entities.authorization.History#keep(int)
     */
    @Override
    public void keep(int number) throws Exception {
        // TODO Auto-generated method stub

    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.security.entities.authorization.History#store(java.util.Date, java.io.Writer)
     */
    @Override
    public void store(Date toDate, Writer out) throws Exception {
        if (out != null) {
            out = new BufferedWriter(out);
            TopiaContextImplementor tx = (TopiaContextImplementor)context.beginTransaction();
            ScrollableResults histories = tx.getHibernate().createCriteria(HistoryImpl.class)
            .add( Restrictions.ge("actionDate", toDate))
            .addOrder( Order.asc("actionDate") )
            .setFlushMode(FlushMode.AUTO)
            .scroll();
            while (histories.next()) {
                Date date = histories.getDate(FIELD_DATE);
                String user = histories.getString(FIELD_USER);
                String action = TopiaSecurityUtil.actionsInt2String(histories.getInteger(FIELD_ACTION));
                String type = histories.getString(FIELD_TYPE);
                String target = histories.getString(FIELD_TARGET);
                
                String text = date + "," + user + "," + action + "," + type + "," + target + "\n";
                out.write(text);
            }
            out.flush();
            tx.closeContext();
        }
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.TopiaHistoryService#findLastAction(java.lang.String, int, int)
     */
    @Override
    public List<String> findLastAction(int limit, String user, String type, Integer ... actions) throws Exception {
        TopiaContextImplementor tx = (TopiaContextImplementor)context.beginTransaction();

        Criteria criteria = tx.getHibernate()
        .createCriteria(HistoryImpl.class)
        .setFlushMode(FlushMode.AUTO)
        .addOrder( Order.desc("actionDate") );

        criteria.add( Restrictions.in("action", Arrays.asList(actions)));
        
        if (user != null) {
            criteria.add( Restrictions.eq("userId", user));
        }
        if (type != null) {
            criteria.add( Restrictions.in("type", new String[]{type, type + "Impl"}));
        }
        
        List<String> result = new ArrayList<String>();

        ScrollableResults histories = criteria.scroll(); 
        for (int i=limit; i == 0 && histories.next(); i--) {
            result.add(histories.getString(FIELD_TARGET));
        }
        
        tx.closeContext();
        return result;
    }

    /**
     * Fait le menage de l'history
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        log.info("Start history schuduling");
        try {
            Writer out = null;
            if (storeFile != null) {
                storeFile.getParentFile().mkdirs();
                out = new FileWriter(storeFile, true);
            }
            Date date = null;
            if (cleanNumber >= 0) {
                // on recupere la date du cleanNumber + 1
                TopiaContextImplementor tx = (TopiaContextImplementor)context.beginTransaction();
                ScrollableResults histories = tx.getHibernate().createCriteria(HistoryImpl.class)
                .setProjection(Property.forName("actionDate"))
                .addOrder( Order.desc("actionDate") )
                .setMaxResults(cleanNumber + 1)
                .setFlushMode(FlushMode.AUTO)
                .scroll();
                histories.last();
                date = histories.getDate(0);
                tx.closeContext();
            } else {
                if (cleanDate < 0) {
                    cleanDate = (int)cleanFrequency;
                }
                Calendar cal = Calendar.getInstance();
                cal.setLenient(true);
                int day = cal.get(Calendar.DAY_OF_YEAR);
                cal.set(Calendar.DAY_OF_YEAR, day - cleanDate);
                date = cal.getTime();
            }
            store(date, out);
            clear(date);
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't clean history", eee);
            }
        }
        log.info("End history schuduling");
    }

}


