package jaxx.runtime.swing.navigation.handler;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.navigation.NavigationContextHelper;
import jaxx.runtime.swing.navigation.NavigationModel;
import jaxx.runtime.swing.navigation.NavigationNode;

import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import java.awt.*;
import java.io.Serializable;

public interface NavigationHandler<E extends NavigationNode<E>> extends Cloneable, Serializable, TreeSelectionModel, TreeSelectionListener {

    /**
     * Strategy of instanciation of ui.
     * <p/>
     * For a given {@code node}, the method {@link #getId(NavigationNode)}
     * returns the id of ui to use.
     */
    enum Strategy {

        /** instanciate a ui for a node */
        PER_NODE {

            @Override
            public String getId(NavigationNode<?> node) {
                return node.getFullPath();
            }
        },
        /**
         * instanciate only one a ui for a type,nodes will share the
         * instanciation
         */
        PER_UI_TYPE {

            @Override
            public String getId(NavigationNode<?> node) {
                return node.getUIClass().getName();
            }
        };

        public abstract String getId(NavigationNode<?> node);
    }

    JAXXContext getContext();

    /** @return le modèle de navigation associé */
    NavigationModel<E> getNavigationTreeModel();

    /**
     * @return le composent actuellement visible associé au noeud courant ou au
     *         noeud précédent lors d'un changement de noeud.
     */
    Component getCurrentUI();

    /**
     * @param node le noeud associé à l'ui à retrouver
     * @return l'ui associé au novueau noeud sélectionné
     */
    Component getUI(E node);

    /**
     * @param component le composent actuellement visible
     * @return <code>true</code> si le composent a bien été fermé,
     *         <code>false</code> sinon
     * @throws Exception if any
     */
    boolean closeUI(Component component) throws Exception;

    /**
     * Instancie une nouvelle ui associé à un noeud de l'arbre de navigation
     *
     * @param node le noeud associé à l'ui à créer
     * @return la nouvelle ui associée au noeud
     * @throws Exception if any
     */
    Component createUI(E node) throws Exception;

    /**
     * Ouvre l'ui associée au noeud sélectionné dans l'arbre de navigation.
     *
     * @param newUI l'ui associé au noeud sélectionné à ouvrir
     * @param node  le node de l'ui a ouvrir
     * @throws Exception if any
     */
    void openUI(Component newUI, E node) throws Exception;

    /**
     * Traitement des exceptions.
     *
     * @param e l'erreur recontrée (ou null si pas d"erreur)
     */
    void treateError(Exception e);


    @Override
    void valueChanged(TreeSelectionEvent event);

    NavigationContextHelper<E> getContextHelper();

    @Override
    void setSelectionPath(TreePath path);

    @Override
    void setSelectionPaths(TreePath[] paths);
}
