/*
 * *##% 
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.navigation.handler;

import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.navigation.NavigationContentUI;
import jaxx.runtime.swing.navigation.NavigationContextHelper;
import jaxx.runtime.swing.navigation.NavigationMultiContentUI;
import jaxx.runtime.swing.navigation.NavigationNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.event.TreeSelectionEvent;
import javax.swing.tree.DefaultTreeSelectionModel;
import javax.swing.tree.TreePath;
import java.awt.*;
import java.util.Arrays;

/**
 * The handler of a navigation tree.
 * <p/>
 * This is also the selection model to use, since we must check before moving
 * from a node we can not just listen selection model changed, we must control
 * it.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.7.2
 */
public abstract class AbstractNavigationHandler<E extends NavigationNode<E>> extends DefaultTreeSelectionModel implements NavigationHandler<E> {

    private static final long serialVersionUID = 1L;

    /** Logger */
    static private final Log log =
            LogFactory.getLog(AbstractNavigationHandler.class);

    /** UI which contains navigation tree */
    protected JAXXContext context;

    /** UI Instanciation strategy */
    protected Strategy strategy;

    /**
     * JAXXContext access helper.
     *
     * @since 1.7.2
     */
    protected NavigationContextHelper<E> contextHelper;

    protected AbstractNavigationHandler(String contextPrefix,
                                        JAXXObject context,
                                        Strategy strategy) {
        contextHelper = new NavigationContextHelper<E>(contextPrefix);
        this.context = context;
        this.strategy = strategy;
        addTreeSelectionListener(this);
    }

    @Override
    public void valueChanged(TreeSelectionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Value changed for path : " + event.getPath());
            log.debug("Old lead selection path : " +
                      event.getOldLeadSelectionPath());
        }
        if (event.getOldLeadSelectionPath() != null &&
            event.getOldLeadSelectionPath().equals(event.getPath())) {
            // do not treate this if no path changed
            return;
        }
        E node = (E) event.getPath().getLastPathComponent();
        selectNodeUI(node);
    }

    @Override
    public JAXXContext getContext() {
        return context;
    }

    @Override
    public NavigationContextHelper<E> getContextHelper() {
        return contextHelper;
    }

    @Override
    public void setSelectionPath(TreePath path) {
        if (path.equals(getSelectionPath())) {
            // stay on same node, can skip
            if (log.isDebugEnabled()) {
                log.debug("skip stay on path " + path);
            }
            return;
        }
        Component component = getCurrentUI();

        try {
            if (!closeUI(component)) {
                if (log.isDebugEnabled()) {
                    log.debug("changing node canceled!");
                }
                // can not changed current node
                return;
            }
            if (component instanceof NavigationContentUI<?>) {
                ((NavigationContentUI<E>) component).closeUI((E) path.getLastPathComponent());
            }
        } catch (Exception ex) {
            treateError(ex);
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("will select path " + path);
        }
        // ok can safely select the new path
        super.setSelectionPath(path);
    }

    @Override
    public void setSelectionPaths(TreePath[] paths) {
        //FIXME-TC20100315 this can not work...
        // can't test like this paths as Array...
        if (Arrays.equals(paths, getSelectionPaths())) {
            // stay on same node, can skip
            if (log.isDebugEnabled()) {
                log.debug("skip stay on path " + Arrays.toString(paths));
            }
            return;
        }
        Component component = getCurrentUI();

        try {
            if (!closeUI(component)) {
                if (log.isDebugEnabled()) {
                    log.debug("changing node canceled!");
                }
                // can not changed current node
                return;
            }
            if (component instanceof NavigationMultiContentUI<?>) {
                if (paths.length == 1) {
                    NavigationContextHelper<E> helper = getContextHelper();
                    ((NavigationMultiContentUI<E>) component).closeUI(helper.getSelectedNodes(getContext()));
                }
            }
        } catch (Exception ex) {
            treateError(ex);
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("will select " + paths.length + " paths");
        }
        // ok can safely select the new path
        super.setSelectionPaths(paths);
    }

    protected void selectNodeUI(E node) {

        try {

            String path = node.getFullPath();

            if (log.isTraceEnabled()) {
                log.trace(path);
            }

            Component newUI = getUI(node);

            // now, we are free to open the ui associated with the selected node in navigation

            // get the bean associated with the node
            Object data = getNavigationTreeModel().getBean(path);

            // save it in context (must be done before init ui)
            addSelectedBeanInContext(node, data);

            if (newUI == null) {

                // a new ui instance is required
                newUI = createUI(node);
            }

            JAXXContext ctxt = getContext();
            NavigationContextHelper<E> helper = getContextHelper();

            // save in context current node context path
            helper.setSelectedPath(ctxt, node.getFullPath());

            // save in context current node
            helper.setSelectedNode(ctxt, node);

            // save in context current ui
//            helper.setSelectedUI(ctxt, newUI);

            // really open the ui associated with the selected node
            // init ui before to be visible
            if (newUI instanceof NavigationContentUI<?>) {
                ((NavigationContentUI<E>) newUI).openUI(node);
            }

            // set ui in content
            openUI(newUI, node);

        } catch (Exception e) {
            // remove data from context

            // if any error, go back to previvous node
            treateError(e);
        }
    }

    /**
     * Prepare le context a utiliser pour initialiser une nouvelle ui.
     *
     * @param node le noeud associé à l'ui à créer
     * @return le context à utiliser pour instancier l'ui
     * @throws Exception if any
     */
    protected JAXXContext createUIContext(E node) throws Exception {

        if (node.getUIHandlerClass() == null) {
            if (log.isWarnEnabled()) {
                log.warn("no action associated with ui " + node.getUIClass());
            }
            // no action associated, just
            return getContext();
        }

        JAXXAction action = node.getUIHandlerClass().newInstance();

        // init context with
        JAXXInitialContext uiContext = action.init(getContext());
        return uiContext;
    }

    protected void addSelectedBeanInContext(E node, Object data) {

        if (log.isDebugEnabled()) {
            log.debug("find data for contextPath <" + node.getFullPath() + "> : " + (data == null ? null : data.getClass()));
        }
        JAXXContext ctxt = getContext();
        NavigationContextHelper<E> helper = getContextHelper();

        // remove previous selected bean
        //TODO-TC-20091004 should have an automatic clean context method while
        helper.setSelectedBean(ctxt, null);

        if (data != null) {

            helper.setSelectedBean(ctxt, data);
        }
    }
}
